"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Calibration Service*

:details: CalibrationService:
    Delivers full information of the calibration system and allows complete overwriting of it.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    CalibrationService_simulation.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-19T13:54:01.473144
:date: (last modification) 2021-03-19T13:54:01.473144

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import CalibrationService_pb2 as CalibrationService_pb2
# from .gRPC import CalibrationService_pb2_grpc as CalibrationService_pb2_grpc

# import default arguments
from .CalibrationService_default_arguments import default_dict

from sila2lib_implementations.Flowmeter.FlowmeterService.lib import decode_response_read, decode_response_write


class CalibrationServiceSimulation:
    """
    Implementation of the *Calibration Service* in *Simulation* mode
        This is a flowmeter service
    """

    def __init__(self, status):
        """Class initialiser"""
        self.status = status
        self.protocol = 'ascii'
        self.node = '03'
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]


    def GetCalibrationCertificate(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.GetCalibrationCertificate_Responses:
        """
        Executes the unobservable command "Get calibration certificate"
            Number of calibration certificate (last basic calibration)
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.LastCalibrationCertificate (Last calibration certificate): Number of calibration certificate (last basic calibration)
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '08'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '08'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4             # 04=Request parameter
            process_query = 113     # Process number in decimal
            parameter_query =8      # Process response in decimal
            process_response = 113  # Parameter query in decimal
            parameter_response = 8  # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302710801\r\n'

                else:
                    read = b'\x10\x02\x01\x03\x04\x02q\x08\x01\x10\x03'

                type_answer = default_dict['GetCalibrationCertificate_Responses']['LastCalibrationCertificate'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'LastCalibrationCertificate': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = CalibrationService_pb2.GetCalibrationCertificate_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.GetCalibrationCertificate_Responses(
                **default_dict['GetCalibrationCertificate_Responses']
            )
    
        return return_value
    
    
    def SetCalibrationCertificate(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.SetCalibrationCertificate_Responses:
        """
        Executes the unobservable command "Set calibration certificate"
            Set new calibration certificate
    
        :param request: gRPC request containing the parameters passed:
            request.SetCalibrationCertificate (Set new calibration certificate):
            Set new certificate of the last basic calibration
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set calibration service
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set calibration service status applies
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetCalibrationCertificate.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '08'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1          # 01=Write parameter
            process_query = 113  # Process number in decimal
            parameter_query = 8  # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000005\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'

                type_answer = default_dict['SetCalibrationCertificate_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = CalibrationService_pb2.SetCalibrationCertificate_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.SetCalibrationCertificate_Responses(
                **default_dict['SetCalibrationCertificate_Responses']
            )
    
        return return_value
    
    
    def GetCalibrationDate(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.GetCalibrationDate_Responses:
        """
        Executes the unobservable command "Get Calibration date"
            Date of last (basic) calibration
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.LastCalibrationDate (Last calibration date): Date of last (basic) calibration
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '09'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '09'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 9         # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 9      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302710901\r\n'

                else:
                    read = b'\x10\x02\x01\x03\x04\x02q\t\x01\x10\x03'

                type_answer = default_dict['GetCalibrationDate_Responses']['LastCalibrationDate'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'LastCalibrationDate': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = CalibrationService_pb2.GetCalibrationDate_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.GetCalibrationDate_Responses(
                **default_dict['GetCalibrationDate_Responses']
            )
    
        return return_value
    
    
    def SetCalibrationDate(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.SetCalibrationDate_Responses:
        """
        Executes the unobservable command "Set calibration date"
            Set new calibration date
    
        :param request: gRPC request containing the parameters passed:
            request.SetCalibrationDate (Set calibration date):
            New date of the last basic calibration
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set calibration date
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set calibration date status applies
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetCalibrationDate.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '09'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write  parameter
            process_query = 113     # Process number in decimal
            parameter_query = 9     # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000005\r\n'

                else:
                    read =  b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'

                response = decode_response_write(protocol=self.protocol, message_str=read)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = CalibrationService_pb2.SetCalibrationDate_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.SetCalibrationDate_Responses(
                **default_dict['SetCalibrationDate_Responses']
            )
    
        return return_value
    
    
    def GetServiceNumber(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.GetServiceNumber_Responses:
        """
        Executes the unobservable command "Get Service number"
            Service number for repair/re-building/re-calibration
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.LastServiceNumber (Last service number):
            Last service number for repair/re-building/re-calibration
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '0a'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '0a'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 10        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 10     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302710a01\r\n'

                else:
                    read =  b'\x10\x02\x01\x03\x04\x02q\n\x01\x10\x03'

                type_answer = default_dict['GetServiceNumber_Responses']['LastServiceNumber'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'LastServiceNumber': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = CalibrationService_pb2.GetServiceNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.GetServiceNumber_Responses(
                **default_dict['GetServiceNumber_Responses']
            )
    
        return return_value
    
    
    def SetServiceNumber(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.SetServiceNumber_Responses:
        """
        Executes the unobservable command "Set service number"
            Set new service number for repair/re-building/re-calibration
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set service number
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set service number status applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetServiceNumber.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '0a'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write  parameter
            process_query = 113     # Process number in decimal
            parameter_query = 10    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000005\r\n'

                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetServiceNumber_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = CalibrationService_pb2.SetServiceNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.SetServiceNumber_Responses(
                **default_dict['SetServiceNumber_Responses']
            )
    
        return return_value
    
    
    def GetServiceDate(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.GetServiceDate_Responses:
        """
        Executes the unobservable command "Get service date"
            Date of last maintenance service
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.LastServiceDate (Last service date): Date of last service
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '0b'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '0b'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 11        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 11     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302710410921\r\n'

                else:
                    read = b'\x10\x02\x01\x03\x04\x02q\x0b\x01\x10\x03'

                type_answer = default_dict['GetServiceDate_Responses']['LastServiceDate'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'LastServiceDate': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = CalibrationService_pb2.GetServiceDate_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.GetServiceDate_Responses(
                **default_dict['GetServiceDate_Responses']
            )
    
        return return_value
    
    
    def SetServiceDate(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.SetServiceDate_Responses:
        """
        Executes the unobservable command "Set service date"
            Set new date of last maintenance service
    
        :param request: gRPC request containing the parameters passed:
            request.SetServiceDate (Set service date):
            Set new date of last maintenance service
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set service date
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set service date status applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetServiceDate.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '0b'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write  parameter
            process_query = 113     # Process number in decimal
            parameter_query = 11    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000000\r\n'

                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetServiceDate_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = CalibrationService_pb2.SetServiceDate_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.SetServiceDate_Responses(
                **default_dict['SetServiceDate_Responses']
            )
    
        return return_value
    
    
    def GetSensorCalibrationTemperature(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.GetSensorCalibrationTemperature_Responses:
        """
        Executes the unobservable command "Get Sensor calibration temperature"
            Sensor temperature from the last calibration in degree Celsius
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorCalibrationTemperature (Current sensor calibration temperature in degree Celsius ): Sensor temperature from the last calibration in
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '74'        # Process number in hexadecimal
            parameter_query = '60'      # Parameter query in hexadecimal
            process_response = '74'     # Process response in hexadecimal
            parameter_response = '60'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 116         # Process number in decimal
            parameter_query = 96        # Process response in decimal
            process_response = 116      # Parameter query in decimal
            parameter_response = 96     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302746025\r\n'

                else:
                    read =  b'\x10\x02\x01\x03\x04\x02t`\x01\x10\x03'

                type_answer = default_dict['GetSensorCalibrationTemperature_Responses']['CurrentSensorCalibrationTemperature'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)


                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentSensorCalibrationTemperature': silaFW_pb2.Real(
                        value=float(response['message']))
                }
                return_value = CalibrationService_pb2.GetSensorCalibrationTemperature_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.GetSensorCalibrationTemperature_Responses(
                **default_dict['GetSensorCalibrationTemperature_Responses']
            )
    
        return return_value
    
    
    def SetSensorCalibrationTemperature(self, request, context: grpc.ServicerContext) \
            -> CalibrationService_pb2.SetSensorCalibrationTemperature_Responses:
        """
        Executes the unobservable command "Set sensor calibration temperature"
            Set sensor calibration temperature in degree Celsius
    
        :param request: gRPC request containing the parameters passed:
            request.SetSensorCalibrationTemperature (Set sensor calibration temperature): Set new sensor calibration temperature in degree Celsuis
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status):
            Command status: Request response of command set sensor calibration temperature
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set sensor calibration temperature status applies
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetSensorCalibrationTemperature.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '74'    # Process number in hexadecimal
            parameter_query = '60'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write  parameter
            process_query = 116     # Process number in decimal
            parameter_query = 96    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0480000005\r\n'

                else:
                    read =  b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'

                type_answer = default_dict['SetSensorCalibrationTemperature_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.Real(value=float(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = CalibrationService_pb2.SetSensorCalibrationTemperature_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = CalibrationService_pb2.SetSensorCalibrationTemperature_Responses(
                **default_dict['SetSensorCalibrationTemperature_Responses']
            )
    
        return return_value
