"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Service*

:details: DeviceService:
    Delivers full information of the device, actual parameters, relevant information and allows modification of device
    parameters
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    12.02.2020

:file:    DeviceService_real.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-21T13:49:44.218707
:date: (last modification) 2021-03-21T13:49:44.218707

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceService_pb2 as DeviceService_pb2
# from .gRPC import DeviceService_pb2_grpc as DeviceService_pb2_grpc

# import default arguments
from .DeviceService_default_arguments import default_dict

from sila2lib_implementations.Flowmeter.FlowmeterService.lib import decode_response_read, decode_response_write

# noinspection PyPep8Naming,PyUnusedLocal
class DeviceServiceReal:
    """
    Implementation of the *Device Service* in *Real* mode
        This is a flowmeter service
    """

    def __init__(self, ser, status):
        """Class initialiser"""
        self.ser = ser
        self.status = status
        self.protocol = 'ascii'
        self.node = '03'
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetIdentificationString(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetIdentificationString_Responses:
        """
        Executes the unobservable command "Get Identification String"
            Identification number of the device and software version
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentIdentificationString (Current identification string): Identification number of the device and software version
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '00'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '00'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query =0          # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 0      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetIdentificationString_Responses']['CurrentIdentificationString'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentIdentificationString': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetIdentificationString_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetIdentificationString_Responses(
                **default_dict['GetIdentificationString_Responses']
            )

        return return_value
    
    
    def GetPrimaryNodeAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetPrimaryNodeAddress_Responses:
        """
        Executes the unobservable command "Get primary node address "
            Primary node address: network parameter Flow-Bus
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPrimaryNodeAddress (Current primary node address): Primary node address: network parameter Flow-Bus
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '01'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '01'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 1         # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 1      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetPrimaryNodeAddress_Responses']['CurrentPrimaryNodeAddress'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                par_dict = {
                    'CurrentPrimaryNodeAddress': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetPrimaryNodeAddress_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetPrimaryNodeAddress_Responses(
                **default_dict['GetPrimaryNodeAddress_Responses']
            )
    
        return return_value
    
    
    def SetPrimaryNodeAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetPrimaryNodeAddress_Responses:
        """
        Executes the unobservable command "Set primary node address"
            Primary node address: network parameter Flow-Bus
    
        :param request: gRPC request containing the parameters passed:
            request.SetPrimaryNodeAddress (Set primary node address): Primary node address: network parameter Flow-Bus
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set primary node address
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set primary node address applies
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetPrimaryNodeAddress.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '00'    # Process number in hexadecimal
            parameter_query = '01'  # Parameter query in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write parameter
            process_query = 0       # Process number in decimal
            parameter_query = 1     # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':

                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))

                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')

                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetPrimaryNodeAddress_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = DeviceService_pb2.SetPrimaryNodeAddress_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetPrimaryNodeAddress_Responses(
                **default_dict['SetPrimaryNodeAddress_Responses']
            )
    
        return return_value
    
    
    def GetSecondaryNodeAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetSecondaryNodeAddress_Responses:
        """
        Executes the unobservable command "Get secondary node address "
            Secondary node address: network parameter Flow-Bus
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSecondaryNodeAddress (Current secondary node address): Secondary node address: network parameter Flow-Bus
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '02'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '02'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 2         # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 2      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetSecondaryNodeAddress_Responses']['CurrentSecondaryNodeAddress'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentSecondaryNodeAddress': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetSecondaryNodeAddress_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetSecondaryNodeAddress_Responses(
                **default_dict['GetSecondaryNodeAddress_Responses']
            )
    
        return return_value
    
    
    def GetNextNodeAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetNextNodeAddress_Responses:
        """
        Executes the unobservable command "Get next node address"
            Next node address: network parameter Flow-Bus
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentNextNodeAddress (Current next node address): Next node address: network parameter Flow-Bus
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '03'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '03'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 3         # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 3      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetNextNodeAddress_Responses']['CurrentNextNodeAddress'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentNextNodeAddress': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetNextNodeAddress_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetNextNodeAddress_Responses(
                **default_dict['GetNextNodeAddress_Responses']
            )
    
        return return_value
    
    
    def GetLastNodeAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetLastNodeAddress_Responses:
        """
        Executes the unobservable command "Get last node address "
            Last node address: network parameter Flow-Bus
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentLastNodeAddress (current last node address): Last node address: network parameter Flow-Bus
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '04'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '04'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 4         # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 4      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetLastNodeAddress_Responses']['CurrentLastNodeAddress'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentLastNodeAddress': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetLastNodeAddress_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetLastNodeAddress_Responses(
                **default_dict['GetLastNodeAddress_Responses']
            )
    
        return return_value
    
    
    def GetSensorType(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetSensorType_Responses:
        """
        Executes the unobservable command "Get sensor type"
            Sensor type information for actual reading and sensor/controller indication
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (current sensor type): Sensor type information for actual reading and sensor/controller indication
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '0e'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '0e'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 14        # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 14     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetSensorType_Responses']['CurrentSensorType'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentSensorType': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetSensorType_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetSensorType_Responses(
                **default_dict['GetSensorType_Responses']
            )
    
        return return_value
    
    
    def GetAlarmInformation(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetAlarmInformation_Responses:
        """
        Executes the unobservable command "Get alarm information"
            Information of several alarms/errors in the instrument
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentAlarmInformation (Current alarm information): Information of several alarms/errors in the instrument
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '14'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '14'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 20        # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 20     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetAlarmInformation_Responses']['CurrentAlarmInformation'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentAlarmInformation': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetAlarmInformation_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetAlarmInformation_Responses(
                **default_dict['GetAlarmInformation_Responses']
            )
    
        return return_value
    
    
    def GetDeviceType(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetDeviceType_Responses:
        """
        Executes the unobservable command "Get device type"
            Flow bus device type information
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDeviceType (Current device type): Current flow bus device type information
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '01'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '01'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4             # 04=Request parameter
            process_query = 113     # Process number in decimal
            parameter_query = 1     # Process response in decimal
            process_response = 113  # Parameter query in decimal
            parameter_response = 1  # Parameter response
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetDeviceType_Responses']['CurrentDeviceType'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentDeviceType': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetDeviceType_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetDeviceType_Responses(
                **default_dict['GetDeviceType_Responses']
            )
    
        return return_value
    
    
    def GetFirmwareVersion(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetFirmwareVersion_Responses:
        """
        Executes the unobservable command "Get firmware version"
            revision number of the firmware
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFirmwareVersion (Current firmware version): revision number of the firmware
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'                  # 04=Request parameter
            process_query = '71'            # Process number in hexadecimal
            parameter_query = '65'          # Parameter query in hexadecimal
            process_response = '71'         # Process response in hexadecimal
            parameter_response = '6506'     # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 5         # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 5      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetFirmwareVersion_Responses']['CurrentFirmwareVersion'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentFirmwareVersion': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetFirmwareVersion_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetFirmwareVersion_Responses(
                **default_dict['GetFirmwareVersion_Responses']
            )
    
        return return_value
    
    
    def GetPressureSensorType(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetPressureSensorType_Responses:
        """
        Executes the unobservable command "Get pressure sensor type"
            type of pressure sensor
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPressureSensorType (Current pressure sensor type): type of pressure sensor
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '09'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '09'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 115         # Process number in decimal
            parameter_query = 9         # Process response in decimal
            process_response = 115      # Parameter query in decimal
            parameter_response = 9      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetPressureSensorType_Responses']['CurrentPressureSensorType'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentPressureSensorType': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetPressureSensorType_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetPressureSensorType_Responses(
                **default_dict['GetPressureSensorType_Responses']
            )
    
        return return_value
    
    
    def GetSensorName(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetSensorName_Responses:
        """
        Executes the unobservable command "Get sensor name"
            label with information about stop sensor
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorName (Current sensor name): label with information about stop sensor
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '11'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '11'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 17        # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 17     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetSensorName_Responses']['CurrentSensorName'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentSensorName': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetSensorName_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetSensorName_Responses(
                **default_dict['GetSensorName_Responses']
            )
    
        return return_value
    
    
    def GetIdentificationNumber(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetIdentificationNumber_Responses:
        """
        Executes the unobservable command "Get identification number"
            Identification number type of instrument/device
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentIdentificationNumber (Current identification number): Identification number type of instrument/device
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '0c'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '0c'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 12        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 12     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetIdentificationNumber_Responses']['CurrentIdentificationNumber'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentIdentificationNumber': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetIdentificationNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetIdentificationNumber_Responses(
                **default_dict['GetIdentificationNumber_Responses']
            )
    
        return return_value
    
    
    def GetPowerMode(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetPowerMode_Responses:
        """
        Executes the unobservable command "Get power mode"
            power suppy indication in Vdc
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPowerMode (Current power mode): power supply indication in Vdc
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '73'        # Process number in hexadecimal
            parameter_query = '0c'      # Parameter query in hexadecimal
            process_response = '73'     # Process response in hexadecimal
            parameter_response = '0c'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 115         # Process number in decimal
            parameter_query = 12        # Process response in decimal
            process_response = 115      # Parameter query in decimal
            parameter_response = 12     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetPowerMode_Responses']['CurrentPowerMode'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentPowerMode': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetPowerMode_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetPowerMode_Responses(
                **default_dict['GetPowerMode_Responses']
            )
    
        return return_value
    
    
    def GetBusDiagnostic(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetBusDiagnostic_Responses:
        """
        Executes the unobservable command "Get bus diagnostic"
            Fieldbus baudrate (top interface)
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentBusDiagnostic (Current bus diagnostic): Fieldbus baudrate (top interface)
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '7d'        # Process number in hexadecimal
            parameter_query = '14'      # Parameter query in hexadecimal
            process_response = '7d'     # Process response in hexadecimal
            parameter_response = '14'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 125         # Process number in decimal
            parameter_query = 20        # Process response in decimal
            process_response = 125      # Parameter query in decimal
            parameter_response = 20     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetBusDiagnostic_Responses']['CurrentBusDiagnostic'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentBusDiagnostic': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetBusDiagnostic_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetBusDiagnostic_Responses(
                **default_dict['GetBusDiagnostic_Responses']
            )
    
        return return_value
    
    
    def GetFieldbus(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetFieldbus_Responses:
        """
        Executes the unobservable command "Get fieldbus string"
            Fieldbus string name of the top interface
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFieldbus (Current bus string): Fieldbus string name of the top interface
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '7d'        # Process number in hexadecimal
            parameter_query = '15'      # Parameter query in hexadecimal
            process_response = '7d'     # Process response in hexadecimal
            parameter_response = '15'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 125         # Process number in decimal
            parameter_query = 21        # Process response in decimal
            process_response = 125      # Parameter query in decimal
            parameter_response = 21     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetFieldbus_Responses']['CurrentFieldbus'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentFieldbus': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetFieldbus_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetFieldbus_Responses(
                **default_dict['GetFieldbus_Responses']
            )
    
        return return_value
    
    
    def GetInstrumentProperties(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetInstrumentProperties_Responses:
        """
        Executes the unobservable command "Get instrument properties"
            instrument properties
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentInstrumentProperties (Current instrument properties): Instrument properties
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '5f'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '5f'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 95        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 95     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetInstrumentProperties_Responses']['CurrentInstrumentProperties'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentInstrumentProperties': silaFW_pb2.String(
                        value=str(response['message']))
                }
                return_value = DeviceService_pb2.GetInstrumentProperties_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetInstrumentProperties_Responses(
                **default_dict['GetInstrumentProperties_Responses']
            )
    
        return return_value
    
    
    def GetCommunicationProtocol(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetCommunicationProtocol_Responses:
        """
        Executes the unobservable command "Get communication protocol"
            Current communication protocol between the program adn the device.
            2 possible options: "binary" and "ascii"
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCommunicationProtocol (Current communication protocol): Primary node address: network parameter Flow-Bus
        """
    
        # initialise the return value
        return_value = None
    
        # TODO:
        #   Add implementation of Real for command GetCommunicationProtocol here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetCommunicationProtocol_Responses(
                **default_dict['GetCommunicationProtocol_Responses']
            )
    
        return return_value
    
    
    def SetCommunicationProtocol(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetCommunicationProtocol_Responses:
        """
        Executes the unobservable command "Set communication protocol"
            Allows to choose the communication protocol between the program adn the device.
            2 possible options:_ "binary" and "ascii"
    
        :param request: gRPC request containing the parameters passed:
            request.SetCommunicationProtocol (Set communication protocol):
            Allows to choose the communication protocol between the program adn the device.
            2 possible options:_ "binary" and "ascii"
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.EmptyResponse (Empty Response): An empty response data type used if no response is required.
        """

        # initialise the return value
        return_value = None

        # TODO:
        #   Add implementation of Real for command SetCommunicationProtocol here and write the resulting response
        #   in return_value

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetCommunicationProtocol_Responses(
                **default_dict['SetCommunicationProtocol_Responses']
            )
    
        return return_value
    
    
    def GetSerialPort(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetSerialPort_Responses:
        """
        Executes the unobservable command "Get serial port"
            Current devices serial port
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.EmptyResponse (Empty Response): An empty response data type used if no response is required.
        """
    
        # initialise the return value
        return_value = None
    
        # TODO:
        #   Add implementation of Real for command GetSerialPort here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetSerialPort_Responses(
                **default_dict['GetSerialPort_Responses']
            )
    
        return return_value
    
    
    def SetSerialPort(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetSerialPort_Responses:
        """
        Executes the unobservable command "Set serial port"
            Sets devices new serial port
    
        :param request: gRPC request containing the parameters passed:
            request.SetSerialPort (Set serial port): Sets devices new serial port
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.EmptyResponse (Empty Response): An empty response data type used if no response is required.
        """
    
        # initialise the return value
        return_value = None
    
        # TODO:
        #   Add implementation of Real for command SetSerialPort here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetSerialPort_Responses(
                **default_dict['SetSerialPort_Responses']
            )
    
        return return_value
    

    
