"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Fluid Definition Service*

:details: FluidDefinitionService:
    Allows full set of the fluid properties as well as the current information of the working fluid.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    FluidDefinitionService_simulation.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-19T13:54:01.497768
:date: (last modification) 2021-03-19T13:54:01.497768

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect          # used for status determination
import grpc             # used for type hinting only
from struct import *    # used to code binary protocol


# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import FluidDefinitionService_pb2 as FluidDefinitionService_pb2
# from .gRPC import FluidDefinitionService_pb2_grpc as FluidDefinitionService_pb2_grpc

# import default arguments
from .FluidDefinitionService_default_arguments import default_dict

from sila2lib_implementations.Flowmeter.FlowmeterService.lib import decode_response_read, decode_response_write


class FluidDefinitionServiceSimulation:
    """
    Implementation of the *Fluid Definition Service* in *Simulation* mode
        This is a flowmeter service
    """

    def __init__(self, status):
        """Class initialiser"""
        self.status = status
        self.protocol = 'ascii'
        self.node = '03'
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetFluidSetProperties(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.GetFluidSetProperties_Responses:
        """
        Executes the unobservable command "Get Fluidset properties"
            The current fluidset properties for the calculations performed by the program
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidsetProperties (Current fluidset properties): The current fluidset properties for the calculations performed by the program
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '1F'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '1F'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 31        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 31     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':05030211F01\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02a!\x01\x10\x03'
                type_answer = default_dict['GetFluidSetProperties_Responses']['CurrentFluidsetProperties'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentFluidsetProperties': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = FluidDefinitionService_pb2.GetFluidSetProperties_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.GetFluidSetProperties_Responses(
                **default_dict['GetFluidSetProperties_Responses']
            )
    
        return return_value
    
    
    def SetFluidsetProperties(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.SetFluidsetProperties_Responses:
        """
        Executes the unobservable command "Set fluid properties"
            The fluidset properties to work with
    
        :param request: gRPC request containing the parameters passed:
            request.SetFluidProperties (Set Fluid set properties):
            The fluidset properties to work with
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set fluid properties
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the Set fluid properties applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetFluidProperties.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)
        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '21'    # Process number in hexadecimal
            parameter_query = '1F'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1              # 01=write parameter
            process_query = 33       # Process number in decimal
            parameter_query = 31     # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0403000001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetFluidsetProperties_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = FluidDefinitionService_pb2.SetFluidsetProperties_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.SetFluidsetProperties_Responses(
                **default_dict['SetFluidsetProperties_Responses']
            )
    
        return return_value
    
    
    def GetFluidNumber(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.GetFluidNumber_Responses:
        """
        Executes the unobservable command "Get fluid number"
            Current fluid ID-Nr.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFuidNumber (Current fluid number): Current fluid ID-Nr
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '10'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '10'   # Parameter response in decimaldecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 16        # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 16     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302001001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02a\x03\x01\x10\x03'
                type_answer = default_dict['GetFluidNumber_Responses']['CurrentFuidNumber'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentFuidNumber': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = FluidDefinitionService_pb2.GetFluidNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.GetFluidNumber_Responses(
                **default_dict['GetFluidNumber_Responses']
            )
    
        return return_value
    
    
    def SetFluidNumber(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.SetFluidNumber_Responses:
        """
        Executes the unobservable command "Set Fluid Number"
            Set a number of the fluid to work with
    
        :param request: gRPC request containing the parameters passed:
            request.SetFluidNumber (Set fluid number):
            The number of the fluid to work with
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set fluid number
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which set fluid number applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetFluidNumber.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'              # 01=write parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '10'      # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1                 # 01=write parameter
            process_query = 1           # Process number in decimal
            parameter_query = 16        # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0403000001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetFluidNumber_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = FluidDefinitionService_pb2.SetFluidNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.SetFluidNumber_Responses(
                **default_dict['SetFluidNumber_Responses']
            )
    
        return return_value
    
    
    def GetFluidName(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.GetFluidName_Responses:
        """
        Executes the unobservable command "Get fluid name"
            The name of the current fluid
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidName (Current fluid name): Current fluid name
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '11'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '11'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 17        # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 17     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302001101\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x04\x02\x00\x11\x01\x10\x03'
                type_answer = default_dict['GetFluidName_Responses']['CurrentFluidName'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentFluidName': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = FluidDefinitionService_pb2.GetFluidName_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.GetFluidName_Responses(
                **default_dict['GetFluidName_Responses']
            )
    
        return return_value
    
    
    def SetFluidName(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.SetFluidName_Responses:
        """
        Executes the unobservable command "Set Fluid Name"
            Set the name of the fluid to work with
    
        :param request: gRPC request containing the parameters passed:
            request.SetFluidName (Set fluid name):
            The name of the fluid to work with
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set fluid name
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which fluid name applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetFluidName.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '01'    # Process number in hexadecimal
            parameter_query = '11'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 1       # Process number in decimal
            parameter_query = 17    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0403000001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetFluidName_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = FluidDefinitionService_pb2.SetFluidName_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.SetFluidName_Responses(
                **default_dict['SetFluidName_Responses']
            )
    
        return return_value
    
    
    def GetHeatCapacity(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.GetHeatCapacity_Responses:
        """
        Executes the unobservable command "Get heat capacity"
            The heat capacity of the fluid in Joule per K
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentHeatCapacity (Current heat capacity): Current heat capacity
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '52'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '52'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 82        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 82     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302715299\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02qR\x01\x10\x03'

                type_answer = default_dict['GetHeatCapacity_Responses']['CurrentHeatCapacity'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentHeatCapacity': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = FluidDefinitionService_pb2.GetHeatCapacity_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.GetHeatCapacity_Responses(
                **default_dict['GetHeatCapacity_Responses']
            )
    
        return return_value
    
    
    def SetHeatCapacity(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.SetHeatCapacity_Responses:
        """
        Executes the unobservable command "Set heat capacity"
            Set the heat capacity of the fluid in Joule Per Kelvin
    
        :param request: gRPC request containing the parameters passed:
            request.SetHeatCapacity (Set heat capacity):
            Set the heat capacity of the fluid in Joule per Kelvin.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set heat capacity
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which set heat capacity applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetHeatCapacity.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '52'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1               # 01=write parameter
            process_query = 113       # Process number in decimal
            parameter_query = 82      # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0403000001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetHeatCapacity_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = FluidDefinitionService_pb2.SetHeatCapacity_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.SetHeatCapacity_Responses(
                **default_dict['SetHeatCapacity_Responses']
            )
    
        return return_value
    
    
    def GetViscosity(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.GetViscosity_Responses:
        """
        Executes the unobservable command "Get viscosity"
            Viscosity of the fluid in cP
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentViscosity (Current viscosity): Current viscosity in cP
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '55'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '55'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 85        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 85     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302715516\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02qU\x01\x10\x03'

                type_answer = default_dict['GetViscosity_Responses']['CurrentViscosity'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentViscosity': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = FluidDefinitionService_pb2.GetViscosity_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.GetViscosity_Responses(
                **default_dict['GetViscosity_Responses']
            )
    
        return return_value
    
    
    def SetViscosity(self, request, context: grpc.ServicerContext) \
            -> FluidDefinitionService_pb2.SetViscosity_Responses:
        """
        Executes the unobservable command "Set viscosity"
            Set the viscosity of the fluid in cP
    
        :param request: gRPC request containing the parameters passed:
            request.SetViscosity (Set viscosity):
            Set the viscosity of the fluid in cP
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set viscosity
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which set viscosity applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetViscosity.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '55'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1               # 01=write parameter
            process_query = 113       # Process number in decimal
            parameter_query = 85      # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':0403000001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x03\x00\x00\x01\x10\x03'
                type_answer = default_dict['SetViscosity_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = FluidDefinitionService_pb2.SetViscosity_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = FluidDefinitionService_pb2.SetViscosity_Responses(
                **default_dict['SetViscosity_Responses']
            )
    
        return return_value
