"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Measurement Provider*

:details: MeasurementProvider:
    Used to acquire general status of the current flow of the flowmeter.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    MeasurementProvider_simulation.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-19T13:54:01.506006
:date: (last modification) 2021-03-19T13:54:01.506006

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol


# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import MeasurementProvider_pb2 as MeasurementProvider_pb2
# from .gRPC import MeasurementProvider_pb2_grpc as MeasurementProvider_pb2_grpc

# import default arguments
from .MeasurementProvider_default_arguments import default_dict

from sila2lib_implementations.Flowmeter.FlowmeterService.lib import decode_response_read, decode_response_write


class MeasurementProviderSimulation:
    """
    Implementation of the *Measurement Provider* in *Simulation* mode
        This is a flowmeter service
    """

    def __init__(self, status):
        """Class initialiser"""
        self.status = status
        self.protocol = 'ascii'
        self.node = '03'
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetTemperature(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetTemperature_Responses:
        """
        Executes the unobservable command "GetTemperature"
            Absolute temperature in  degree Celsius
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentTemperature (Current temperature): Absolute temperature in degree Celsius
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '47'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '47'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 71        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 71     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302214701\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x01G!\x01\x10\x03'

                type_answer = default_dict['GetTemperature_Responses']['CurrentTemperature'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentTemperature': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetTemperature_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetTemperature_Responses(
                **default_dict['GetTemperature_Responses']
            )
    
        return return_value
    
    
    def GetPressure(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetPressure_Responses:
        """
        Executes the unobservable command "Get pressure"
            Absolute pressure in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPressure (Current pressure): Current absolute pressure in millibar
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '48'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '48'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 72        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 72     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302214801\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02!H\x01\x10\x03'

                type_answer = default_dict['GetPressure_Responses']['CurrentPressure'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentPressure': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetPressure_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetPressure_Responses(
                **default_dict['GetPressure_Responses']
            )
    
        return return_value
    
    
    def GetTime(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetTime_Responses:
        """
        Executes the unobservable command "Get time"
            
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.RealTime (Real Time): Real time in milliseconds
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '49'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '49'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 73        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 73     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302214901\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02!I\x01\x10\x03'

                type_answer = type(silaFW_pb2.Real)
                type_answer = default_dict['GetTime_Responses']['RealTime'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'RealTime': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetTime_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetTime_Responses(
                **default_dict['GetTime_Responses']
            )
    
        return return_value
    
    
    def GetCalibratedVolume(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetCalibratedVolume_Responses:
        """
        Executes the unobservable command "Get calibrated volume"
            Calibrated volume
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCalibratedVolume (Current calibrated volume): Current calibrated volume
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '50'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '50'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 80        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 80     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302215001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02!J\x01\x10\x03'

                type_answer = default_dict['GetCalibratedVolume_Responses']['CurrentCalibratedVolume'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentCalibratedVolume': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetCalibratedVolume_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetCalibratedVolume_Responses(
                **default_dict['GetCalibratedVolume_Responses']
            )
    
        return return_value
    
    
    def GetSensorNumber(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetSensorNumber_Responses:
        """
        Executes the unobservable command "Get sensor number"
            Pointer to sensor number in calibration tube FPP
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorNumber (Current sensor number): Pointer to sensor number in calibration tube FPP
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '10'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '10'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 16        # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 16     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302001001\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02\x00\x10\x01\x10\x03'

                type_answer = default_dict['GetSensorNumber_Responses']['CurrentSensorNumber'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentSensorNumber': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetSensorNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetSensorNumber_Responses(
                **default_dict['GetSensorNumber_Responses']
            )
    
        return return_value
    
    
    def GetNormalVolumeFlow(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetNormalVolumeFlow_Responses:
        """
        Executes the unobservable command "Get normal volume flow"
            Volume flow referenced to normal conditions (0 Celsius degree;  1013.25HPa=1atm) in liters/min
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentNormalVolumeFlow (Current normal volume flow):
            Volume flow referenced to normal conditions (0 Celsius degree;  1013.25HPa=1atm) in liters/min
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '45'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '45'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 69        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 69     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302214501\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x04\x02!E\x01\x10\x03'

                type_answer = default_dict['GetNormalVolumeFlow_Responses']['CurrentNormalVolumeFlow'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentNormalVolumeFlow': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetNormalVolumeFlow_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetNormalVolumeFlow_Responses(
                **default_dict['GetNormalVolumeFlow_Responses']
            )
    
        return return_value
    
    
    def GetVolumeFlow(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetVolumeFlow_Responses:
        """
        Executes the unobservable command "Get volume flow"
            Volume flow at actual conditions in liters per min
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVolumeFlow (Current volume flow):
            VVolume flow at actual conditions in liters permin
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '46'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '46'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 70        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 70     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302214601\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02!F\x01\x10\x03'

                type_answer = default_dict['GetVolumeFlow_Responses']['CurrentVolumeFlow'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentVolumeFlow': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetVolumeFlow_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetVolumeFlow_Responses(
                **default_dict['GetVolumeFlow_Responses']
            )
    
        return return_value
    
    
    def GetDeltaPressure(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetDeltaPressure_Responses:
        """
        Executes the unobservable command "Get delta pressure"
            Relative pressure between atmosphere and sensor in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDeltaPressure (Current delta pressure):
            Relative pressure between atmosphere and sensor in millibar
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '51'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '51'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 75        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 75     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302215101\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02!K\x01\x10\x03'

                type_answer = default_dict['GetDeltaPressure_Responses']['CurrentDeltaPressure'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentDeltaPressure': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetDeltaPressure_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetDeltaPressure_Responses(
                **default_dict['GetDeltaPressure_Responses']
            )
    
        return return_value
    
    
    def GetMassFlow(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetMassFlow_Responses:
        """
        Executes the unobservable command "Get mass flow"
            real mass flow in Kg pro min
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentMassFlow (Current mass flow):
            Real mass flow in Kg pro min
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '44'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '44'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 68        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 68     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302214401\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x04\x02!D\x01\x10\x03'

                type_answer = default_dict['GetMassFlow_Responses']['CurrentMassFlow'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentMassFlow': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetMassFlow_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetMassFlow_Responses(
                **default_dict['GetMassFlow_Responses']
            )
    
        return return_value
    
    
    def GetMass(self, request, context: grpc.ServicerContext) \
            -> MeasurementProvider_pb2.GetMass_Responses:
        """
        Executes the unobservable command "Get mass"
            Net mass in g
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentMass (Current mass): Net Mass in g
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '63'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '63'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 87        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 87     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    read = ':050302612101\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x05\x04!W!W\x10\x03'

                type_answer = default_dict['GetMass_Responses']['CurrentMass'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentMass': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementProvider_pb2.GetMass_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementProvider_pb2.GetMass_Responses(
                **default_dict['GetMass_Responses']
            )
    
        return return_value



