"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Measurement Settings Service*

:details: MeasurementSettingsService:
    Used to acquire and modify general status and parameters of the flowmeter.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    MeasurementSettingsService_real.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-03-19T13:54:01.527793
:date: (last modification) 2021-03-19T13:54:01.527793

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol


# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import MeasurementSettingsService_pb2 as MeasurementSettingsService_pb2
# from .gRPC import MeasurementSettingsService_pb2_grpc as MeasurementSettingsService_pb2_grpc

# import default arguments
from .MeasurementSettingsService_default_arguments import default_dict

from sila2lib_implementations.Flowmeter.FlowmeterService.lib import decode_response_read, decode_response_write


class MeasurementSettingsServiceReal:
    """
    Implementation of the *Measurement Settings Service* in *Real* mode
        This is a flowmeter service
    """

    def __init__(self, ser, status):
        """Class initialiser"""
        self.ser = ser
        self.status = status
        self.protocol = 'binary'
        self.node = '03'
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetPistonOperationMode(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetPistonOperationMode_Responses:
        """
        Executes the unobservable command "Get piston operation mode"
            Piston prover operation mode and its status information
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPistonOperationMode (Current piston operation mode): Current piston prover operation mode and its status information
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '73'        # Process number in hexadecimal
            parameter_query = '0a'      # Parameter query in hexadecimal
            process_response = '73'     # Process response in hexadecimal
            parameter_response = '0a'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 115         # Process number in decimal
            parameter_query = 10        # Process response in decimal
            process_response = 115      # Parameter query in decimal
            parameter_response = 10     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetPistonOperationMode_Responses']['CurrentPistonOperationMode'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentPistonOperationMode': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetPistonOperationMode_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetPistonOperationMode_Responses(
                **default_dict['GetPistonOperationMode_Responses']
            )
    
        return return_value
    
    
    def SetPistonOperationMode(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetPistonOperationMode_Responses:
        """
        Executes the unobservable command "Set piston operation mode"
            Set the new piston prover operation mode
    
        :param request: gRPC request containing the parameters passed:
            request.SetPistonOperationMode (Set piston operation mode):
            The the new piston prover operation mode
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set piston operation mode
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the Set piston operation mode applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetPistonOperationMode.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'              # 01=write parameter
            process_query = '73'        # Process number in hexadecimal
            parameter_query = '0a'      # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1                 # 01=write parameter
            process_query = 115         # Process number in decimal
            parameter_query = 10        # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetPistonOperationMode_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetPistonOperationMode_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetPistonOperationMode_Responses(
                **default_dict['SetPistonOperationMode_Responses']
            )
    
        return return_value
    
    
    def GetTimeOut(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetTimeOut_Responses:
        """
        Executes the unobservable command "Get time out"
            Maximum admitted duration time for specific procedure (in 0,1seconds; 100 milliseconds)
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentTimeOut (Current time out):
            Current maximum admitted duration time for specific procedure (in 0,1seconds; 100 milliseconds)
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '22'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '22'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 34        # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 34     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetTimeOut_Responses']['CurrentTimeOut'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentTimeOut': silaFW_pb2.Integer(value=int(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetTimeOut_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetTimeOut_Responses(
                **default_dict['GetTimeOut_Responses']
            )
    
        return return_value
    
    
    def SetTimeOut(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetTimeOut_Responses:
        """
        Executes the unobservable command "Set time out"
            Set the new maximum admitted duration time for specific procedure (in 0,1seconds; 100 milliseconds)
    
        :param request: gRPC request containing the parameters passed:
            request.SetTimeOut (Set time out):
            The the new admitted duration time for specific procedure in 0,1seconds; 100milliseconds.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set time out
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set time out applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetTimeOut.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '00'    # Process number in hexadecimal
            parameter_query = '22'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 0       # Process number in decimal
            parameter_query = 34    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetTimeOut_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetTimeOut_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetTimeOut_Responses(
                **default_dict['SetTimeOut_Responses']
            )
    
        return return_value
    
    
    def GetFrequency(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetFrequency_Responses:
        """
        Executes the unobservable command "Get frequency"
            Piston frequency in Hz
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFrequency (Current frequency):
            Current piston frequency in Hz
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '49'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '49'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 71        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 71     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')

                type_answer = default_dict['GetFrequency_Responses']['CurrentFrequency'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentFrequency': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetFrequency_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetFrequency_Responses(
                **default_dict['GetFrequency_Responses']
            )
    
        return return_value
    
    
    def SetFrequency(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetFrequency_Responses:
        """
        Executes the unobservable command "Set frequency"
            Set the new piston frequency in Hz
    
        :param request: gRPC request containing the parameters passed:
            request.SetFrequency (set frequency):
            New piston frequency to be set in Hz
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set frequency
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set frequency applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetFrequency.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '21'    # Process number in hexadecimal
            parameter_query = '49'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 21      # Process number in decimal
            parameter_query = 71    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetFrequency_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetFrequency_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetFrequency_Responses(
                **default_dict['SetFrequency_Responses']
            )
    
        return return_value
    
    
    def GetDensityActual(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetDensityActual_Responses:
        """
        Executes the unobservable command "Get density actual"
            reference density of selected fluid in Kg pro m3
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDensity (Current density actual): Reference density of selected fluid in Kgm pro m3
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '21'        # Process number in hexadecimal
            parameter_query = '61'      # Parameter query in hexadecimal
            process_response = '21'     # Process response in hexadecimal
            parameter_response = '61'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 33          # Process number in decimal
            parameter_query = 85        # Process response in decimal
            process_response = 33       # Parameter query in decimal
            parameter_response = 85     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetDensityActual_Responses']['CurrentDensity'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentDensity': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetDensityActual_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetDensityActual_Responses(
                **default_dict['GetDensityActual_Responses']
            )
    
        return return_value
    
    
    def SetDensityActual(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetDensityActual_Responses:
        """
        Executes the unobservable command "Set densityActual"
            Set the new reference density of the selected fluid in Kg prom3
    
        :param request: gRPC request containing the parameters passed:
            request.SetDensityActual (set densityActual):
            New density of selected fluid in Kg/m3
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set density actual
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set density actual applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetDensityActual.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '21'    # Process number in hexadecimal
            parameter_query = '61'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 33      # Process number in decimal
            parameter_query = 85    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetDensityActual_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetDensityActual_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetDensityActual_Responses(
                **default_dict['SetDensityActual_Responses']
            )
    
        return return_value
    
    
    def GetPressureInlet(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetPressureInlet_Responses:
        """
        Executes the unobservable command "Get pressure inlet"
            Pressure inlet (upstream) of fluid in bar, for the first fluid-Nr. only in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPressureInlet (Current pressure inlet):
            Pressure inlet (upstream) of fluid in bar, for the first fluid-Nr. only in millibar
        """

        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '53'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '53'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 77        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 77     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')

                type_answer = default_dict['GetPressureInlet_Responses']['CurrentPressureInlet'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')
                par_dict = {
                    'CurrentPressureInlet': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetPressureInlet_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetPressureInlet_Responses(
                **default_dict['GetPressureInlet_Responses']
            )
    
        return return_value
    
    
    def SetPressureInlet(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetPressureInlet_Responses:
        """
        Executes the unobservable command "Set pressure inlet"
            Set the new pressure inlet (upstream) of fluid in bar (first fluid only)
    
        :param request: gRPC request containing the parameters passed:
            request.SetPressureInlet (set pressure inlet):
            New pressure inlet in millibar
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set pressure inlet
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set pressure inlet applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetPressureInlet.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '53'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 113     # Process number in decimal
            parameter_query = 77    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetPressureInlet_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetPressureInlet_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetPressureInlet_Responses(
                **default_dict['SetPressureInlet_Responses']
            )
    
        return return_value
    
    
    def GetPressureOutlet(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetPressureOutlet_Responses:
        """
        Executes the unobservable command "Get pressure outlet"
            Pressure outlet (downstream) of fluid in bar, for the first fluid-Nr. only. in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPressureInlet (Current pressure inlet):
            Pressure outlet (downstream) of fluid in bar, for the first fluid-Nr. only. in millibar
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '54'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '54'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 78        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 78     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetPressureOutlet_Responses']['CurrentPressureOutlet'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentPressureOutlet': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetPressureOutlet_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetPressureOutlet_Responses(
                **default_dict['GetPressureOutlet_Responses']
            )
    
        return return_value
    
    
    def SetPressureOutlet(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetPressureOutlet_Responses:
        """
        Executes the unobservable command "Set pressure outlet"
            Set the new pressure outlet (downstream) of fluid in bar (first fluid only) in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.SetPressureOutlet (set pressure outlet):
            New pressure outlet (downstream) of fluid in bar (first fluid only) in millibar
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status):
            Command status: Request response of command set pressure outlet in millibar
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set pressure outlet applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetPressureOutlet.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '54'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 113     # Process number in decimal
            parameter_query = 78    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetPressureOutlet_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetPressureOutlet_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetPressureOutlet_Responses(
                **default_dict['SetPressureOutlet_Responses']
            )
    
        return return_value
    
    
    def GetFluidTemperature(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetFluidTemperature_Responses:
        """
        Executes the unobservable command "Get fluid temperature"
            temperature of fluid through instrument (for first fluid only) in degree Celsius
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidTemperature (Current fluid temperature): Fluid temperature through instrument (for first fluid only) in  degree Celsius
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '56'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '56'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 80        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 80     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetFluidTemperature_Responses']['CurrentFluidTemperature'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentFluidTemperature': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetFluidTemperature_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetFluidTemperature_Responses(
                **default_dict['GetFluidTemperature_Responses']
            )
    
        return return_value
    
    
    def SetFluidTemperature(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetFluidTemperature_Responses:
        """
        Executes the unobservable command "Set fluid temperature"
            Set the new temperature of fluid through instrument (for first fluid only) in Celsius degree
    
        :param request: gRPC request containing the parameters passed:
            request.SetFluidTemperature (Set fluid temperature):
            Set the new temperature of fluid through instrument (for first fluid only) in grades Celsius degree
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set fluid temperature
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set fluid temperature applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetFluidTemperature.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '56'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 113     # Process number in decimal
            parameter_query = 80    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetFluidTemperature_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetFluidTemperature_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetFluidTemperature_Responses(
                **default_dict['SetFluidTemperature_Responses']
            )
    
        return return_value
    
    
    def GetThermalConductivity(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetThermalConductivity_Responses:
        """
        Executes the unobservable command "Get thermal conductivity"
            Thermal conductivity at sensor conditions at Watt/(m*K). Value is equal or bigger than 0
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentThermalConductivity (Current thermal conductivity):
            Thermal conductivity at sensor conditions at Watt/(m*K). Value is equal or bigger than 0
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '60'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '60'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 84        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 84     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetThermalConductivity_Responses']['CurrentThermalConductivity'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentThermalConductivity': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetThermalConductivity_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetThermalConductivity_Responses(
                **default_dict['GetThermalConductivity_Responses']
            )
    
        return return_value
    
    
    def SetThermalConductivity(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetThermalConductivity_Responses:
        """
        Executes the unobservable command "SetThermalConductivity"
            Set the new thermal conductivity in sensor conditions in Watt/(m*K)
    
        :param request: gRPC request containing the parameters passed:
            request.SetThermalConductivity (set thermal conductivity):
            Set the new thermal conductivity in sensor conditions in Watt/(m*K)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set thermal conductivity
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set thermal conductivity applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetThermalConductivity.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)
        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '60'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 113     # Process number in decimal
            parameter_query = 84    # Process response in decimal
            etx = 3

            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetThermalConductivity_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetThermalConductivity_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetThermalConductivity_Responses(
                **default_dict['SetThermalConductivity_Responses']
            )
    
        return return_value
    
    
    def GetStandardMassFlow(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetStandardMassFlow_Responses:
        """
        Executes the unobservable command "Get standard mass flow"
            Standard mass flow in l/min air (20°C, 1atm, 1.01325bar) or g/h H2O equivalent. Value bigger than 0
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentStandardMassFlow (Current standard mass flow):
            Standard mass flow in l/min air (20°C, 1atm, 1.01325bar) or g/h H2O equivalent. Value bigger than 0
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '62'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '62'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 86        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 86     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetStandardMassFlow_Responses']['CurrentStandardMassFlow'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentStandardMassFlow': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetStandardMassFlow_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetStandardMassFlow_Responses(
                **default_dict['GetStandardMassFlow_Responses']
            )
    
        return return_value
    
    
    def SetStandardMassFlow(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetStandardMassFlow_Responses:
        """
        Executes the unobservable command "Set standard mass flow"
            Set the new standard mass flow in l/min air (20°C, 1atm, 1.01325bar) or g/h H2O equivalent. Value bigger than 0
    
        :param request: gRPC request containing the parameters passed:
            request.SetStandardFlow (set standard flow):
            Set the new standard mass flow in l/min air (20°C, 1atm, 1.01325bar) or g/h H2O equivalent. Value bigger than 0
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set standard flow
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set standard flow
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetStandardFlow.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)
        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '62'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 113     # Process number in decimal
            parameter_query = 86    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetStandardMassFlow_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetStandardMassFlow_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetStandardMassFlow_Responses(
                **default_dict['SetStandardMassFlow_Responses']
            )
    
        return return_value
    
    
    def GetOrificeDiameter(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetOrificeDiameter_Responses:
        """
        Executes the unobservable command "Get orifice diameter"
            Orifice diameter in mm
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentOrificeDiameter (Current orifice diameter): Orifice diameter in mm
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '55'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '55'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 79        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 79     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetOrificeDiameter_Responses']['CurrentOrificeDiameter'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentOrificeDiameter': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetOrificeDiameter_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetOrificeDiameter_Responses(
                **default_dict['GetOrificeDiameter_Responses']
            )
    
        return return_value
    
    
    def SetOrificeDiameter(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetOrificeDiameter_Responses:
        """
        Executes the unobservable command "Set orifice diameter"
            Set new orifice diameter in mm
    
        :param request: gRPC request containing the parameters passed:
            request.SetOrificeDiameter (Set new orifice diameter):
            New orifice diameter to be set
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set orifice diameter
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set orifice diameter applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetOrificeDiameter.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)
        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '71'    # Process number in hexadecimal
            parameter_query = '55'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 113     # Process number in decimal
            parameter_query = 79    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetOrificeDiameter_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetOrificeDiameter_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetOrificeDiameter_Responses(
                **default_dict['SetOrificeDiameter_Responses']
            )
    
        return return_value
    
    
    def GetBarometerPressure(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetBarometerPressure_Responses:
        """
        Executes the unobservable command "Get barometer pressure"
            barometer atmospheric pressure in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentBarometerPressure (Current barometer pressure): Barometer atmospheric pressure in mbar
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '74'        # Process number in hexadecimal
            parameter_query = '48'      # Parameter query in hexadecimal
            process_response = '74'     # Process response in hexadecimal
            parameter_response = '48'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 116         # Process number in decimal
            parameter_query = 72        # Process response in decimal
            process_response = 116      # Parameter query in decimal
            parameter_response = 72     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetBarometerPressure_Responses']['CurrentBarometerPressure'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentBarometerPressure': silaFW_pb2.Real(value=float(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetBarometerPressure_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetBarometerPressure_Responses(
                **default_dict['GetBarometerPressure_Responses']
            )
    
        return return_value
    
    
    def SetBarometerPressure(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetBarometerPressure_Responses:
        """
        Executes the unobservable command "Set barometer pressure"
            set current barometer atmospheric pressure in millibar
    
        :param request: gRPC request containing the parameters passed:
            request.SetBarometerPressure (Set barometer pressure): Set current barometer atmospheric pressure in millibar
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set barometer pressure
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set barometer pressure applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetBarometerPressure.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)
        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '74'    # Process number in hexadecimal
            parameter_query = '48'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 116     # Process number in decimal
            parameter_query = 72    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetBarometerPressure_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetBarometerPressure_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetBarometerPressure_Responses(
                **default_dict['SetBarometerPressure_Responses']
            )
    
        return return_value
    
    
    def GetNumberVanes(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.GetNumberVanes_Responses:
        """
        Executes the unobservable command "Get number vanes"
            Current number of vanes used in rotor
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentNumberVanes (Current Number vanes): Current number of vanes used in rotor
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '73'        # Process number in hexadecimal
            parameter_query = '16'      # Parameter query in hexadecimal
            process_response = '73'     # Process response in hexadecimal
            parameter_response = '16'   # Parameter response in hexadecimal
            query = f':{length}{self.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 115         # Process number in decimal
            parameter_query = 22        # Process response in decimal
            process_response = 115      # Parameter query in decimal
            parameter_response = 22     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['GetNumberVanes_Responses']['CurrentNumberVanes'].value
                response = decode_response_read(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentNumberVanes': silaFW_pb2.String(value=str(response['message']))
                }
                return_value = MeasurementSettingsService_pb2.GetNumberVanes_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.GetNumberVanes_Responses(
                **default_dict['GetNumberVanes_Responses']
            )
    
        return return_value
    
    
    def SetNumberVanes(self, request, context: grpc.ServicerContext) \
            -> MeasurementSettingsService_pb2.SetNumberVanes_Responses:
        """
        Executes the unobservable command "Set number vanes"
            Set number of vanes used in rotor
    
        :param request: gRPC request containing the parameters passed:
            request.SetNumberVanes (Set number of vanes used in rotor):
            number of vanes used in rotor
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set number vanes
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which the set number vanes
        """
    
        # initialise the return value
        return_value = None
        read = b''
        self.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.status}')

        input_integer_decimal = request.SetNumberVanes.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)
        if self.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '73'    # Process number in hexadecimal
            parameter_query = '16'  # Parameter query in hexadecimal

            query = f':{length}{self.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=write parameter
            process_query = 115     # Process number in decimal
            parameter_query = 22    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({self.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if self.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetNumberVanes_Responses']['Status'].value
                response = decode_response_write(protocol=self.protocol, message_str=read, type_answer=type_answer)
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementSettingsService_pb2.SetNumberVanes_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementSettingsService_pb2.SetNumberVanes_Responses(
                **default_dict['SetNumberVanes_Responses']
            )
    
        return return_value
