"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Service*

:details: DeviceService:
    Allows full control of the device features of the flowmeter.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    DeviceService_simulation.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-04-14T09:48:28.878095
:date: (last modification) 2021-04-14T09:48:28.878095

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceService_pb2 as DeviceService_pb2
# from .gRPC import DeviceService_pb2_grpc as DeviceService_pb2_grpc

# import default arguments
from .DeviceService_default_arguments import default_dict
from sila2lib_implementations.Flowmeter_ELFLOWPrestige.FlowmeterElflowPrestigeService.lib import  decode_response_read
from sila2lib_implementations.Flowmeter_ELFLOWPrestige.FlowmeterElflowPrestigeService import shared

# noinspection PyPep8Naming,PyUnusedLocal
class DeviceServiceSimulation:
    """
    Implementation of the *Device Service* in *Simulation* mode
        This is a flowmeter service
    """
    def __init__(self, port):
        """Class initialiser"""
        self.port = port

        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetValveOutput(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetValveOutput_Responses:
        """
        Executes the unobservable command "Get valve output"
            It represents the controller output signal for control valve operation. It goes from 0 to 300mAdc.
            In reality it reaches until 250mAdc
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentValveOutput (Current valve output):
            It shows the setpoint in the capacity unit for which the instrument is set.
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '72'        # Process number in hexadecimal
            parameter_query = '41'      # Parameter query in hexadecimal
            process_response = '72'     # Process response in hexadecimal
            parameter_response = '41'   # Parameter response in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 114         # Process number in decimal
            parameter_query = 65        # Process response in decimal
            process_response = 114      # Parameter query in decimal
            parameter_response = 65     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    read = ':080302724100013412\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02qc\x00\x07XB\x10\x03'
                message_length= len(read)
                message_type = 'value'
                type_answer = default_dict['GetValveOutput_Responses']['CurrentValveOutput'].value
                response = decode_response_read(protocol=shared.protocol,
                                                message_str=read,
                                                type_answer=type_answer,
                                                message_length=message_length,
                                                message_type=message_type
                                                )
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentValveOutput': silaFW_pb2.Real(
                        value=float(response['message_dec']))
                }
                return_value = DeviceService_pb2.GetValveOutput_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetValveOutput_Responses(
                **default_dict['GetValveOutput_Responses']
            )
    
        return return_value
    
    
    def GetCapacity100(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetCapacity100_Responses:
        """
        Executes the unobservable command "Get capacity 100"
            This parameter sets the maximum readout/control value (100%) for the current fluid in readout units corresponding to Capacity Unit.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCapacity100 (Current capacity 100):
            This parameter sets the maximum readout/control value (100%) for the current fluid in readout units corresponding to Capacity Unit.
        """
    
        # initialise the return value
        return_value = None

        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '4d'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '4d'   # Parameter response in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 77        # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 77     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    read = ':080302014D401081221\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x07\x02\x01M@\xe0\x00\x00\x10\x03'

                message_length = len(read)
                message_type = 'value'
                type_answer = default_dict['GetCapacity100_Responses']['CurrentCapacity100'].value
                response = decode_response_read(protocol=shared.protocol,
                                                message_str=read,
                                                type_answer=type_answer,
                                                message_length=message_length,
                                                message_type=message_type
                                                )
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentCapacity100': silaFW_pb2.Real(
                        value=float(response['message_dec']))
                }
                return_value = DeviceService_pb2.GetCapacity100_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetCapacity100_Responses(
                **default_dict['GetCapacity100_Responses']
            )
    
        return return_value
    
    
    def GetCapacityUnit(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetCapacityUnit_Responses:
        """
        Executes the unobservable command "Get capacity unit"
            This parameter defines the units in which is the measured value and set value are represented.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCapacityUnit (Current capacity unit):
            current  parameter which defines the units in which is the measured value and set value are represented.
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '07'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '7f'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '7f'   # Parameter response in hexadecimal
            length_response = '07'      # Length of the string response
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}{length_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 6
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 127       # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 127    # Parameter response in decimal
            length_response = 7         # Length of the string response
            etx = 3
            query = pack('BbbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response,length_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    read = ':0C0302017F076C6E2F6D696E20\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x0b\x02\x01\x7f\x07ln/min \x10\x03'
                message_length = len(read)
                message_type = 'text'
                type_answer = default_dict['GetCapacityUnit_Responses']['CurrentCapacityUnit'].value
                response = decode_response_read(protocol=shared.protocol,
                                                               message_str=read,
                                                               type_answer=type_answer,
                                                               message_length=message_length,
                                                               message_type=message_type
                                                               )
                logging.info(f'Simulation response: {response}')

                par_dict = {
                    'CurrentCapacityUnit': silaFW_pb2.String(
                        value=str(response['message_text']))
                }
                return_value = DeviceService_pb2.GetCapacityUnit_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetCapacityUnit_Responses(
                **default_dict['GetCapacityUnit_Responses']
            )
    
        return return_value
    
    
    def GetSerialNumber(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetSerialNumber_Responses:
        """
        Executes the unobservable command "Get serial number"
            Serial number of the device and software version
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSerialNumber (Current serial number): it gets the serial number of the device and software version
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '07'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '63'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '63'   # Parameter response in hexadecimal
            length_response = '00'      # Length of the string response
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}{length_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 6
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 99        # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 99     # Parameter response in decimal
            length_response = 0         # Length of the string response
            etx = 3
            query = pack('BbbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, length_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    read = ':1003027163004D31383230313835324100\r\n'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02qc\x01\x03\x05\x07\t\x0b\r\x0f\x13\x15\x17\x19\x10\x03'
                message_type='text'
                message_length = len(read)
                type_answer = default_dict['GetSerialNumber_Responses']['CurrentSerialNumber'].value
                response = decode_response_read(protocol=shared.protocol,
                                                               message_str=read,
                                                               type_answer=type_answer,
                                                               message_length=message_length,
                                                               message_type=message_type
                                                               )
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentSerialNumber': silaFW_pb2.String(
                        value=str(response['message_text']))
                }
                return_value = DeviceService_pb2.GetSerialNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetSerialNumber_Responses(
                **default_dict['GetSerialNumber_Responses']
            )
    
        return return_value
    
    
    def GetPrimaryNodeAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetPrimaryNodeAddress_Responses:
        """
        Executes the unobservable command "Get primary node address "
            Primary node address: network parameter Flow-Bus
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPrimaryNodeAddress (Current primary node address): Primary node address: network parameter Flow-Bus
        """
    
        # initialise the return value
        return_value = None

        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '00'        # Process number in hexadecimal
            parameter_query = '01'      # Parameter query in hexadecimal
            process_response = '00'     # Process response in hexadecimal
            parameter_response = '01'   # Parameter response in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 0           # Process number in decimal
            parameter_query = 1         # Process response in decimal
            process_response = 0        # Parameter query in decimal
            parameter_response = 1      # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to simulation device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    read = ':050302000103'
                else:
                    read = b'\x10\x02\x01\x03\x04\x02qc\x03\x10\x03'
                message_type = 'value'
                message_length = len(read)
                type_answer = default_dict['GetPrimaryNodeAddress_Responses']['CurrentPrimaryNodeAddress'].value
                response = decode_response_read(protocol=shared.protocol,
                                                               message_str=read,
                                                               type_answer=type_answer,
                                                               message_length=message_length,
                                                               message_type=message_type
                                                )
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentPrimaryNodeAddress': silaFW_pb2.String(
                        value=str(response['message_hex']))
                }
                return_value = DeviceService_pb2.GetPrimaryNodeAddress_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetPrimaryNodeAddress_Responses(
                **default_dict['GetPrimaryNodeAddress_Responses']
            )
    
        return return_value
    
    
    def GetFirmwareVersion(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetFirmwareVersion_Responses:
        """
        Executes the unobservable command "Get firmware version"
            revision number of the firmware
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFirmwareVersion (Current firmware version): revision number of the firmware
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')
        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '07'
            command = '04'              # 04=Request parameter
            process_query = '71'        # Process number in hexadecimal
            parameter_query = '65'      # Parameter query in hexadecimal
            process_response = '71'     # Process response in hexadecimal
            parameter_response = '65'   # Parameter response in hexadecimal
            length_response = '06'      # Length of the string response
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}{length_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 6
            command = 4                 # 04=Request parameter
            process_query = 113         # Process number in decimal
            parameter_query = 101       # Process response in decimal
            process_response = 113      # Parameter query in decimal
            parameter_response = 101    # Parameter response in decimal
            length_response = 6         # Length of the string response
            etx = 3
            query = pack('BbbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, length_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    read = ':0B030271650656312E303268\r\n'
                else:
                    read =  b'\x10\x02\x01\x03\x04\x02qc\x01\x03\x05\x07\t\x0b\r\x10\x03'
                message_type = 'text'
                message_length = len(read)
                type_answer = default_dict['GetFirmwareVersion_Responses']['CurrentFirmwareVersion'].value
                response = decode_response_read(protocol=shared.protocol,
                                                               message_str=read,
                                                               type_answer=type_answer,
                                                               message_length=message_length,
                                                               message_type=message_type
                                                               )
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentFirmwareVersion': silaFW_pb2.String(
                        value=str(response['message_text']))
                }
                return_value = DeviceService_pb2.GetFirmwareVersion_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetFirmwareVersion_Responses(
                **default_dict['GetFirmwareVersion_Responses']
            )
    
        return return_value
    
    
    def GetCommunicationProtocol(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetCommunicationProtocol_Responses:
        """
        Executes the unobservable command "Get communication protocol"
            Current communication protocol between the program adn the device.
            2 possible options: "binary" and "ascii"
            Binary is the default protocol
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCommunicationProtocol (Current communication protocol): Current communication protocol working with
        """
    
        # initialise the return value
        return_value = None
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        read = str(shared.protocol)
        logging.debug(f'Read-out: {read}')
        par_dict = {
            'CurrentCommunicationProtocol': silaFW_pb2.String(
                value=read)
        }
        return_value = DeviceService_pb2.GetCommunicationProtocol_Responses(**par_dict)


        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetCommunicationProtocol_Responses(
                **default_dict['GetCommunicationProtocol_Responses']
            )
    
        return return_value
    
    
    def SetCommunicationProtocolBinary(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetCommunicationProtocolBinary_Responses:
        """
        Executes the unobservable command "Set communication protocol binary"
            it sets the protocol communication to binary
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCommunicationProtocolBinary (Current communication protocol): Current Communication Protocol: binary
        """
    
        # initialise the return value
        return_value = None
        read = None
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        shared.protocol = "binary"
        logging.debug(f'Setting communication protocol to: {shared.protocol}')
        par_dict = {
            'CurrentCommunicationProtocolBinary': silaFW_pb2.String(value=shared.protocol),
        }
        return_value = DeviceService_pb2.SetCommunicationProtocolBinary_Responses(**par_dict)

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetCommunicationProtocolBinary_Responses(
                **default_dict['SetCommunicationProtocolBinary_Responses']
            )
    
        return return_value
    
    
    def SetCommunicationProtocolAscii(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetCommunicationProtocolAscii_Responses:
        """
        Executes the unobservable command "Set communication protocol ascii"
            it sets the protocol communication to ascii
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCommunicationProtocolAscii (Current communication protocol ascii): Current Communication Protocol: ascii
        """
    
        # initialise the return value
        return_value = None
        read = None
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        shared.protocol = "ascii"
        logging.debug(f'Setting communication protocol to: {shared.protocol}')
        par_dict = {
            'CurrentCommunicationProtocolAscii': silaFW_pb2.String(value=shared.protocol),
        }
        return_value = DeviceService_pb2.SetCommunicationProtocolAscii_Responses(**par_dict)

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetCommunicationProtocolAscii_Responses(
                **default_dict['SetCommunicationProtocolAscii_Responses']
            )
    
        return return_value
    
    
    def GetSerialPort(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetSerialPort_Responses:
        """
        Executes the unobservable command "Get serial port"
            Current devices serial port. "50001" is the predefined value
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSerialPort (Current serial port): Current serial port. "50001" is the predefined value
        """
    
        # initialise the return value
        return_value = None
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        read = str(self.port)
        logging.debug(f'Read-out: {read}')
        par_dict = {
            'CurrentSerialPort': silaFW_pb2.String(
                value=read)
        }
        return_value = DeviceService_pb2.GetSerialPort_Responses(**par_dict)

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetSerialPort_Responses(
                **default_dict['GetSerialPort_Responses']
            )
    
        return return_value
    
    
    def SetSerialPort(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetSerialPort_Responses:
        """
        Executes the unobservable command "Set serial port"
            Sets devices new serial port
    
        :param request: gRPC request containing the parameters passed:
            request.SetSerialPort (Set serial port):
            Sets devices new serial port. "50001" is the predefined value
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.NewSerialPort (New serial port): New serial port.
        """
    
        # initialise the return value
        return_value = None
        read = None
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        value = request.SetSerialPort.value
        self.port = value

        for i in range(0, 3, 1):
            try:

                par_dict = {
                    'NewSerialPort': silaFW_pb2.String(value=str(self.port)),
                }
                return_value = DeviceService_pb2.SetSerialPort_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=value, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {value}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetSerialPort_Responses(
                **default_dict['SetSerialPort_Responses']
            )
    
        return return_value
    

