"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Measurement Service*

:details: MeasurementService:
    Allows full control of the available fluid features of the flowmeter.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020

:file:    MeasurementService_real.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-04-14T09:48:28.886982
:date: (last modification) 2021-04-14T09:48:28.886982

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import inspect      # used for status determination
import grpc         # used for type hinting only
from struct import * #used to code binary protocol


# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import MeasurementService_pb2 as MeasurementService_pb2
# from .gRPC import MeasurementService_pb2_grpc as MeasurementService_pb2_grpc

# import default arguments
from .MeasurementService_default_arguments import default_dict
from sila2lib_implementations.Flowmeter_ELFLOWPrestige.FlowmeterElflowPrestigeService.lib import decode_response_write, \
    decode_response_read, decode_dec_to_binary
from sila2lib_implementations.Flowmeter_ELFLOWPrestige.FlowmeterElflowPrestigeService import shared

# noinspection PyPep8Naming,PyUnusedLocal
class MeasurementServiceReal:
    """
    Implementation of the *Measurement Service* in *Real* mode
        This is a flowmeter service
    """
    def __init__(self, ser, port):
        """Class initialiser"""
        self.ser = ser
        self.port =  port
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None


        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def _get_function_name(self):
        return inspect.stack()[1][3]

    def GetMeasuredValue(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetMeasuredValue_Responses:
        """
        Executes the unobservable command "Get measured value"
            It shows the measured value in the capacity unit for which the instrument is set.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentMeasuredValue (Current measured value):
            Current measured value in the capacity unit for which the instrument is set.
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '21'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '20'   # Parameter response in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 1          # Process number in decimal
            parameter_query =33         # Process response in decimal
            process_response = 1       # Parameter query in decimal
            parameter_response = 33     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                message_length = len(read)
                message_type = 'value'
                type_answer = default_dict['GetMeasuredValue_Responses']['CurrentMeasuredValue'].value
                response = decode_response_read(protocol=shared.protocol,
                                                               message_str=read,
                                                               type_answer=type_answer,
                                                               message_length=message_length,
                                                               message_type=message_type
                                                               )
                logging.info(f'Response: {response}')

                par_dict = {
                    'CurrentMeasuredValue': silaFW_pb2.Real(
                        value=float('%.2f' % (round((int(response['message_hex']) / 320), 2))))
                }
                return_value = MeasurementService_pb2.GetMeasuredValue_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementService_pb2.GetMeasuredValue_Responses(
                **default_dict['GetMeasuredValue_Responses']
            )
    
        return return_value
    
    
    def GetSetpointValue(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetSetpointValue_Responses:
        """
        Executes the unobservable command "Get setpoint value"
            It shows the current setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSetpointValue (Current setpoint value):
            Current setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent.
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '21'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '21'   # Parameter response in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 33        # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 33     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                message_length = len(read)
                message_type = 'value'
                type_answer = default_dict['GetSetpointValue_Responses']['GetSetpointValue'].value
                response = decode_response_read(protocol=shared.protocol,
                                                message_str=read,
                                                type_answer=type_answer,
                                                message_length=message_length,
                                                message_type=message_type
                                                )
                logging.info(f'Response: {response}')
                par_dict = {
                    'GetSetpointValue': silaFW_pb2.Real(value=float('%.2f' % (round((int(response['message_dec']) / 320), 2))))
                }
                return_value = MeasurementService_pb2.GetSetpointValue_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementService_pb2.GetSetpointValue_Responses(
                **default_dict['GetSetpointValue_Responses']
            )
    
        return return_value
    
    
    def SetSetpointValue(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.SetSetpointValue_Responses:
        """
        Executes the unobservable command "Set setpoint value"
            It sets the new setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent
    
        :param request: gRPC request containing the parameters passed:
            request.SetSetpointValue (Set Setpoint value):
            It sets the new setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command alarm limit maximum
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm limit maximum status applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        input_in_percent = request.SetSetpointValue.value
        input_integer_decimal = 320 * input_in_percent
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(4)
        split_at1= 2
        hex1, hex2 = hex_input_padded[:split_at1], hex_input_padded[split_at1:]

        hex1=int(hex1,16)
        hex2=int(hex2,16)
        hex1=decode_dec_to_binary(a=hex1)
        hex2=decode_dec_to_binary(a=hex2)

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '01'          # 01=write parameter
            process_query = '01'    # Process number in hexadecimal
            parameter_query = '21'  # Parameter query in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write parameter
            process_query = 1       # Process number in decimal
            parameter_query = 33    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(hex1), int(hex2), dle, etx)
            print(query)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':

                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))

                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')

                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetSetpointValue_Responses']['Status'].value
                response = decode_response_write(protocol=shared.protocol,
                                                 message_str=read,
                                                 type_answer=type_answer
                                                 )
                logging.info(f'Response: {response}')
                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementService_pb2.SetSetpointValue_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementService_pb2.SetSetpointValue_Responses(
                **default_dict['SetSetpointValue_Responses']
            )
    
        return return_value
    
    
    def GetFluidNumber(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetFluidNumber_Responses:
        """
        Executes the unobservable command "Get fluid number"
            Current fluid ID-Nr.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidNumber (Current fluid number): Current fluid ID-Nr. Preset of Fluid number is delimited up to 8 fluids.
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '06'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '10'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '10'   # Parameter response in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 16        # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 16     # Parameter response in decimal
            etx = 3
            query = pack('BbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                message_length = len(read)
                message_type = 'value'
                type_answer = default_dict['GetFluidNumber_Responses']['CurrentFluidNumber'].value
                response = decode_response_read(protocol=shared.protocol,
                                                message_str=read,
                                                type_answer=type_answer,
                                                message_length=message_length,
                                                message_type=message_type
                                                )
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentFluidNumber': silaFW_pb2.Integer(value=int(response['message_hex']))
                }
                return_value = MeasurementService_pb2.GetFluidNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementService_pb2.GetFluidNumber_Responses(
                **default_dict['GetFluidNumber_Responses']
            )
    
        return return_value
    
    
    def SetFluidNumber(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.SetFluidNumber_Responses:
        """
        Executes the unobservable command "Set Fluid Number"
            Set a number of the fluid to work with
    
        :param request: gRPC request containing the parameters passed:
            request.SetFluidNumber (Set fluid number):
            The number of the fluid to work with
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set fluid number
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which set fluid number applies
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        input_integer_decimal = request.SetFluidNumber.value
        hex_input = hex(int(input_integer_decimal))
        hex_input_wo_head = hex_input.split('x')[1]
        hex_input_padded = hex_input_wo_head.zfill(2)

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '05'
            command = '01'          # 01=write parameter
            process_query = '01'    # Process number in hexadecimal
            parameter_query = '10'  # Parameter query in hexadecimal
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{hex_input_padded}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 5
            command = 1             # 01=Write parameter
            process_query = 1       # Process number in decimal
            parameter_query = 16    # Process response in decimal
            etx = 3
            query = pack('BbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         int(input_integer_decimal), dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:
                if shared.protocol == 'ascii':

                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))

                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')

                logging.debug(f'Read-out: {read}')
                type_answer = default_dict['SetFluidNumber_Responses']['Status'].value
                response = decode_response_write(protocol=shared.protocol,
                                                 message_str=read,
                                                 type_answer=type_answer
                                                 )
                logging.info(f'Response: {response}')

                par_dict = {
                    'Status': silaFW_pb2.String(value=str(response['message'])),
                    'IndexPointing': silaFW_pb2.String(value=str(response['index']))
                }
                return_value = MeasurementService_pb2.SetFluidNumber_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementService_pb2.SetFluidNumber_Responses(
                **default_dict['SetFluidNumber_Responses']
            )
    
        return return_value
    
    
    def GetFluidName(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetFluidName_Responses:
        """
        Executes the unobservable command "Get fluid name"
            The name of the current fluid
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidName (Current fluid name): Current fluid name
        """
    
        # initialise the return value
        return_value = None
        read = b''
        shared.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {shared.status}')

        if shared.protocol == 'ascii':
            # ascii transfer protocol
            length = '07'
            command = '04'              # 04=Request parameter
            process_query = '01'        # Process number in hexadecimal
            parameter_query = '71'      # Parameter query in hexadecimal
            process_response = '01'     # Process response in hexadecimal
            parameter_response = '71'   # Parameter response in hexadecimal
            length_response = '0A'      # Length of the string response
            query = f':{length}{shared.node}{command}{process_query}{parameter_query}{process_response}' \
                    f'{parameter_response}{length_response}\r\n'
        else:
            # Binary transfer protocol.
            dle = 16
            stx = 2
            seq = 1
            node = 3
            length = 6
            command = 4                 # 04=Request parameter
            process_query = 1           # Process number in decimal
            parameter_query = 113       # Process response in decimal
            process_response = 1        # Parameter query in decimal
            parameter_response = 113    # Parameter response in decimal
            length_response = 10        # Length of the string response
            etx = 3
            query = pack('BbbbbbbbbbbBb', dle, stx, seq, node, length, command, process_query, parameter_query,
                         process_response, parameter_response, length_response, dle, etx)
        logging.debug(f'Writing to device ({shared.protocol}): {query}')

        for i in range(0, 3, 1):
            try:

                if shared.protocol == 'ascii':
                    self.ser.write(str.encode(query))
                    read = str(bytes.decode(self.ser.read_until('\r\n')))
                else:
                    self.ser.write(query)
                    read = self.ser.read_until('\r\n')
                logging.debug(f'Read-out: {read}')
                message_length = len(read)
                message_type = 'text'
                type_answer = default_dict['GetFluidName_Responses']['CurrentFluidName'].value
                response = decode_response_read(protocol=shared.protocol,
                                                message_str=read,
                                                type_answer=type_answer,
                                                message_length=message_length,
                                                message_type=message_type
                                                )
                logging.info(f'Simulation response: {response}')
                par_dict = {
                    'CurrentFluidName': silaFW_pb2.String(value=str(response['message_text']))
                }
                return_value = MeasurementService_pb2.GetFluidName_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=query, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = MeasurementService_pb2.GetFluidName_Responses(
                **default_dict['GetFluidName_Responses']
            )
    
        return return_value

