"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Measurement Service*

:details: MeasurementService:
    Allows full control of the available fluid features of the flowmeter.
    By Lukas Bromig and Jose de Jesus Pina, Institute of Biochemical Engineering, Technical University of Munich,
    02.12.2020
           
:file:    MeasurementService_servicer.py
:authors: Lukas Bromig and Jose de Jesus Pina

:date: (creation)          2021-04-14T10:22:07.868358
:date: (last modification) 2021-04-14T10:22:07.868358

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import MeasurementService_pb2 as MeasurementService_pb2
from .gRPC import MeasurementService_pb2_grpc as MeasurementService_pb2_grpc

# import simulation and real implementation
from .MeasurementService_simulation import MeasurementServiceSimulation
from .MeasurementService_real import MeasurementServiceReal


class MeasurementService(MeasurementService_pb2_grpc.MeasurementServiceServicer):
    """
    This is a flowmeter service
    """
    implementation: Union[MeasurementServiceSimulation, MeasurementServiceReal]
    simulation_mode: bool

    def __init__(self, ser, lock, port, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.lock = lock
        self.ser = ser
        self.port = port
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(MeasurementServiceSimulation(port=self.port))
        else:
            self._inject_implementation(MeasurementServiceReal(ser=self.ser, port=self.port))

    def _inject_implementation(self,
                               implementation: Union[MeasurementServiceSimulation,
                                                     MeasurementServiceReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the FlowmeterElflowPrestigeServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(MeasurementServiceSimulation(port=self.port))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(MeasurementServiceReal(ser=self.ser, port=self.port))

    def GetMeasuredValue(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetMeasuredValue_Responses:
        """
        Executes the unobservable command "Get measured value"
            It shows the measured value in the capacity unit for which the instrument is set.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentMeasuredValue (Current measured value):
            Current measured value in the capacity unit for which the instrument is set.
        """
    
        logging.debug(
            "GetMeasuredValue called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetMeasuredValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetSetpointValue(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetSetpointValue_Responses:
        """
        Executes the unobservable command "Get setpoint value"
            It shows the current setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.GetSetpointValue (Get setpoint value):
            Current setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent.
        """
    
        logging.debug(
            "GetSetpointValue called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetSetpointValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetSetpointValue(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.SetSetpointValue_Responses:
        """
        Executes the unobservable command "Set setpoint value"
            It sets the new setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent
    
        :param request: gRPC request containing the parameters passed:
            request.SetSetpointValue (Set Setpoint value):
            It sets the new setpoint value in the capacity unit for which the instrument is set.
            Range in Percentage. From 0.0 to 100.0 percent
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command alarm limit maximum
            request.IndexPointing (Index pointing): Index pointing to the first byte in the send message for which the alarm limit maximum status applies
        """
    
        logging.debug(
            "SetSetpointValue called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetSetpointValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFluidNumber(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetFluidNumber_Responses:
        """
        Executes the unobservable command "Get fluid number"
            Current fluid ID-Nr.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidNumber (Current fluid number): Current fluid ID-Nr. Preset of Fluid number is delimited up to 8 fluids.
        """
    
        logging.debug(
            "GetFluidNumber called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFluidNumber(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def SetFluidNumber(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.SetFluidNumber_Responses:
        """
        Executes the unobservable command "Set Fluid Number"
            Set a number of the fluid to work with
    
        :param request: gRPC request containing the parameters passed:
            request.SetFluidNumber (Set fluid number):
            The number of the fluid to work with
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Status (Status): Command status: Request response of command set fluid number
            request.IndexPointing (Index pointing):
            Index pointing to the first byte in the send message for which set fluid number applies
        """
    
        logging.debug(
            "SetFluidNumber called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.SetFluidNumber(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def GetFluidName(self, request, context: grpc.ServicerContext) \
            -> MeasurementService_pb2.GetFluidName_Responses:
        """
        Executes the unobservable command "Get fluid name"
            The name of the current fluid
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFluidName (Current fluid name): Current fluid name
        """
    
        logging.debug(
            "GetFluidName called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        try:
            return self.implementation.GetFluidName(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    
