"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Information Provider*

:details: DeviceInformationProvider:
    General device information regarding firmware and hardware can be retrieved and changed within this feature.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 07.04.2021

:file:    DeviceInformationProvider_real.py
:authors: Lukas Bromig

:date: (creation)          2021-04-09T13:29:06.783095
:date: (last modification) 2021-04-09T13:29:06.783095

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only
import inspect      # used for status determination

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceInformationProvider_pb2 as DeviceInformationProvider_pb2
# from .gRPC import DeviceInformationProvider_pb2_grpc as DeviceInformationProvider_pb2_grpc

# import default arguments
from .DeviceInformationProvider_default_arguments import default_dict

# import SiLA Defined Error factories
from .DeviceInformationProvider_defined_errors import generate_def_error_InternalError, generate_def_error_LogicalError


# noinspection PyPep8Naming,PyUnusedLocal
class DeviceInformationProviderReal:
    """
    Implementation of the *Device Information Provider* in *Real* mode
        This is a Mettler Toledo Viper SW balance service
    """

    def __init__(self, hardware_interface=None):
        """Class initialiser"""

        self.hardware_interface = hardware_interface

        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    @staticmethod
    def _get_function_name():
        return inspect.stack()[1][3]

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def Reset(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Reset_Responses:
        """
        Executes the unobservable command "Reset"
            Get the current status of the device from the state machine of the SiLA server.
    
        :param request: gRPC request containing the parameters passed:
            request.Confirmation (Confirmation): Confirm the reset command with Y.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SerialNumber (Serial Number): The serial number is emitted; the weigh module/balance is ready for operation.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'@\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    read = read[:-2]
                    read = read.split('A "')[1]
                    logging.debug('Executed command Reset in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    par_dict = {
                        'SerialNumber': silaFW_pb2.String(value=read),
                    }
                    return_value = DeviceInformationProvider_pb2.Reset_Responses(**par_dict)
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Reset_Responses(
                **default_dict['Reset_Responses']
                # SerialNumber=silaFW_pb2.String(value='default string')
            )
    
        return return_value

    def Get_ImplementedCommands(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Get_ImplementedCommands_Responses:
        """
        Requests the unobservable property Implemented Commands
            Lists all commands implemented in the present software version. All commands (MT-SICS) ordered according to level in alphabetical order.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.ImplementedCommands (Implemented Commands): Lists all commands implemented in the present software version. All commands (MT-SICS) ordered according to level in alphabetical order.
        """
    
        # initialise the return value
        return_value: DeviceInformationProvider_pb2.Get_ImplementedCommands_Responses = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'I0\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    logging.debug('Executed command Get_ImplementedCommands in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    read = read[:-2]
                    if read != 'I0 I':
                        implemented_command_list = []
                        for implemented_command in read:
                            implemented_command_list.append(silaFW_pb2.String(value=implemented_command))
                        par_dict = {
                            'ImplementedCommands': implemented_command_list,
                        }
                        return_value = DeviceInformationProvider_pb2.Get_ImplementedCommands_Responses(**par_dict)
                    elif read == 'I0 I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Get_ImplementedCommands_Responses(
                **default_dict['Get_ImplementedCommands_Responses']
                # ImplementedCommands=[silaFW_pb2.String(value='default string')]
            )
    
        return return_value
    
    def Get_DeviceType(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Get_DeviceType_Responses:
        """
        Requests the unobservable property Device Type
            Query device type.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.DeviceType (Device Type): Query device type.
        """
    
        # initialise the return value
        return_value: DeviceInformationProvider_pb2.Get_DeviceType_Responses = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'I2\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    logging.debug('Executed command Get_DeviceType in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    read = read[:-2]
                    if 'I2 A' in read:
                        msg = read.split('A "')[1]
                        device_type = msg[0]
                        if len(msg) >= 3:
                            for entry in msg[1:-2]:
                                device_type = device_type + ' ' + entry
                        par_dict = {
                            'DeviceType': silaFW_pb2.String(value=device_type)
                        }
                        return_value = DeviceInformationProvider_pb2.Get_DeviceType_Responses(**par_dict)
                    elif read == 'I2 I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Get_DeviceType_Responses(
                **default_dict['Get_DeviceType_Responses']
                # DeviceType=silaFW_pb2.String(value='default string')
            )
    
        return return_value
    
    def Get_WeighingCapacity(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Get_WeighingCapacity_Responses:
        """
        Requests the unobservable property Weighing Capacity
            Query weighing capacity. The maximum allowed balance capacity in g.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.WeighingCapacity (Weighing Capacity): Query weighing capacity. The maximum allowed balance capacity in g.
        """
    
        # initialise the return value
        return_value: DeviceInformationProvider_pb2.Get_WeighingCapacity_Responses = None
    
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')
        unit_conversion_dict = {'mg': 10 ** -3, 'g': 10 ** 0, 'kg': 10 ** 3, 't': 10 ** 6}
        command = f'I2\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    logging.debug('Executed command Get_WeighingCapacity in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    read = read[:-2]
                    if 'I2 A' in read:
                        msg = read.split('"')[1]
                        capacity, unit = msg.split('-')[-2:]
                        value = float(capacity) * unit_conversion_dict[unit]
                        par_dict = {
                            'WeighingCapacity': silaFW_pb2.Real(value=value)
                        }
                        return_value = DeviceInformationProvider_pb2.Get_WeighingCapacity_Responses(**par_dict)
                    elif read == 'I2 I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Get_WeighingCapacity_Responses(
                **default_dict['Get_WeighingCapacity_Responses']
                # WeighingCapacity=silaFW_pb2.Real(value=1.0)
            )
    
        return return_value

    def Get_FirmwareVersion(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Get_FirmwareVersion_Responses:
        """
        Requests the unobservable property Firmware Version
            Provides the software version number
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.FirmwareVersion (Firmware Version): Provides the software version number
        """
    
        # initialise the return value
        return_value: DeviceInformationProvider_pb2.Get_FirmwareVersion_Responses = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'I3\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    logging.debug('Executed command Get_FirmwareVersion in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    read = read[:-2]
                    if 'I3 A' in read:
                        msg = read.split('"')[1]
                        firmware_version = msg.split('-')[0]
                        par_dict = {
                            'FirmwareVersion': silaFW_pb2.String(value=firmware_version)
                        }
                        return_value = DeviceInformationProvider_pb2.Get_FirmwareVersion_Responses(**par_dict)
                    elif read == 'I3 I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Get_FirmwareVersion_Responses(
                **default_dict['Get_FirmwareVersion_Responses']
                # FirmwareVersion=silaFW_pb2.String(value='default string')
            )
    
        return return_value
    
    def Get_TypeDefinitionNumber(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Get_TypeDefinitionNumber_Responses:
        """
        Requests the unobservable property Type Definition Number
            Provides the type definition number
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TypeDefinitionNumber (Type Definition Number): Provides the type definition number
        """
    
        # initialise the return value
        return_value: DeviceInformationProvider_pb2.Get_TypeDefinitionNumber_Responses = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'I3\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    logging.debug('Executed command Get_TypeDefinitionNumber in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    read = read[:-2]
                    if 'I3 A' in read:
                        msg = read.split('"')[1]
                        type_definition_number = msg.split(' ')[1]
                        par_dict = {
                            'TypeDefinitionNumber': silaFW_pb2.String(value=type_definition_number)
                        }
                        return_value = DeviceInformationProvider_pb2.Get_TypeDefinitionNumber_Responses(**par_dict)
                    elif read == 'I3 I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Get_TypeDefinitionNumber_Responses(
                **default_dict['Get_TypeDefinitionNumber_Responses']
                # TypeDefinitionNumber=silaFW_pb2.String(value='default string')
            )
    
        return return_value
    
    def Get_SerialNumber(self, request, context: grpc.ServicerContext) \
            -> DeviceInformationProvider_pb2.Get_SerialNumber_Responses:
        """
        Requests the unobservable property Serial Number
            Query the serial number of the balance terminal. The serial number agrees with that on the model plate and is different for every MT balance. If no terminal is present, the SN of the bridge is issued instead.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.SerialNumber (Serial Number): Query the serial number of the balance terminal. The serial number agrees with that on the model plate and is different for every MT balance. If no terminal is present, the SN of the bridge is issued instead.
        """
    
        # initialise the return value
        return_value: DeviceInformationProvider_pb2.Get_SerialNumber_Responses = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'I4\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r\n'))))
                    logging.debug('Executed command Get_SerialNumber in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    read = read[:-2]
                    if 'I4 A' in read:
                        serial_number = read.split('"')[1]
                        par_dict = {
                            'SerialNumber': silaFW_pb2.String(value=serial_number)
                        }
                        return_value = DeviceInformationProvider_pb2.Get_SerialNumber_Responses(**par_dict)
                    elif read == 'I4 I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceInformationProvider_pb2.Get_SerialNumber_Responses(
                **default_dict['Get_SerialNumber_Responses']
                # SerialNumber=silaFW_pb2.String(value='default string')
            )
    
        return return_value
