"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Balance Service*

:details: BalanceService:
    This feature enables several modes of weight measuring and allows taring and zeroing of the balance.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 07.04.2021

:file:    BalanceService_real.py
:authors: Lukas Bromig

:date: (creation)          2021-04-07T13:30:01.304996
:date: (last modification) 2021-04-07T13:30:01.304996

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only
import inspect      # used for status determination

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import BalanceService_pb2 as BalanceService_pb2
# from .gRPC import BalanceService_pb2_grpc as BalanceService_pb2_grpc

# import default arguments
from .BalanceService_default_arguments import default_dict

# import SiLA Defined Error factories
from .BalanceService_defined_errors import generate_def_error_OverloadError, generate_def_error_UnderloadError, generate_def_error_InternalError, generate_def_error_LogicalError
from sila2lib.error_handling.server_err import SiLAExecutionError


# noinspection PyPep8Naming,PyUnusedLocal
class BalanceServiceReal:
    """
    Implementation of the *Balance Service* in *Real* mode
        This is a Mettler Toledo Viper SW balance service
    """

    def __init__(self, hardware_interface=None):
        """Class initialiser"""

        self.hardware_interface = hardware_interface

        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    @staticmethod
    def _get_function_name():
        return inspect.stack()[1][3]

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def Zero(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Zero_Responses:
        """
        Executes the unobservable command "Zero"
            Set a new zero; all weight values (including tare weight) will be measured relative to this zero. After zeroing has taken place, the following values applky: tare weight = 0; net weight = (=gross weight) = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Success (Success): Zero setting successfully performed. Gross, net and tare = 0.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'Z\r\n'
        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command Zero in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    if read == 'Z A':
                        par_dict = {
                            'Success': silaFW_pb2.Boolean(value=True),
                        }
                        return_value = BalanceService_pb2.Zero_Responses(**par_dict)
                    elif read == 'Z I':
                        raise generate_def_error_InternalError()
                    elif read == 'Z +':
                        raise generate_def_error_OverloadError()
                    elif read == 'Z -':
                        raise generate_def_error_UnderloadError()
                    else:
                        raise ValueError
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.Zero_Responses(
                **default_dict['Zero_Responses']
            )
    
        return return_value

    def ZeroImmediate(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.ZeroImmediate_Responses:
        """
        Executes the unobservable command "Zero Immediate"
            Set a new zero immediately, regardless of balance stability. All weight values (including tare weight) will be measured relative to this zero. After zeroing has taken place, the following values applky: tare weight = 0; net weight = (=gross weight) = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ReZero (ReZero): Zero setting successfully performed. Gross, net and tare = 0.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        possible_responses = ['ZI D', 'ZI S', 'ZI I', 'ZI +', 'ZI -']

        command = f'ZI\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command ZeroImmediate in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    if read == 'ZI D':
                        par_dict = {
                            'ReZero': BalanceService_pb2.ZeroImmediate_Responses.ReZero_Struct(
                                Success=silaFW_pb2.Boolean(value=True),
                                IsStable=silaFW_pb2.Boolean(value=bool(False)),
                            )
                        }
                        return_value = BalanceService_pb2.ZeroImmediate_Responses(**par_dict)
                    elif read == 'ZI S':
                        par_dict = {
                            'ReZero': BalanceService_pb2.ZeroImmediate_Responses.ReZero_Struct(
                                Success=silaFW_pb2.Boolean(value=True),
                                IsStable=silaFW_pb2.Boolean(value=True),
                            )
                        }
                        return_value = BalanceService_pb2.ZeroImmediate_Responses(**par_dict)
                    elif read == 'ZI I':
                        raise generate_def_error_InternalError()
                    elif read == 'ZI +':
                        raise generate_def_error_OverloadError()
                    elif read == 'ZI -':
                        raise generate_def_error_UnderloadError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.ZeroImmediate_Responses(
                **default_dict['ZeroImmediate_Responses']
            )
    
        return return_value

    def Tare(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Tare_Responses:
        """
        Executes the unobservable command "Tare"
            Tare the balance. The next stable weight value will be saved in the tare memory.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareValue (Tare Value): The tare value if taring has been successfully performed. The tare weight value returned corresponds to the weight change on the balance in the unit actually set under display unit since the last zero setting.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        unit_conversion_dict = {'mg': 10**-3, 'g': 10**0, 'kg': 10**3, 't': 10**6}
        command = f'T\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command Tare in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    if 'T S' in read:
                        tmp = read.split(' ')
                        unit = tmp[-1]
                        value = float(tmp[-2])*unit_conversion_dict[unit]
                        par_dict = {
                            'TareValue': silaFW_pb2.Real(value=value)
                        }
                        return_value = BalanceService_pb2.Tare_Responses(**par_dict)
                    elif read == 'T L':
                        raise generate_def_error_LogicalError()
                    elif read == 'T I':
                        raise generate_def_error_InternalError()
                    elif read == 'T +':
                        raise generate_def_error_OverloadError()
                    elif read == 'T -':
                        raise generate_def_error_UnderloadError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.Tare_Responses(
                **default_dict['Tare_Responses']
                # TareValue=silaFW_pb2.Real(value=1.0)
            )
    
        return return_value

    def WeightValueOnChange(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.CommandConfirmation:
        """
        Executes the observable command "Weight Value on Change"
            Request the current stable weight value in display unit followed by weight values after predefined minimum weight changes until the command is stopped, i.e. sends the current stable weight value followed by every load change of predefined amount of g.
    
        :param request: gRPC request containing the parameters passed:
            request.WeightChange (WeightChange): Predefined minimum weight change in g. If no value is entered, the weight change must be at least 12.5% of the last stable weight value.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A command confirmation object with the following information:
            commandId: A command id with which this observable command can be referenced in future calls
            lifetimeOfExecution: The (maximum) lifetime of this command call.
        """
    
        # initialise default values
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        lifetime_of_execution: silaFW_pb2.Duration = None
    
        # TODO:
        #   Execute the actual command
        #   Optional: Generate a lifetime_of_execution
    
        # respond with UUID and lifetime of execution
        command_uuid = silaFW_pb2.CommandExecutionUUID(value=str(uuid.uuid4()))
        if lifetime_of_execution is not None:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid,
                lifetimeOfExecution=lifetime_of_execution
            )
        else:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid
            )

    def WeightValueOnChange_Info(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.ExecutionInfo:
        """
        Returns execution information regarding the command call :meth:`~.WeightValueOnChange`.
    
        :param request: A request object with the following properties
            commandId: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: An ExecutionInfo response stream for the command with the following fields:
            commandStatus: Status of the command (enumeration)
            progressInfo: Information on the progress of the command (0 to 1)
            estimatedRemainingTime: Estimate of the remaining time required to run the command
            updatedLifetimeOfExecution: An update on the execution lifetime
        """
        # Get the UUID of the command
        command_uuid = request.value
    
        # Get the current state
        execution_info = self._get_command_state(command_uuid=command_uuid)
    
        # construct the initial return dictionary in case while is not executed
        return_values = {'commandStatus': execution_info.commandStatus}
        if execution_info.HasField('progressInfo'):
            return_values['progressInfo'] = execution_info.progressInfo
        if execution_info.HasField('estimatedRemainingTime'):
            return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
        if execution_info.HasField('updatedLifetimeOfExecution'):
            return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
        # we loop only as long as the command is running
        while execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.waiting \
                or execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.running:
            # TODO:
            #   Evaluate the command status --> command_status. Options:
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.running
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedSuccessfully
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedWithError
            #   Optional:
            #       * Determine the progress (progressInfo)
            #       * Determine the estimated remaining time
            #       * Update the Lifetime of execution
    
            # Update all values
            execution_info = self._get_command_state(command_uuid=command_uuid)
    
            # construct the return dictionary
            return_values = {'commandStatus': execution_info.commandStatus}
            if execution_info.HasField('progressInfo'):
                return_values['progressInfo'] = execution_info.progressInfo
            if execution_info.HasField('estimatedRemainingTime'):
                return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
            if execution_info.HasField('updatedLifetimeOfExecution'):
                return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
            # we add a small delay to give the client a chance to keep up.
            time.sleep(0.5)
        else:
            # one last time yield the status
            yield silaFW_pb2.ExecutionInfo(**return_values)

    def WeightValueOnChange_Result(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.WeightValueOnChange_Responses:
        """
        Returns the final result of the command call :meth:`~.WeightValueOnChange`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.WeightValue (Weight Value): The returned weight value. Either stable weight value on first or next change, or dynamic weight value in g.
        """
    
        # initialise the return value
        return_value: BalanceService_pb2.WeightValueOnChange_Responses = None
    
        # Get the UUID of the command
        command_uuid = request.value
    
        # TODO:
        #   Add implementation of Real for command WeightValueOnChange here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.WeightValueOnChange_Responses(
                **default_dict['WeightValueOnChange_Responses']
                # WeightValue=a
            )
    
        return return_value

    def PresetTareWeight(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.PresetTareWeight_Responses:
        """
        Executes the unobservable command "PresetTareWeight"
            Preset a known tare weight value in g.
    
        :param request: gRPC request containing the parameters passed:
            request.TarePresetValue (Tare Preset Value): The preset value to be set for tare in g.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareWeightValue (Tare Weight Value): Get the current tare weight value in g.
        """
    
        # initialise the return value
        return_value = None
        read = None

        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        tare_preset_value = '{:.2f}'.format(request.TarePresetValue.value)
        unit = 'g'
        command = f'TA {tare_preset_value} {unit}\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command PresetTareWeight in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    if 'TA A' in read:
                        tmp = read.split(' ')
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'TareWeightValue': silaFW_pb2.Real(value=value)
                        }
                        return_value = BalanceService_pb2.PresetTareWeight_Responses(**par_dict)
                    elif read == 'TA L':
                        raise generate_def_error_LogicalError()
                    elif read == 'TA I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.PresetTareWeight_Responses(
                **default_dict['PresetTareWeight_Responses']
                # TareWeightValue=silaFW_pb2.Real(value=1.0)
            )
    
        return return_value

    def ClearTareValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.ClearTareValue_Responses:
        """
        Executes the unobservable command "Clear Tare Value"
            Clear tare weight value from memory.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareWeightValue (Tare Weight Value): The reset tare weight value in g. Is always 0.
        """
    
        # initialise the return value
        return_value = None
        read = None

        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        command = f'TAC\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command ClearTareValue in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    if read == 'TAC A':
                        value = 0.0
                        par_dict = {
                            'TareWeightValue': silaFW_pb2.Real(value=value)
                        }
                        return_value = BalanceService_pb2.PresetTareWeight_Responses(**par_dict)
                    elif read == 'TAC L':
                        raise generate_def_error_LogicalError()
                    elif read == 'TAC I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.ClearTareValue_Responses(
                **default_dict['ClearTareValue_Responses']
                # TareWeightValue=silaFW_pb2.Real(value=1.0)
            )
    
        return return_value

    def TareImmediately(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.TareImmediately_Responses:
        """
        Executes the unobservable command "Tare Immediately"
            Tare the balance immediately and independently of balance stability.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareWeightValue (Tare Weight Value): Taring performed successfully with the stable or non-stable taring weight value  in g. The new tare value corresponds to the weight change on the balance since the last zero setting.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        unit_conversion_dict = {'mg': 10 ** -3, 'g': 10 ** 0, 'kg': 10 ** 3, 't': 10 ** 6}
        command = f'TI\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command TareImmediately in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    tmp = read.split(' ')
                    if 'TI S' in read:
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'TareWeightValue': BalanceService_pb2.TareImmediately_Responses.TareWeightValue_Struct(
                                TareWeightValue=silaFW_pb2.Real(value=value),
                                IsStable=silaFW_pb2.Boolean(value=True))
                        }
                        return_value = BalanceService_pb2.TareImmediately_Responses(**par_dict)
                    if 'TI D' in read:
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'TareWeightValue': BalanceService_pb2.TareImmediately_Responses.TareWeightValue_Struct(
                                TareWeightValue=silaFW_pb2.Real(value=value),
                                IsStable=silaFW_pb2.Boolean(value=False))
                        }
                        return_value = BalanceService_pb2.TareImmediately_Responses(**par_dict)
                    elif read == 'TI L':
                        raise generate_def_error_LogicalError()
                    elif read == 'TI I':
                        raise generate_def_error_InternalError()
                    elif read == 'TI +':
                        raise generate_def_error_OverloadError()
                    elif read == 'TI -':
                        raise generate_def_error_UnderloadError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.TareImmediately_Responses(
                **default_dict['TareImmediately_Responses']
                # TareWeightValue=a
            )
    
        return return_value

    def Get_StableWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Get_StableWeightValue_Responses:
        """
        Requests the unobservable property Stable Weight Value
            Get the current stable weight value in g.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.StableWeightValue (Stable Weight Value): Get the current stable weight value in g.
        """
    
        # initialise the return value
        return_value: BalanceService_pb2.Get_StableWeightValue_Responses = None
    
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        unit_conversion_dict = {'mg': 10 ** -3, 'g': 10 ** 0, 'kg': 10 ** 3, 't': 10 ** 6}
        command = f'S\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command Get_StableWeightValue in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    tmp = read.split(' ')
                    if 'S S' in read:
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'StableWeightValue': silaFW_pb2.Real(value=value)
                        }
                        return_value = BalanceService_pb2.Get_StableWeightValue_Responses(**par_dict)
                    elif read == 'S I':
                        raise generate_def_error_InternalError()
                    elif read == 'S +':
                        raise generate_def_error_OverloadError()
                    elif read == 'S -':
                        raise generate_def_error_UnderloadError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.Get_StableWeightValue_Responses(
                **default_dict['Get_StableWeightValue_Responses']
                # StableWeightValue=silaFW_pb2.Real(value=1.0)
            )
    
        return return_value

    def Get_ImmediateWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Get_ImmediateWeightValue_Responses:
        """
        Requests the unobservable property ImmediateWeightValue
            Get the immediate (potentially unstable) weight value
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.ImmediateWeightValue (ImmediateWeightValue): Get the immediate (potentially unstable) weight value
        """
    
        # initialise the return value
        return_value: BalanceService_pb2.Get_ImmediateWeightValue_Responses = None
        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        unit_conversion_dict = {'mg': 10 ** -3, 'g': 10 ** 0, 'kg': 10 ** 3, 't': 10 ** 6}
        command = f'SI\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command Get_ImmediateWeightValue in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    tmp = read.split(' ')
                    if 'S S' in read:
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'ImmediateWeightValue': BalanceService_pb2.Get_ImmediateWeightValue_Responses.ImmediateWeightValue_Struct(
                                WeightValue=silaFW_pb2.Real(value=value),
                                IsStable=silaFW_pb2.Boolean(value=True))
                        }
                        return_value = BalanceService_pb2.Get_ImmediateWeightValue_Responses(**par_dict)
                    if 'S D' in read:
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'ImmediateWeightValue': BalanceService_pb2.Get_ImmediateWeightValue_Responses.ImmediateWeightValue_Struct(
                                WeightValue=silaFW_pb2.Real(value=value),
                                IsStable=silaFW_pb2.Boolean(value=False))
                        }
                        return_value = BalanceService_pb2.Get_ImmediateWeightValue_Responses(**par_dict)
                    elif read == 'S L':
                        raise generate_def_error_LogicalError()
                    elif read == 'S I':
                        raise generate_def_error_InternalError()
                    elif read == 'S +':
                        raise generate_def_error_OverloadError()
                    elif read == 'S -':
                        raise generate_def_error_UnderloadError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None

        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.Get_ImmediateWeightValue_Responses(
                **default_dict['Get_ImmediateWeightValue_Responses']
                # ImmediateWeightValue=a
            )
    
        return return_value

    def Subscribe_CurrentWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Subscribe_CurrentWeightValue_Responses:
        """
        Requests the observable property Current Weight Value
            Subscribe to the current weight value
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.CurrentWeightValue (Current Weight Value): Subscribe to the current weight value
        """
    
        # initialise the return value
        return_value: BalanceService_pb2.Subscribe_CurrentWeightValue_Responses = None
    
        # we could use a timeout here if we wanted
        unit_conversion_dict = {'mg': 10 ** -3, 'g': 10 ** 0, 'kg': 10 ** 3, 't': 10 ** 6}
        command = f'SIR\r\n'
        while True:
            i=0
            try:
                for i in range(0, 3, 1):
                    try:
                        with self.hardware_interface.acquire_timeout_lock():
                            self.hardware_interface.ser.write(str.encode(command))
                            read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                            logging.debug('Executed command Subscribe_CurrentWeightValue in mode: {mode} with response: {response}'
                                          .format(mode='Real', response=read))
                            tmp = read.split(' ')
                            if 'S S' in read:
                                unit = tmp[-1]
                                value = float(tmp[-2]) * unit_conversion_dict[unit]
                                par_dict = {
                                    'CurrentWeightValue': BalanceService_pb2.Subscribe_CurrentWeightValue_Responses.CurrentWeightValue_Struct(
                                        WeightValue=silaFW_pb2.Real(value=value),
                                        IsStable=silaFW_pb2.Boolean(value=True))
                                }
                                return_value = BalanceService_pb2.Subscribe_CurrentWeightValue_Responses(**par_dict)
                            if 'S D' in read:
                                unit = tmp[-1]
                                value = float(tmp[-2]) * unit_conversion_dict[unit]
                                par_dict = {
                                    'CurrentWeightValue': BalanceService_pb2.Subscribe_CurrentWeightValue_Responses.CurrentWeightValue_Struct(
                                        WeightValue=silaFW_pb2.Real(value=value),
                                        IsStable=silaFW_pb2.Boolean(value=False))
                                }
                                return_value = BalanceService_pb2.Subscribe_CurrentWeightValue_Responses(**par_dict)

                            elif read == 'S I':
                                raise generate_def_error_InternalError()
                            elif read == 'S +':
                                raise generate_def_error_OverloadError()
                            elif read == 'S -':
                                raise generate_def_error_UnderloadError()
                            time.sleep(2)
                    except SiLAExecutionError:
                        # Todo: Forward the SiLAExecutionError properly. Right now it is not evaluated properly
                        if read == 'S I':
                            return_value = generate_def_error_InternalError
                            print(return_value)
                        elif read == 'S +':

                            return_value = generate_def_error_OverloadError
                            print(return_value)
                        elif read == 'S -':
                            return_value = generate_def_error_UnderloadError
                            print(return_value)
                        continue
                    break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
            # create the default value
            if return_value is None:
                return_value = BalanceService_pb2.Subscribe_CurrentWeightValue_Responses(
                    **default_dict['Subscribe_CurrentWeightValue_Responses']
                    # CurrentWeightValue=a
                )
            yield return_value

    def Get_TareWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Get_TareWeightValue_Responses:
        """
        Requests the unobservable property Tare Weight Value
            Get the current tare weight value in g.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TareWeightValue (Tare Weight Value): Get the current tare weight value in g.
        """
    
        # initialise the return value
        return_value: BalanceService_pb2.Get_TareWeightValue_Responses = None

        read = None
        self.hardware_interface.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.hardware_interface.status}')

        unit_conversion_dict = {'mg': 10 ** -3, 'g': 10 ** 0, 'kg': 10 ** 3, 't': 10 ** 6}
        command = f'TA\r\n'

        for i in range(0, 3, 1):
            try:
                with self.hardware_interface.acquire_timeout_lock():
                    self.hardware_interface.ser.write(str.encode(command))
                    read = str(bytes.decode(self.hardware_interface.ser.read_until(str.encode('\r'))))
                    logging.debug('Executed command Get_TareWeightValue in mode: {mode} with response: {response}'
                                  .format(mode='Real', response=read))
                    tmp = read.split(' ')
                    if 'TA A' in read:
                        unit = tmp[-1]
                        value = float(tmp[-2]) * unit_conversion_dict[unit]
                        par_dict = {
                            'TareWeightValue': silaFW_pb2.Real(value=value)
                        }
                        return_value = BalanceService_pb2.Get_TareWeightValue_Responses(**par_dict)
                    elif read == 'TA L':
                        raise generate_def_error_LogicalError()
                    elif read == 'TA I':
                        raise generate_def_error_InternalError()
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = BalanceService_pb2.Get_TareWeightValue_Responses(
                **default_dict['Get_TareWeightValue_Responses']
                # TareWeightValue=silaFW_pb2.Real(value=1.0)
            )
    
        return return_value
