"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Balance Service*

:details: BalanceService:
    This feature enables several modes of weight measuring and allows taring and zeroing of the balance.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 07.04.2021
           
:file:    BalanceService_servicer.py
:authors: Lukas Bromig

:date: (creation)          2021-04-07T13:30:01.275445
:date: (last modification) 2021-04-07T13:30:01.275445

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError, SiLAValidationError

# import gRPC modules for this feature
from .gRPC import BalanceService_pb2 as BalanceService_pb2
from .gRPC import BalanceService_pb2_grpc as BalanceService_pb2_grpc

# import simulation and real implementation
from .BalanceService_simulation import BalanceServiceSimulation
from .BalanceService_real import BalanceServiceReal

# import SiLA Defined Error factories
from .BalanceService_defined_errors import generate_def_error_OverloadError, generate_def_error_UnderloadError, generate_def_error_InternalError, generate_def_error_LogicalError

class BalanceService(BalanceService_pb2_grpc.BalanceServiceServicer):
    """
    This is a Mettler Toledo Viper SW balance service
    """
    implementation: Union[BalanceServiceSimulation, BalanceServiceReal]
    simulation_mode: bool

    def __init__(self, simulation_mode: bool = True, hardware_interface=None):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        :param hardware_interface (optional): access to shared hardware interface, like serial interface. 
        """

        self.hardware_interface = hardware_interface

        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(BalanceServiceSimulation(hardware_interface=self.hardware_interface))
        else:
            self._inject_implementation(BalanceServiceReal(hardware_interface=self.hardware_interface))

    def _inject_implementation(self,
                               implementation: Union[BalanceServiceSimulation,
                                                     BalanceServiceReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the MT_Viper_SW_Balance_ServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(BalanceServiceSimulation(hardware_interface=self.hardware_interface))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(BalanceServiceReal(hardware_interface=self.hardware_interface))

    def Zero(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Zero_Responses:
        """
        Executes the unobservable command "Zero"
            Set a new zero; all weight values (including tare weight) will be measured relative to this zero. After zeroing has taken place, the following values applky: tare weight = 0; net weight = (=gross weight) = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.Success (Success): Zero setting successfully performed. Gross, net and tare = 0.
        """
    
        logging.debug(
            "Zero called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.Zero(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def ZeroImmediate(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.ZeroImmediate_Responses:
        """
        Executes the unobservable command "Zero Immediate"
            Set a new zero immediately, regardless of balance stability. All weight values (including tare weight) will be measured relative to this zero. After zeroing has taken place, the following values applky: tare weight = 0; net weight = (=gross weight) = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ReZero (ReZero): Zero setting successfully performed. Gross, net and tare = 0.
        """
    
        logging.debug(
            "ZeroImmediate called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.ZeroImmediate(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def Tare(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Tare_Responses:
        """
        Executes the unobservable command "Tare"
            Tare the balance. The next stable weight value will be saved in the tare memory.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareValue (Tare Value): The tare value if taring has been successfully performed. The tare weight value returned corresponds to the weight change on the balance in the unit actually set under display unit since the last zero setting.
        """
    
        logging.debug(
            "Tare called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.Tare(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def WeightValueOnChange(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.CommandConfirmation:
        """
        Executes the observable command "Weight Value on Change"
            Request the current stable weight value in display unit followed by weight values after predefined minimum weight changes until the command is stopped, i.e. sends the current stable weight value followed by every load change of predefined amount of g.
    
        :param request: gRPC request containing the parameters passed:
            request.WeightChange (WeightChange): Predefined minimum weight change in g. If no value is entered, the weight change must be at least 12.5% of the last stable weight value.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A command confirmation object with the following information:
            commandId: A command id with which this observable command can be referenced in future calls
            lifetimeOfExecution: The (maximum) lifetime of this command call.
        """
    
        logging.debug(
            "WeightValueOnChange called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.WeightValueOnChange(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def WeightValueOnChange_Info(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.ExecutionInfo:
        """
        Returns execution information regarding the command call :meth:`~.WeightValueOnChange`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: An ExecutionInfo response stream for the command with the following fields:
            commandStatus: Status of the command (enumeration)
            progressInfo: Information on the progress of the command (0 to 1)
            estimatedRemainingTime: Estimate of the remaining time required to run the command
            updatedLifetimeOfExecution: An update on the execution lifetime
        """
    
        logging.debug(
            "WeightValueOnChange_Info called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            return self.implementation.WeightValueOnChange_Info(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def WeightValueOnChange_Result(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.WeightValueOnChange_Responses:
        """
        Returns the final result of the command call :meth:`~.WeightValueOnChange`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.WeightValue (Weight Value): The returned weight value. Either stable weight value on first or next change, or dynamic weight value in g.
        """
    
        logging.debug(
            "WeightValueOnChange_Result called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            return self.implementation.WeightValueOnChange_Result(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    
    def PresetTareWeight(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.PresetTareWeight_Responses:
        """
        Executes the unobservable command "PresetTareWeight"
            Preset a known tare weight value in g.
    
        :param request: gRPC request containing the parameters passed:
            request.TarePresetValue (Tare Preset Value): The preset value to be set for tare in g.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareWeightValue (Tare Weight Value): Get the current tare weight value in g.
        """
    
        logging.debug(
            "PresetTareWeight called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.PresetTareWeight(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def ClearTareValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.ClearTareValue_Responses:
        """
        Executes the unobservable command "Clear Tare Value"
            Clear tare weight value from memory.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareWeightValue (Tare Weight Value): The reset tare weight value in g. Is always 0.
        """
    
        logging.debug(
            "ClearTareValue called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.ClearTareValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def TareImmediately(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.TareImmediately_Responses:
        """
        Executes the unobservable command "Tare Immediately"
            Tare the balance immediately and independently of balance stability.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TareWeightValue (Tare Weight Value): Taring performed successfully with the stable or non-stable taring weight value  in g. The new tare value corresponds to the weight change on the balance since the last zero setting.
        """
    
        logging.debug(
            "TareImmediately called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            return self.implementation.TareImmediately(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)

    def Get_StableWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Get_StableWeightValue_Responses:
        """
        Requests the unobservable property Stable Weight Value
            Get the current stable weight value in g.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.StableWeightValue (Stable Weight Value): Get the current stable weight value in g.
        """
    
        logging.debug(
            "Property StableWeightValue requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            return self.implementation.Get_StableWeightValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def Get_ImmediateWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Get_ImmediateWeightValue_Responses:
        """
        Requests the unobservable property ImmediateWeightValue
            Get the immediate (potentially unstable) weight value
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.ImmediateWeightValue (ImmediateWeightValue): Get the immediate (potentially unstable) weight value
        """
    
        logging.debug(
            "Property ImmediateWeightValue requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            return self.implementation.Get_ImmediateWeightValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    def Subscribe_CurrentWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Subscribe_CurrentWeightValue_Responses:
        """
        Requests the observable property Current Weight Value
            Subscribe to the current weight value
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response stream with the following fields:
            request.CurrentWeightValue (Current Weight Value): Subscribe to the current weight value
        """
    
        logging.debug(
            "Property CurrentWeightValue requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            return self.implementation.Subscribe_CurrentWeightValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
    
    
    def Get_TareWeightValue(self, request, context: grpc.ServicerContext) \
            -> BalanceService_pb2.Get_TareWeightValue_Responses:
        """
        Requests the unobservable property Tare Weight Value
            Get the current tare weight value in g.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TareWeightValue (Tare Weight Value): Get the current tare weight value in g.
        """
    
        logging.debug(
            "Property TareWeightValue requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            return self.implementation.Get_TareWeightValue(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
