"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Service*

:details: DeviceService:
    Allows full control of the stirrer speed and power. Starts and stops the stirrer of the bioREACTOR48.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 14.02.2020

:file:    DeviceService_real.py
:authors: Lukas Bromig

:date: (creation)          2021-05-17T10:13:19.978501
:date: (last modification) 2021-05-17T10:13:19.978501

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceService_pb2 as DeviceService_pb2
# from .gRPC import DeviceService_pb2_grpc as DeviceService_pb2_grpc

# import default arguments
from .DeviceService_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class DeviceServiceReal:
    """
    Implementation of the *Device Service* in *Real* mode
        This is a Presens MCR Service
    """

    def __init__(self, ser, properties):
        """Class initialiser"""
        self.ser = ser
        self.status = "Test-status. Not implemented yet!"
        self.TotalChannels = properties.TotalChannels
        self.TotalBars = properties.TotalBars
        self.BarSensors = properties.BarSensors
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def GetLog(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.CommandConfirmation:
        """
        Executes the observable command "Get Log"
            Get the current status of the device from the state machine of the SiLA server.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A command confirmation object with the following information:
            commandId: A command id with which this observable command can be referenced in future calls
            lifetimeOfExecution: The (maximum) lifetime of this command call.
        """
    
        # initialise default values
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        lifetime_of_execution: silaFW_pb2.Duration = None
    
        # TODO:
        #   Execute the actual command
        #   Optional: Generate a lifetime_of_execution
    
        # respond with UUID and lifetime of execution
        command_uuid = silaFW_pb2.CommandExecutionUUID(value=str(uuid.uuid4()))
        if lifetime_of_execution is not None:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid,
                lifetimeOfExecution=lifetime_of_execution
            )
        else:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid
            )
    
    def GetLog_Info(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.ExecutionInfo:
        """
        Returns execution information regarding the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            commandId: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: An ExecutionInfo response stream for the command with the following fields:
            commandStatus: Status of the command (enumeration)
            progressInfo: Information on the progress of the command (0 to 1)
            estimatedRemainingTime: Estimate of the remaining time required to run the command
            updatedLifetimeOfExecution: An update on the execution lifetime
        """
        # Get the UUID of the command
        command_uuid = request.value
    
        # Get the current state
        execution_info = self._get_command_state(command_uuid=command_uuid)
    
        # construct the initial return dictionary in case while is not executed
        return_values = {'commandStatus': execution_info.commandStatus}
        if execution_info.HasField('progressInfo'):
            return_values['progressInfo'] = execution_info.progressInfo
        if execution_info.HasField('estimatedRemainingTime'):
            return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
        if execution_info.HasField('updatedLifetimeOfExecution'):
            return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
        # we loop only as long as the command is running
        while execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.waiting \
                or execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.running:
            # TODO:
            #   Evaluate the command status --> command_status. Options:
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.running
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedSuccessfully
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedWithError
            #   Optional:
            #       * Determine the progress (progressInfo)
            #       * Determine the estimated remaining time
            #       * Update the Lifetime of execution
    
            # Update all values
            execution_info = self._get_command_state(command_uuid=command_uuid)
    
            # construct the return dictionary
            return_values = {'commandStatus': execution_info.commandStatus}
            if execution_info.HasField('progressInfo'):
                return_values['progressInfo'] = execution_info.progressInfo
            if execution_info.HasField('estimatedRemainingTime'):
                return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
            if execution_info.HasField('updatedLifetimeOfExecution'):
                return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
            # we add a small delay to give the client a chance to keep up.
            time.sleep(0.5)
        else:
            # one last time yield the status
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
    def GetLog_Result(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetLog_Responses:
        """
        Returns the final result of the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentLogLevel (Current Log Level): The current log level of the latest logs , retrieved from the SiLA server log file.
            request.CurrentLogTimestamp (Current Log Timestamp): The current log timestamp of the latest logs , retrieved from the SiLA server log file.
            request.CurrentLogMessage (Current Log Level): The current log level of the latest logs , retrieved from the SiLA server log file.
        """
    
        # initialise the return value
        return_value: DeviceService_pb2.GetLog_Responses = None
    
        # Get the UUID of the command
        command_uuid = request.value
    
        # TODO:
        #   Add implementation of Real for command GetLog here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetLog_Responses(
                **default_dict['GetLog_Responses']
            )
    
        return return_value

    def GetReport(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetReport_Responses:
        """
        Executes the unobservable command "Get Report"
            Get a full device report.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentReport (Current Report): The current report response.
        """
    
        # initialise the return value
        return_value = None

        repo_command = 'repo\r'
        j: int = 0
        try:
            report = ''
            for j in range(0, self.TotalBars, 1):
                self.ser[j].write(str.encode(repo_command))
                tmp = str(bytes.decode(self.ser[j].readline().rstrip()))
                report = report + tmp + ';'
            par_dict = {'CurrentReport': silaFW_pb2.String(value=report)}
            time.sleep(0.2)
            return_value = DeviceService_pb2.GetReport_Responses(**par_dict)
            logging.debug('Executed command GetReport in mode: {mode} with response: {response}'
                          .format(mode='Real', response=report))
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command} with bar: {Bar}'
                              .format(command=repo_command, Bar=j))
            time.sleep(0.2)

        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetReport_Responses(
                **default_dict['GetReport_Responses']
            )
    
        return return_value

    def GetTComp(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetTComp_Responses:
        """
        Executes the unobservable command "Get Temperature Compensation"
            Get the temperature compensation value.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentTComp (Current Temperature Compensation): The current temperature compensation value.
        """
    
        # initialise the return value
        return_value = None
        # return_value = GreetingProvider_pb2.SayHello_Responses(Greeting=silaFW_pb2.String(value=f'"hi : {request.Name.value}"'))
        
    
        # TODO:
        #   Add implementation of Real for command GetTComp here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetTComp_Responses(
                **default_dict['GetTComp_Responses']
            )
    
        return return_value

    def SetTComp(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetTComp_Responses:
        """
        Executes the unobservable command "Set Temperature Compensation"
            Set the temperature compensation value. Values must be between 0-60 degrees Celsius. Default = 20.
    
        :param request: gRPC request containing the parameters passed:
            request.SetTComp (SetTComp):
            The temperature compensation value to be set.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentTComp (Current Temperature Compensation): The current temperature compensation value.
        """
    
        # initialise the return value
        return_value = None
        # return_value = GreetingProvider_pb2.SayHello_Responses(Greeting=silaFW_pb2.String(value=f'"hi : {request.Name.value}"'))
        
    
        # TODO:
        #   Add implementation of Real for command SetTComp here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetTComp_Responses(
                **default_dict['SetTComp_Responses']
            )
    
        return return_value

    def GetDynAveraging(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.GetDynAveraging_Responses:
        """
        Executes the unobservable command "Get Dynamic Averaging"
            Get the dynamic averaging value.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDynAverage (Current Dynamic Average): The current dynamic averaging value.
        """
    
        # initialise the return value
        return_value = None
        # return_value = GreetingProvider_pb2.SayHello_Responses(Greeting=silaFW_pb2.String(value=f'"hi : {request.Name.value}"'))
        
    
        # TODO:
        #   Add implementation of Real for command GetDynAveraging here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.GetDynAveraging_Responses(
                **default_dict['GetDynAveraging_Responses']
            )
    
        return return_value

    def SetDynAveraging(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SetDynAveraging_Responses:
        """
        Executes the unobservable command "Set Dynamic Averaging"
            Set the dynamic averaging value.
    
        :param request: gRPC request containing the parameters passed:
            request.SetDynAverage (Set Dynamic Average):
            The dynamic averaging value to be set. Must be between 0-9. Default = 4.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDynAverage (Current Dynamic Average): The current dynamic average value.
        """
    
        # initialise the return value
        return_value = None
        # return_value = GreetingProvider_pb2.SayHello_Responses(Greeting=silaFW_pb2.String(value=f'"hi : {request.Name.value}"'))
        
    
        # TODO:
        #   Add implementation of Real for command SetDynAveraging here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SetDynAveraging_Responses(
                **default_dict['SetDynAveraging_Responses']
            )
    
        return return_value

    def SwitchOffDevice(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.SwitchOffDevice_Responses:
        """
        Executes the unobservable command "Switch Off Device"
            Switch off the device to save power.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentStatus (Current Status): The current stop status response.
        """
    
        # initialise the return value
        return_value = None
        # return_value = GreetingProvider_pb2.SayHello_Responses(Greeting=silaFW_pb2.String(value=f'"hi : {request.Name.value}"'))
        
    
        # TODO:
        #   Add implementation of Real for command SwitchOffDevice here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceService_pb2.SwitchOffDevice_Responses(
                **default_dict['SwitchOffDevice_Responses']
            )
    
        return return_value
    

    def Subscribe_CurrentStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceService_pb2.Subscribe_CurrentStatus_Responses:
        """
        Requests the observable property Current Status
            Get the current status of the device from the internal state machine of the SiLA server.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.CurrentStatus (Current Status): Get the current status of the device from the internal state machine of the SiLA server.
        """
    
        # initialise the return value
        return_value: DeviceService_pb2.Subscribe_CurrentStatus_Responses = None
    
        # we could use a timeout here if we wanted
        while True:
            par_dict = {
                'CurrentStatus': silaFW_pb2.String(value=self.status)
            }
            return_value = DeviceService_pb2.Subscribe_CurrentStatus_Responses(**par_dict)
            # TODO:
            #   Add implementation of Real for property CurrentStatus here and write the resulting
            #   response in return_value
    
            # create the default value
            if return_value is None:
                return_value = DeviceService_pb2.Subscribe_CurrentStatus_Responses(
                    **default_dict['Subscribe_CurrentStatus_Responses']
                )
    
    
            yield return_value
    
