"""
________________________________________________________________________

:PROJECT: SiLA2_python

*SensorProvider*

:details: SensorProvider:
    Measurement data of the Presense Multichannel Sensor Reader (MCR) can be accessed with this feature.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 14.02.2020

:file:    SensorProvider_real.py
:authors: Lukas Bromig

:date: (creation)          2021-05-17T10:13:20.865242
:date: (last modification) 2021-05-17T10:13:20.865242

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import SensorProvider_pb2 as SensorProvider_pb2
# from .gRPC import SensorProvider_pb2_grpc as SensorProvider_pb2_grpc

# import default arguments
from .SensorProvider_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class SensorProviderReal:
    """
    Implementation of the *SensorProvider* in *Real* mode
        This is a Presens MCR Service
    """

    def __init__(self, ser, properties):
        """Class initializer"""
        self.ser = ser
        self.TotalChannels = properties.TotalChannels
        self.TotalBars = properties.TotalBars
        self.BarSensors = properties.BarSensors
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def GetSingleO2(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetSingleO2_Responses:
        """
        Executes the unobservable command "Get Single O2"
            Get the oxygen measurement of a specified channel.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            The channel to be addressed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentChannelNumber (Current Channel Number): The current addressed channel number
            request.CurrentSensorType (CurrentSensor Type): The current sensor type number
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitude
            request.CurrentPhase (Current Phase): The current signal phase value
            request.CurrentTemperature (Current Temperature): The current temperature value
            request.CurrentO2 (Current O2): The oxygen reading of the specified channel.
            request.CurrentErrorCode (Current Error Code): The current error code of the respective channel. ER0 = No error.
        """
    
        # initialise the return value
        return_value = None

        #bar = int(request.Channel.value/self.BarSensors)
        bar = int((request.Channel.value-1)/self.BarSensors)
        if request.Channel.value % self.BarSensors == 0:
            reactor = 0
        else:
            reactor = self.BarSensors - (request.Channel.value % self.BarSensors)
        
        
        echo = 'echo0000\r'
        switch_channel_command = 'oxch\r'
        get_data_command = 'call000%s\r' % reactor

        try:
            self.ser[bar].write(str.encode(echo))
            read = str(bytes.decode(self.ser[bar].readline().rstrip(), 'ascii'))
            time.sleep(0.25)
            self.ser[bar].write(str.encode(switch_channel_command))
            read = str(bytes.decode(self.ser[bar].readline().rstrip(), 'ascii'))
            time.sleep(0.25)
            self.ser[bar].write(str.encode(get_data_command))
            read = str(bytes.decode(self.ser[bar].readline().rstrip(), 'ascii'))
            time.sleep(0.25)

            split = read.split(';')
            channel, sig, amp, phase, temp, val, err = [split[i][0:] for i in range(0, len(split) - 1, 1)]
            channel = int(channel[1:])
            if sig == 'S0':
                sig = 'O2'
            if sig == 'S1':
                sig = 'pH'
            amp = int(amp[1:])
            phase = int(phase[1:])/100
            temp = int(temp[1:])/10
            val = int(val[1:])/100
            par_dict = {
                'CurrentChannelNumber': silaFW_pb2.Integer(value=channel),
                'CurrentSensorType': silaFW_pb2.String(value=sig),
                'CurrentAmplitude': silaFW_pb2.Integer(value=amp),
                'CurrentPhase': silaFW_pb2.Real(value=phase),
                'CurrentTemperature': silaFW_pb2.Real(value=temp),
                'CurrentO2': silaFW_pb2.Real(value=val),
                'CurrentErrorCode': silaFW_pb2.String(value=err)
            }
            return_value = SensorProvider_pb2.GetSingleO2_Responses(**par_dict)
            logging.debug('Executed command GetDeviceStatus in mode: {mode} with response: {response}'
                          .format(mode='Real', response=read))
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=get_data_command))

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetSingleO2_Responses(
                **default_dict['GetSingleO2_Responses']
            )
    
        return return_value

    def GetSinglePH(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetSinglePH_Responses:
        """
        Executes the unobservable command "Get Single PH"
            Get the PH measurement of a specified channel.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            The channel to be addressed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentChannelNumber (Current Channel Number): The current addressed channel number
            request.CurrentSensorType (CurrentSensor Type): The current sensor type number
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitude
            request.CurrentPhase (Current Phase): The current signal phase value
            request.CurrentTemperature (Current Temperature): The current temperature value
            request.CurrentPH (Current PH): The pH reading of the specified channel.
            request.CurrentErrorCode (Current Error Code): The current error code of the respective channel. ER0 = No error.
        """
    
        # initialise the return value
        return_value = None

        #bar = int(request.Channel.value/self.BarSensors)
        bar = int((request.Channel.value-1)/self.BarSensors)
        
        if request.Channel.value % self.BarSensors == 0:
            reactor = 0
        else:
            reactor = self.BarSensors - (request.Channel.value % self.BarSensors)

        echo = 'echo0000\r'
        switch_channel_command = "phch\r"
        get_data_command = 'call000%s\r' % reactor

        try:
            self.ser[bar].write(str.encode(echo))
            read = str(bytes.decode(self.ser[bar].readline().rstrip(), 'ascii'))
            time.sleep(0.25)
            self.ser[bar].write(str.encode(switch_channel_command))
            read = str(bytes.decode(self.ser[bar].readline().rstrip(), 'ascii'))
            time.sleep(0.25)
            self.ser[bar].write(str.encode(get_data_command))
            read = str(bytes.decode(self.ser[bar].readline().rstrip(), 'ascii'))
            time.sleep(0.25)

            split = read.split(';')
            channel, sig, amp, phase, temp, val, err = [split[i][0:] for i in range(0, len(split) - 1, 1)]
            channel = int(channel[1:])
            
            if sig == 'S0':
                sig = 'O2'
            if sig == 'S1':
                sig = 'pH'
            amp = int(amp[1:])
            phase = int(phase[1:])/100
            temp = int(temp[1:])/10
            val = int(val[1:])/100
            par_dict = {
                'CurrentChannelNumber': silaFW_pb2.Integer(value=channel),
                'CurrentSensorType': silaFW_pb2.String(value=sig),
                'CurrentAmplitude': silaFW_pb2.Integer(value=amp),
                'CurrentPhase': silaFW_pb2.Real(value=phase),
                'CurrentTemperature': silaFW_pb2.Real(value=temp),
                'CurrentPH': silaFW_pb2.Real(value=val),
                'CurrentErrorCode': silaFW_pb2.String(value=err)
            }
            return_value = SensorProvider_pb2.GetSinglePH_Responses(**par_dict)
            logging.debug('Executed command GetDeviceStatus in mode: {mode} with response: {response}'
                          .format(mode='Real', response=read))
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=get_data_command))

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetSinglePH_Responses(
                **default_dict['GetSinglePH_Responses']
            )
    
        return return_value

    def GetAllO2(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetAllO2_Responses:
        """
        Executes the unobservable command "Get All O2"
            Get oxygen measurements of all connected channels.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (Current Sensor Type): The current sensor type numbers of all channels.
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitudes of all channel.
            request.CurrentPhase (Current Phase): The current signal phase values of all channels
            request.CurrentTemperature (Current Temperature): The current temperature values of all channels.
            request.CurrentO2 (Current O2): The oxygen readings of all channels.
            request.CurrentErrorCode (Current Error Code): The current error codes of all channels. ER0 = No error.
        """
    
        # initialise the return value
        return_value = None
        command = 'callXXXX\r'
        echo = 'echo0000\r'
        switch_channel_command = "oxch\r"

        par_dict = {
            # 'CurrentChannelNumber': [],
            'CurrentSensorType': [silaFW_pb2.String(value='default')] * self.TotalChannels,
            'CurrentAmplitude': [silaFW_pb2.Integer(value=0)] * self.TotalChannels,
            'CurrentPhase': [silaFW_pb2.Real(value=0.0)] * self.TotalChannels,
            'CurrentTemperature': [silaFW_pb2.Real(value=0.0)] * self.TotalChannels,
            'CurrentO2': [silaFW_pb2.Real(value=0.0)] * self.TotalChannels,
            'CurrentErrorCode': [silaFW_pb2.String(value="9999")] * self.TotalChannels
        }
        read: str = ''
        try:
            for i in range(0, self.TotalBars, 1):
                self.ser[i].write(str.encode(echo))
                read = str(bytes.decode(self.ser[i].readline().rstrip()))
            time.sleep(0.25)
            for i in range(0, self.TotalBars, 1):
                self.ser[i].write(str.encode(switch_channel_command))
                read = str(bytes.decode(self.ser[i].readline().rstrip()))
            time.sleep(0.25)

            reverse_bar_sensors = list(range(0, self.BarSensors, 1))
            reverse_bar_sensors.reverse()
            for i, n in enumerate(reverse_bar_sensors):
                for j in range(0, self.TotalBars, 1):
                    current_position = (j * self.BarSensors) + n
                    command = 'call000%s\r' % (i)
                    self.ser[j].write(str.encode(command))
                    read = str(bytes.decode(self.ser[j].readline().rstrip()))
                    split = read.split(';')
                    channel, sig, amp, phase, temp, val, err = [split[i][1:] for i in range(0, len(split) - 1, 1)]
                    channel = channel[:1]
                    if sig == '0':
                        sig = 'O2'
                    if sig == '1':
                        sig = 'pH'
                    amp = int(amp[0:])
                    phase = int(phase) / 100
                    temp = int(temp) / 10
                    val = int(val) / 100
                    # par_dict['CurrentChannelNumber'].append(silaFW_pb2.Integer(value=channel))
                    par_dict['CurrentSensorType'][current_position] = silaFW_pb2.String(value=sig)
                    par_dict['CurrentAmplitude'][current_position] = silaFW_pb2.Integer(value=amp)
                    par_dict['CurrentPhase'][current_position] = silaFW_pb2.Real(value=phase)
                    par_dict['CurrentTemperature'][current_position] = silaFW_pb2.Real(value=temp)
                    par_dict['CurrentO2'][current_position] = silaFW_pb2.Real(value=val)
                    par_dict['CurrentErrorCode'][current_position] = silaFW_pb2.String(value=err)
                time.sleep(0.25)
            return_value = SensorProvider_pb2.GetAllO2_Responses(**par_dict)
            logging.debug('Executed command GetAllO2 in mode: {mode} with response: {response}'
                          .format(mode='Real', response=read))
            time.sleep(0.25)
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=command))
            time.sleep(0.25)

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetAllO2_Responses(
                **default_dict['GetAllO2_Responses']
            )
    
        return return_value

    def GetAllPH(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetAllPH_Responses:
        """
        Executes the unobservable command "Get All PH"
            Get pH measurements of all connected channels.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (Current Sensor Type): The current sensor type numbers of all channels.
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitudes of all channel.
            request.CurrentPhase (Current Phase): The current signal phase values of all channels
            request.CurrentTemperature (Current Temperature): The current temperature values of all channels.
            request.CurrentPH (Current PH): The pH readings of all channels.
            request.CurrentErrorCode (Current Error Code): The current error codes of all channels. ER0 = No error.
        """
    
        # initialise the return value
        return_value = None

        command = 'callXXXX\r'
        echo = 'echo0000\r'
        switch_channel_command = "phch\r"

        par_dict = {
            # 'CurrentChannelNumber': [],
            'CurrentSensorType': [silaFW_pb2.String(value='default')] * self.TotalChannels,
            'CurrentAmplitude': [silaFW_pb2.Integer(value=0)] * self.TotalChannels,
            'CurrentPhase': [silaFW_pb2.Real(value=0.0)] * self.TotalChannels,
            'CurrentTemperature': [silaFW_pb2.Real(value=0.0)] * self.TotalChannels,
            'CurrentPH': [silaFW_pb2.Real(value=0.0)] * self.TotalChannels,
            'CurrentErrorCode': [silaFW_pb2.String(value="9999")] * self.TotalChannels
        }
        read: str = ''
        try:
            for i in range(0, self.TotalBars, 1):
                self.ser[i].write(str.encode(echo))
                read = str(bytes.decode(self.ser[i].readline().rstrip()))
            time.sleep(0.25)
            for i in range(0, self.TotalBars, 1):
                self.ser[i].write(str.encode(switch_channel_command))
                read = str(bytes.decode(self.ser[i].readline().rstrip()))
            time.sleep(0.25)

            reverse_bar_sensors = list(range(0, self.BarSensors, 1))
            reverse_bar_sensors.reverse()
            for i, n in enumerate(reverse_bar_sensors):
                # for i in range(0,self.BarSensors,1):
                for j in range(0, self.TotalBars, 1):
                    current_position = (j * self.BarSensors) + n
                    command = 'call000%s\r' % i
                    self.ser[j].write(str.encode(command))
                    read = str(bytes.decode(self.ser[j].readline().rstrip()))
                    split = read.split(';')
                    channel, sig, amp, phase, temp, val, err = [split[i][1:] for i in range(0, len(split) - 1, 1)]
                    channel = channel[:1]
                    if sig == '0':
                        sig = 'O2'
                    if sig == '1':
                        sig = 'pH'
                    amp = int(amp[0:])
                    phase = int(phase) / 100
                    temp = int(temp) / 10
                    val = int(val) / 100
                    # par_dict['CurrentChannelNumber'][current_position] = silaFW_pb2.Integer(value=channel)
                    par_dict['CurrentSensorType'][current_position] = silaFW_pb2.String(value=sig)
                    par_dict['CurrentAmplitude'][current_position] = silaFW_pb2.Integer(value=amp)
                    par_dict['CurrentPhase'][current_position] = silaFW_pb2.Real(value=phase)
                    par_dict['CurrentTemperature'][current_position] = silaFW_pb2.Real(value=temp)
                    par_dict['CurrentPH'][current_position] = silaFW_pb2.Real(value=val)
                    par_dict['CurrentErrorCode'][current_position] = silaFW_pb2.String(value=err)
            time.sleep(0.25)
            return_value = SensorProvider_pb2.GetAllPH_Responses(**par_dict)
            logging.debug('Executed command GetAllPH in mode: {mode} with response: {response}'
                          .format(mode='Real', response=read))
            time.sleep(0.25)
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=command))
            time.sleep(0.25)
    
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetAllPH_Responses(
                **default_dict['GetAllPH_Responses']
            )
    
        return return_value

    def Get_TotalChannels(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_TotalChannels_Responses:
        """
        Requests the unobservable property Total Channels
            Total number of channels. Default = 48.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalChannels (Total Channels): Total number of channels. Default = 48.
        """
    
        # initialise the return value
        return_value: SensorProvider_pb2.Get_TotalChannels_Responses = self.TotalChannels
        par_dict = {
            'TotalChannels': silaFW_pb2.Integer(value=self.TotalChannels)
        }
        return_value = SensorProvider_pb2.Get_TotalChannels_Responses(**par_dict)
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.Get_TotalChannels_Responses(
                **default_dict['Get_TotalChannels_Responses']
            )
    
        return return_value
    
    def Get_TotalBars(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_TotalBars_Responses:
        """
        Requests the unobservable property TotalBars
            Total number of measurement bars. Default = 6.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalBars (TotalBars): Total number of measurement bars. Default = 6.
        """
    
        # initialise the return value
        return_value: SensorProvider_pb2.Get_TotalBars_Responses = self.TotalBars
        par_dict = {
            'TotalBars': silaFW_pb2.Integer(value=self.TotalBars)
        }
        return_value = SensorProvider_pb2.Get_TotalBars_Responses(**par_dict)
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.Get_TotalBars_Responses(
                **default_dict['Get_TotalBars_Responses']
            )
    
        return return_value
    
    def Get_BarSensors(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_BarSensors_Responses:
        """
        Requests the unobservable property Bar Sensors
            Number of sensors per bar. Default = 8.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.BarSensors (Bar Sensors): Number of sensors per bar. Default = 8.
        """
    
        # initialise the return value
        return_value: SensorProvider_pb2.Get_BarSensors_Responses = self.BarSensors
        par_dict = {
            'BarSensors': silaFW_pb2.Integer(value=self.BarSensors)
        }
        return_value = SensorProvider_pb2.Get_BarSensors_Responses(**par_dict)
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.Get_BarSensors_Responses(
                **default_dict['Get_BarSensors_Responses']
            )
    
        return return_value
