"""
________________________________________________________________________

:PROJECT: SiLA2_python

*SensorProvider*

:details: SensorProvider:
    Measurement data of the Presense Multichannel Sensor Reader (MCR) can be accessed with this feature.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 14.02.2020
           
:file:    SensorProvider_servicer.py
:authors: Lukas Bromig

:date: (creation)          2021-05-17T10:13:20.816238
:date: (last modification) 2021-05-17T10:13:20.816238

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import grpc

# meta packages
from typing import Union

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2
from sila2lib.error_handling.server_err import SiLAError

# import gRPC modules for this feature
from .gRPC import SensorProvider_pb2 as SensorProvider_pb2
from .gRPC import SensorProvider_pb2_grpc as SensorProvider_pb2_grpc

# import simulation and real implementation
from .SensorProvider_simulation import SensorProviderSimulation
from .SensorProvider_real import SensorProviderReal


class SensorProvider(SensorProvider_pb2_grpc.SensorProviderServicer):
    """
    This is a Presens MCR Service
    """
    implementation: Union[SensorProviderSimulation, SensorProviderReal]
    simulation_mode: bool

    def __init__(self, ser, properties, simulation_mode: bool = True):
        """
        Class initialiser.

        :param simulation_mode: Sets whether at initialisation the simulation mode is active or the real mode.
        """
        self.properties = properties
        self.lock =properties.lock
        self.ser = ser
        self.simulation_mode = simulation_mode
        if simulation_mode:
            self._inject_implementation(SensorProviderSimulation(self.properties))
        else:
            self._inject_implementation(SensorProviderReal(self.ser, self.properties))

    def _inject_implementation(self,
                               implementation: Union[SensorProviderSimulation,
                                                     SensorProviderReal]
                               ) -> bool:
        """
        Dependency injection of the implementation used.
            Allows to set the class used for simulation/real mode.

        :param implementation: A valid implementation of the PresensServiceServicer.
        """

        self.implementation = implementation
        return True

    def switch_to_simulation_mode(self):
        """Method that will automatically be called by the server when the simulation mode is requested."""
        self.simulation_mode = True
        self._inject_implementation(SensorProviderSimulation(self.properties))

    def switch_to_real_mode(self):
        """Method that will automatically be called by the server when the real mode is requested."""
        self.simulation_mode = False
        self._inject_implementation(SensorProviderReal(self.ser, self.properties))

    def GetSingleO2(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetSingleO2_Responses:
        """
        Executes the unobservable command "Get Single O2"
            Get the oxygen measurement of a specified channel.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            The channel to be addressed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentChannelNumber (Current Channel Number): The current addressed channel number
            request.CurrentSensorType (CurrentSensor Type): The current sensor type number
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitude
            request.CurrentPhase (Current Phase): The current signal phase value
            request.CurrentTemperature (Current Temperature): The current temperature value
            request.CurrentO2 (Current O2): The oxygen reading of the specified channel.
            request.CurrentErrorCode (Current Error Code): The current error code of the respective channel. ER0 = No error.
        """
    
        logging.debug(
            "GetSingleO2 called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            self.lock.acquire()
            return self.implementation.GetSingleO2(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()

    def GetSinglePH(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetSinglePH_Responses:
        """
        Executes the unobservable command "Get Single PH"
            Get the PH measurement of a specified channel.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            The channel to be addressed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentChannelNumber (Current Channel Number): The current addressed channel number
            request.CurrentSensorType (CurrentSensor Type): The current sensor type number
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitude
            request.CurrentPhase (Current Phase): The current signal phase value
            request.CurrentTemperature (Current Temperature): The current temperature value
            request.CurrentPH (Current PH): The pH reading of the specified channel.
            request.CurrentErrorCode (Current Error Code): The current error code of the respective channel. ER0 = No error.
        """
    
        logging.debug(
            "GetSinglePH called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            self.lock.acquire()
            return self.implementation.GetSinglePH(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()

    def GetAllO2(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetAllO2_Responses:
        """
        Executes the unobservable command "Get All O2"
            Get oxygen measurements of all connected channels.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (Current Sensor Type): The current sensor type numbers of all channels.
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitudes of all channel.
            request.CurrentPhase (Current Phase): The current signal phase values of all channels
            request.CurrentTemperature (Current Temperature): The current temperature values of all channels.
            request.CurrentO2 (Current O2): The oxygen readings of all channels.
            request.CurrentErrorCode (Current Error Code): The current error codes of all channels. ER0 = No error.
        """
    
        logging.debug(
            "GetAllO2 called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            self.lock.acquire()
            return self.implementation.GetAllO2(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()

    def GetAllPH(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetAllPH_Responses:
        """
        Executes the unobservable command "Get All PH"
            Get pH measurements of all connected channels.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (Current Sensor Type): The current sensor type numbers of all channels.
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitudes of all channel.
            request.CurrentPhase (Current Phase): The current signal phase values of all channels
            request.CurrentTemperature (Current Temperature): The current temperature values of all channels.
            request.CurrentPH (Current PH): The pH readings of all channels.
            request.CurrentErrorCode (Current Error Code): The current error codes of all channels. ER0 = No error.
        """
    
        logging.debug(
            "GetAllPH called in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
    
        # parameter validation
        # if request.my_paramameter.value out of scope :
        #        sila_val_err = SiLAValidationError(parameter="myParameter",
        #                                           msg=f"Parameter {request.my_parameter.value} out of scope!")
        #        sila_val_err.raise_rpc_error(context)
    
        try:
            self.lock.acquire()
            return self.implementation.GetAllPH(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()

    def Get_TotalChannels(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_TotalChannels_Responses:
        """
        Requests the unobservable property Total Channels
            Total number of channels. Default = 48.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalChannels (Total Channels): Total number of channels. Default = 48.
        """
    
        logging.debug(
            "Property TotalChannels requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            self.lock.acquire()
            return self.implementation.Get_TotalChannels(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()

    def Get_TotalBars(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_TotalBars_Responses:
        """
        Requests the unobservable property TotalBars
            Total number of measurement bars. Default = 6.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalBars (TotalBars): Total number of measurement bars. Default = 6.
        """
    
        logging.debug(
            "Property TotalBars requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            self.lock.acquire()
            return self.implementation.Get_TotalBars(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()

    def Get_BarSensors(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_BarSensors_Responses:
        """
        Requests the unobservable property Bar Sensors
            Number of sensors per bar. Default = 8.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.BarSensors (Bar Sensors): Number of sensors per bar. Default = 8.
        """
    
        logging.debug(
            "Property BarSensors requested in {current_mode} mode".format(
                current_mode=('simulation' if self.simulation_mode else 'real')
            )
        )
        try:
            self.lock.acquire()
            return self.implementation.Get_BarSensors(request, context)
        except SiLAError as err:
            err.raise_rpc_error(context=context)
        finally:
            self.lock.release()
