"""
________________________________________________________________________

:PROJECT: SiLA2_python

*SensorProvider*

:details: SensorProvider:
    Measurement data of the Presense Multichannel Sensor Reader (MCR) can be accessed with this feature.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 14.02.2020

:file:    SensorProvider_simulation.py
:authors: Lukas Bromig, Nikolas von den Eichen 

:date: (creation)          2021-05-17T10:13:20.842239
:date: (last modification) 2021-05-17T10:13:20.842239

.. note:: Code generated by sila2codegenerator 0.3.6

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import grpc         # used for type hinting only
import random       # generate random data for simulation mode
import math
import csv # used for importing historical data
import bisect # used for finding the right datapoints in the historical data
import os

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import SensorProvider_pb2 as SensorProvider_pb2
# from .gRPC import SensorProvider_pb2_grpc as SensorProvider_pb2_grpc

# import default arguments
from .SensorProvider_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class SensorProviderSimulation:
    """
    Implementation of the *SensorProvider* in *Simulation* mode
        This is a Presens MCR Service
    """

    def __init__(self, properties):
        """Class initializer"""
        self.TotalChannels = properties.TotalChannels
        self.TotalBars = properties.TotalBars
        self.BarSensors = properties.BarSensors
        self.ServerStartTime= properties.server_start_time
        self.DriftDirection = []
        self.DriftSpeed = []
        self.DriftAmplitude = []
        drift_max_speed = 10
        drift_max_amplitude = 1
        self.communication_delay = 0 # simulates the delay of the serial communication with the device 
        
        for i in range(self.TotalBars * self.BarSensors):
            curvedirection=random.choice([True, False])
            self.DriftDirection.append(curvedirection)
            self.DriftSpeed.append(random.uniform(0, drift_max_speed))
            self.DriftAmplitude.append(random.uniform(0, drift_max_amplitude))
        logging.debug(f'The drift speeds are {self.DriftSpeed}')
        logging.debug(f'The drift directions are {self.DriftDirection}')
        logging.debug(f'The drift amplitudes are {self.DriftAmplitude}')

        
        self.ph_mode = properties.ph_simulation_mode
        logging.debug(f'GetAllpH is in mode {self.ph_mode}')
        # reading the historical csv data to dict
        if self.ph_mode == 'historical_data':
            
            self.historical_ph_data = {}
            csv_location = 'pHSimulationData.csv'
            csv_location = os.path.join(os.path.dirname(__file__), csv_location)

            with open(csv_location, newline='') as csvfile:
                reader = csv.reader(csvfile, delimiter = ';')
                header = next(reader)
                for row in reader:
                    self.historical_ph_data[float(row[0])] = float(row[1]) 
            logging.debug('Finished reading historical pH data')

            
        
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def GetSingleO2(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetSingleO2_Responses:
        """
        Executes the unobservable command "Get Single O2"
            Get the oxygen measurement of a specified channel.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            The channel to be addressed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentChannelNumber (Current Channel Number): The current addressed channel number
            request.CurrentSensorType (CurrentSensor Type): The current sensor type number
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitude
            request.CurrentPhase (Current Phase): The current signal phase value
            request.CurrentTemperature (Current Temperature): The current temperature value
            request.CurrentO2 (Current O2): The oxygen reading of the specified channel.
            request.CurrentErrorCode (Current Error Code): The current error code of the respective channel. ER0 = No error.
        """
    
        # initialise the return value
        return_value = None

        bar = int((request.Channel.value-1)/self.BarSensors)
        if request.Channel.value % self.BarSensors == 0:
            reactor = 0
        else:
            reactor = self.BarSensors - (request.Channel.value % self.BarSensors)

        switch_channel_command = 'oxch\r'
        get_data_command = 'call000%s\r' % reactor

        try:
            read = "C6;S0;A38;P-1960;T370;V2100;E4;"
            time.sleep(0.2)
            split = read.split(';')
            channel, sig, amp, phase, temp, val, err = [split[i][0:] for i in range(0, len(split) - 1, 1)]
            if sig == 'S0':
                sig = 'O2'
            if sig == 'S1':
                sig = 'pH'
            amp = int(amp[1:])
            phase = int(phase[1:])/100
            temp = int(temp[1:])/10
            val = int(val[1:])/100
            par_dict = {
                'CurrentChannelNumber': silaFW_pb2.Integer(value=reactor),
                'CurrentSensorType': silaFW_pb2.String(value=sig),
                'CurrentAmplitude': silaFW_pb2.Integer(value=amp),
                'CurrentPhase': silaFW_pb2.Real(value=phase),
                'CurrentTemperature': silaFW_pb2.Real(value=temp),
                'CurrentO2': silaFW_pb2.Real(value=val),
                'CurrentErrorCode': silaFW_pb2.String(value=err)
            }
            return_value = SensorProvider_pb2.GetSingleO2_Responses(**par_dict)
            logging.debug('Executed command GetDeviceStatus in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=get_data_command))

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetSingleO2_Responses(
                **default_dict['GetSingleO2_Responses']
            )
    
        return return_value

    def GetSinglePH(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetSinglePH_Responses:
        """
        Executes the unobservable command "Get Single PH"
            Get the PH measurement of a specified channel.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            The channel to be addressed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentChannelNumber (Current Channel Number): The current addressed channel number
            request.CurrentSensorType (CurrentSensor Type): The current sensor type number
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitude
            request.CurrentPhase (Current Phase): The current signal phase value
            request.CurrentTemperature (Current Temperature): The current temperature value
            request.CurrentPH (Current PH): The pH reading of the specified channel.
            request.CurrentErrorCode (Current Error Code): The current error code of the respective channel. ER0 = No error.
        """
    
        # initialise the return value
        return_value = None

        bar = int((request.Channel.value-1)/self.BarSensors)
        if request.Channel.value == self.BarSensors:
            reactor = 0
        else:
            reactor = self.BarSensors - (request.Channel.value % self.BarSensors)

        switch_channel_command = "phch\r"
        get_data_command = 'call000%s\r' % reactor

        try:
            read = "C6;S1;A38;P-1960;T370;V600;E4;"
            time.sleep(0.2)

            split = read.split(';')
            channel, sig, amp, phase, temp, val, err = [split[i][0:] for i in range(0, len(split) - 1, 1)]
            channel = int(channel[1:])
            if sig == 'S0':
                sig = 'O2'
            if sig == 'S1':
                sig = 'pH'
            amp = int(amp[1:])
            phase = int(phase[1:])/100
            temp = int(temp[1:])/10
            val = int(val[1:])/100
            par_dict = {
                'CurrentChannelNumber': silaFW_pb2.Integer(value=reactor),
                'CurrentSensorType': silaFW_pb2.String(value=sig),
                'CurrentAmplitude': silaFW_pb2.Integer(value=amp),
                'CurrentPhase': silaFW_pb2.Real(value=phase),
                'CurrentTemperature': silaFW_pb2.Real(value=temp),
                'CurrentPH': silaFW_pb2.Real(value=val),
                'CurrentErrorCode': silaFW_pb2.String(value=err)
            }
            return_value = SensorProvider_pb2.GetSinglePH_Responses(**par_dict)
            logging.debug('Executed command GetDeviceStatus in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except ConnectionError:
            logging.exception('Communication failed executing the command: {command}'.format(command=get_data_command))

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetSinglePH_Responses(
                **default_dict['GetSinglePH_Responses']
            )
    
        return return_value

    def GetAllO2(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetAllO2_Responses:
        """
        Executes the unobservable command "Get All O2"
            Get oxygen measurements of all connected channels.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (Current Sensor Type): The current sensor type numbers of all channels.
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitudes of all channel.
            request.CurrentPhase (Current Phase): The current signal phase values of all channels
            request.CurrentTemperature (Current Temperature): The current temperature values of all channels.
            request.CurrentO2 (Current O2): The oxygen readings of all channels.
            request.CurrentErrorCode (Current Error Code): The current error codes of all channels. ER0 = No error.
        """


        def randomize_data(positions):
            logging.debug(f'The Number of positions is : {positions}')
            initial_drift_value = random.uniform(0, 120)# typical measurement range of PreSens MCR spots for O2
            logging.debug(f'The initial drift value is : {initial_drift_value}')
            value_list = []
            for i in range(positions):
                curr_value = initial_drift_value + random.uniform(-15, 15)
                curr_value = round(curr_value, 2)
                value_list.append(curr_value)
            logging.debug(value_list)
            return value_list


        random_o2_data = randomize_data((self.BarSensors * self.TotalBars))
        # initialise the return value
        return_value = None

        switch_channel_command = "oxch\r"
        

        par_dict = {
            #'CurrentChannelNumber': [],
            'CurrentSensorType': [],
            'CurrentAmplitude': [],
            'CurrentPhase': [],
            'CurrentTemperature': [],
            'CurrentO2': [],
            'CurrentErrorCode': []
        }
        command: str = ''
        read: str = ''
        loopcounter = 0
        try:
            for i in range(0, self.BarSensors, 1):
                for j in range(0, self.TotalBars, 1):
                    command = 'call000%s\r' % i
                    read = "C%s;S0;A38;P-1960;T370;V2100;E4;"% i
                    split = read.split(';')
                    channel, sig, amp, phase, temp, val, err = [split[i][1:] for i in range(0, len(split) - 1, 1)]
                    channel = channel[:1]
                    if sig == '0':
                        sig = 'O2'
                    if sig == '1':
                        sig = 'pH'
                    amp = int(amp[0:])
                    phase = int(phase) / 100
                    temp = int(temp) / 10
                    val = int(val) / 100
                    #par_dict['CurrentChannelNumber'].append(silaFW_pb2.Integer(value=channel))
                    par_dict['CurrentSensorType'].append(silaFW_pb2.String(value=sig))
                    par_dict['CurrentAmplitude'].append(silaFW_pb2.Integer(value=amp))
                    par_dict['CurrentPhase'].append(silaFW_pb2.Real(value=phase))
                    par_dict['CurrentTemperature'].append(silaFW_pb2.Real(value=temp))
                    par_dict['CurrentO2'].append(silaFW_pb2.Real(value=random_o2_data[loopcounter]))
                    par_dict['CurrentErrorCode'].append(silaFW_pb2.String(value=err))
                    loopcounter = loopcounter + 1
                time.sleep(self.communication_delay)
            return_value = SensorProvider_pb2.GetAllO2_Responses(**par_dict)
            logging.debug('Executed command GetAllO2 in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            time.sleep(0.2)

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetAllO2_Responses(
                **default_dict['GetAllO2_Responses']
            )
    
        return return_value

    def GetAllPH(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.GetAllPH_Responses:
        """
        Executes the unobservable command "Get All PH"
            Get pH measurements of all connected channels.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSensorType (Current Sensor Type): The current sensor type numbers of all channels.
            request.CurrentAmplitude (CurrentAmplitude): The current signal amplitudes of all channel.
            request.CurrentPhase (Current Phase): The current signal phase values of all channels
            request.CurrentTemperature (Current Temperature): The current temperature values of all channels.
            request.CurrentPH (Current PH): The pH readings of all channels.
            request.CurrentErrorCode (Current Error Code): The current error codes of all channels. ER0 = No error.
        """



        def historical_data(positions):
            runtime_h = (time.time() - self.ServerStartTime)/ (60*60)
            local_runtime = runtime_h % 10 # historical data only given for 10 hours
            logging.debug(f'The corrected runtime is {local_runtime}')
            drift_noise = 0.02
            # get closest resemblence from dict
            keylist = list(self.historical_ph_data.keys())
            result = bisect.bisect_left(keylist, local_runtime) # search in ordered dict
            closest_time = list(self.historical_ph_data.keys())[result]
            initial_ph_value = self.historical_ph_data[closest_time]
            logging.debug(f'The initial pH value is {initial_ph_value}')

            value_list = []
            for i in range(positions):
                curr_value = initial_ph_value + random.uniform(-1*drift_noise, drift_noise)
                curr_value = round(curr_value, 3)
                value_list.append(curr_value)
            logging.debug(value_list)
            return value_list
                
            

        def randomize_data(positions):
            logging.debug(f'The Number of positions is : {positions}')
            runtime_h = (time.time() - self.ServerStartTime)/ (60*60)
            logging.debug(f'The Runtime in h is: {runtime_h}')
            drift_noise = 0.02
            value_list = []
            for i in range(positions):
                drift_speed = self.DriftSpeed[i]
                drift_amplitude = self.DriftAmplitude[i]
                if self.DriftDirection[i] == False:
                    initial_drift_value = 7 + drift_amplitude * math.sin(runtime_h*drift_speed)
                if self.DriftDirection[i] == True:
                    initial_drift_value = 7 - drift_amplitude * math.sin(runtime_h* drift_speed)
                curr_value = initial_drift_value + random.uniform(-1*drift_noise, drift_noise)
                curr_value = round(curr_value, 3)
                value_list.append(curr_value)
            logging.debug(value_list)
            return value_list

        if self.ph_mode == 'historical_data':
            random_ph_data = historical_data((self.BarSensors * self.TotalBars))
        if self.ph_mode == 'random_oscillations':
            random_ph_data = randomize_data((self.BarSensors * self.TotalBars))
        # initialise the return value
        return_value = None

        switch_channel_command = "phch\r"

        par_dict = {
            #'CurrentChannelNumber': [],
            'CurrentSensorType': [],
            'CurrentAmplitude': [],
            'CurrentPhase': [],
            'CurrentTemperature': [],
            'CurrentPH': [],
            'CurrentErrorCode': []
        }
        read: str = ''
        loopcounter = 0
        try:
            for i in range(0, self.BarSensors, 1):
                for j in range(0, self.TotalBars, 1):
                    command = 'call000%s\r' % (i)
                    read = "C%s;S1;A38;P-1960;T370;V700;E4;" % i
                    split = read.split(';')
                    channel, sig, amp, phase, temp, val, err = [split[i][1:] for i in range(0, len(split) - 1, 1)]
                    channel = channel[:1]
                    if sig == '0':
                        sig = 'O2'
                    if sig == '1':
                        sig = 'pH'
                    amp = int(amp[0:])
                    phase = int(phase) / 100
                    temp = int(temp) / 10
                    val = int(val) / 100
                    #par_dict['CurrentChannelNumber'].append(silaFW_pb2.Integer(value=channel))
                    par_dict['CurrentSensorType'].append(silaFW_pb2.String(value=sig))
                    par_dict['CurrentAmplitude'].append(silaFW_pb2.Integer(value=amp))
                    par_dict['CurrentPhase'].append(silaFW_pb2.Real(value=phase))
                    par_dict['CurrentTemperature'].append(silaFW_pb2.Real(value=temp))
                    par_dict['CurrentPH'].append(silaFW_pb2.Real(value=random_ph_data[loopcounter]))
                    par_dict['CurrentErrorCode'].append(silaFW_pb2.String(value=err))
                    loopcounter = loopcounter + 1
                time.sleep(self.communication_delay)
            return_value = SensorProvider_pb2.GetAllPH_Responses(**par_dict)
            logging.debug('Executed command GetAllPH in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
            time.sleep(self.communication_delay)
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            time.sleep(self.communication_delay)
    
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.GetAllPH_Responses(
                **default_dict['GetAllPH_Responses']
            )
    
        return return_value

    def Get_TotalChannels(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_TotalChannels_Responses:
        """
        Requests the unobservable property Total Channels
            Total number of channels. Default = 48.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalChannels (Total Channels): Total number of channels. Default = 48.
        """
    
        # initialise the return value
        return_value: SensorProvider_pb2.Get_TotalChannels_Responses = self.TotalChannels
    
        par_dict = {
            'TotalChannels': silaFW_pb2.Integer(value=self.TotalChannels)
        }
        return_value = SensorProvider_pb2.Get_TotalChannels_Responses(**par_dict)
    
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.Get_TotalChannels_Responses(
                **default_dict['Get_TotalChannels_Responses']
            )
    
        return return_value
    
    def Get_TotalBars(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_TotalBars_Responses:
        """
        Requests the unobservable property TotalBars
            Total number of measurement bars. Default = 6.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.TotalBars (TotalBars): Total number of measurement bars. Default = 6.
        """
    
        # initialise the return value
        return_value: SensorProvider_pb2.Get_TotalBars_Responses = self.TotalBars
    
        par_dict = {
            'TotalBars': silaFW_pb2.Integer(value=self.TotalBars)
        }
        return_value = SensorProvider_pb2.Get_TotalBars_Responses(**par_dict)

        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.Get_TotalBars_Responses(
                **default_dict['Get_TotalBars_Responses']
            )
    
        return return_value
    
    def Get_BarSensors(self, request, context: grpc.ServicerContext) \
            -> SensorProvider_pb2.Get_BarSensors_Responses:
        """
        Requests the unobservable property Bar Sensors
            Number of sensors per bar. Default = 8.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.BarSensors (Bar Sensors): Number of sensors per bar. Default = 8.
        """
    
        # initialise the return value
        return_value: SensorProvider_pb2.Get_BarSensors_Responses = int(self.BarSensors)
        par_dict = {
            'BarSensors': silaFW_pb2.Integer(value=self.BarSensors)
        }
        return_value = SensorProvider_pb2.Get_BarSensors_Responses(**par_dict)
    
        # fallback to default
        if return_value is None:
            return_value = SensorProvider_pb2.Get_BarSensors_Responses(
                **default_dict['Get_BarSensors_Responses']
            )
    
        return return_value
