"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Calibration Servicer*

:details: CalibrationServicer:
    Calibrate the sensor bars by adjusting the temperature compensation and the dynamic averaging value.
    By Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 14.02.2020

:file:    CalibrationServicer_simulation.py
:authors: Lukas Bromig

:date: (creation)          2020-04-16T10:19:35.447175
:date: (last modification) 2020-04-16T10:19:35.447175

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import CalibrationServicer_pb2 as CalibrationServicer_pb2
# from .gRPC import CalibrationServicer_pb2_grpc as CalibrationServicer_pb2_grpc

# import default arguments
from .CalibrationServicer_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class CalibrationServicerSimulation:
    """
    Implementation of the *Calibration Servicer* in *Simulation* mode
        This is a Presens MCR Service
    """

    def __init__(self, properties):
        """Class initializer"""
        self.TotalChannels = properties.TotalChannels
        self.TotalBars = properties.TotalBars
        self.BarSensors = properties.BarSensors
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def GetO2CalLow(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetO2CalLow_Responses:
        """
        Executes the unobservable command "Get O2 Calibration Low"
            Get the O2 calibration point value at 0% dissolved oxygen.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentO2CalLow (Current O2 Calibration Low): The current value of the lower (0% dissolved oxygen)
            calibration point. Default = 57.
        """
    
        # initialise the return value
        return_value = None

        command = 'cmoa\r'
        par_dict = {
            'CurrentO2CalLow': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '5400'
                CurrentO2CalLow = float(value) / 100
                par_dict['CurrentO2CalLow'].append(silaFW_pb2.Real(value=CurrentO2CalLow))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetO2CalLow_Responses(**par_dict)
            logging.debug('Executed command GetReport in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentO2CalLow']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetO2CalLow_Responses(
                **default_dict['GetO2CalLow_Responses']
            )
    
        return return_value

    def SetO2CalLow(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetO2CalLow_Responses:
        """
        Executes the unobservable command "Set O2 Calibration Low"
            Set the O2 calibration point value at 0% dissolved oxygen. Default = 57.
    
        :param request: gRPC request containing the parameters passed:
            request.SetO2CalLow (Setpoint O2 Calibration Low):
            The lower calibration point value setpoint for O2 up to two decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentO2CalLow (Current O2 Calibration Low): The current lower calibration point value for O2.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentO2CalLow': []
        }
        command = 'cmoaXXXX\r'
        if len(request.SetO2CalLow) == self.TotalBars:

            for i in range(0, len(request.SetO2CalLow), 1):
                value = str(int(float(request.SetO2CalLow[i].value) * 100))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cmoa%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentO2CalLow'].append(silaFW_pb2.Real(value=request.SetO2CalLow[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentO2CalLow'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentO2CalLow'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetO2CalLow_Responses(**par_dict)
            logging.debug('Executed command SetO2CalLow in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentO2CalLow']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetO2CalLow), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetO2CalLow_Responses(
                **default_dict['SetO2CalLow_Responses']
            )
    
        return return_value

    def GetO2CalHigh(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetO2CalHigh_Responses:
        """
        Executes the unobservable command "Get O2 Calibration High"
            Get the O2 calibration point value at 100% dissolved oxygen. Default = 27.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentO2CalHigh (Current O2 Calibration High): The current value of the higher (100% dissolved
            oxygen) calibration point in %.
        """
    
        # initialise the return value
        return_value = None
    
        command = 'cmob\r'
        par_dict = {
            'CurrentO2CalHigh': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '2600'
                CurrentO2CalHigh = float(value) / 100
                par_dict['CurrentO2CalHigh'].append(silaFW_pb2.Real(value=CurrentO2CalHigh))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetO2CalHigh_Responses(**par_dict)
            logging.debug('Executed command GetReport in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentO2CalHigh']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetO2CalHigh_Responses(
                **default_dict['GetO2CalHigh_Responses']
            )
    
        return return_value

    def SetO2CalHigh(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetO2CalHigh_Responses:
        """
        Executes the unobservable command "Set O2 Calibration High"
            Set the O2 calibration point value at 100% dissolved oxygen. Default = 27.
    
        :param request: gRPC request containing the parameters passed:
            request.SetO2CalHigh (Setpoint O2 Calibration High in percent):
            The higher calibration point value setpoint for O2 up to two decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentO2CalHigh (Current O2 Calibration High): The current higher calibration point value for O2
            in %.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentO2CalHigh': []
        }
        command = 'cmobXXXX\r'
        if len(request.SetO2CalHigh) == self.TotalBars:

            for i in range(0, len(request.SetO2CalHigh), 1):
                value = str(int(float(request.SetO2CalHigh[i].value) * 100))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cmob%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentO2CalHigh'].append(silaFW_pb2.Real(value=request.SetO2CalHigh[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentO2CalHigh'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentO2CalHigh'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetO2CalHigh_Responses(**par_dict)
            logging.debug('Executed command SetO2CalHigh in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentO2CalHigh']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetO2CalHigh), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetO2CalHigh_Responses(
                **default_dict['SetO2CalHigh_Responses']
            )
    
        return return_value

    def GetO2CalTemp(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetO2CalTemp_Responses:
        """
        Executes the unobservable command "Get O2 Calibration Temperature"
            Get the value of the oxygen sensor calibration temperature in degrees Celsius. Default = 20.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentO2CalTemp (Current O2 Calibration Temperature): The current value of the oxygen calibration
            temperature in degree Celsius.
        """
    
        # initialise the return value
        return_value = None

        command = 'coxt\r'
        par_dict = {
            'CurrentO2CalTemp': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '0280'
                CurrentO2CalTemp = float(value) / 10
                par_dict['CurrentO2CalTemp'].append(silaFW_pb2.Real(value=CurrentO2CalTemp))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetO2CalTemp_Responses(**par_dict)
            logging.debug('Executed command GetReport in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentO2CalTemp']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetO2CalTemp_Responses(
                **default_dict['GetO2CalTemp_Responses']
            )
    
        return return_value

    def SetO2CalTemp(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetO2CalTemp_Responses:
        """
        Executes the unobservable command "Set O2 Calibration Temperature"
            Set the value of the oxygen calibration temperature in degree Celsius. Default = 20.
    
        :param request: gRPC request containing the parameters passed:
            request.SetO2CalTemp (Setpoint O2 Calibration Temperature):
            The oxygen calibration temperature value to be set up to one decimal point in degree Celsius.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentO2CalTemp (Current O2 Calibration Temperature): The current value of the oxygen sensor
            calibration temperature in degree Celsius.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentO2CalTemp': []
        }
        command = 'coxtXXXX\r'
        if len(request.SetO2CalTemp) == self.TotalBars:

            for i in range(0, len(request.SetO2CalTemp), 1):
                value = str(int(float(request.SetO2CalTemp[i].value) * 10))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'coxt%s\r' % value
                        # self.ser[i].write(str.encode(command))
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentO2CalTemp'].append(silaFW_pb2.Real(value=request.SetO2CalTemp[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentO2CalTemp'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentO2CalTemp'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetO2CalTemp_Responses(**par_dict)
            logging.debug('Executed command SetO2CalTemp in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentO2CalTemp']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetO2CalTemp), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetO2CalTemp_Responses(
                **default_dict['SetO2CalTemp_Responses']
            )
    
        return return_value

    def GetPHlmax(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetPHlmax_Responses:
        """
        Executes the unobservable command "Get PH lmax"
            Get the given value of the first calibration point (lmax) of the pH sensor. Default = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHlmax (Current PH lmax): The current value of the first calibration point (lmax) of the
            pH sensor.
        """
    
        # initialise the return value
        return_value = None

        command = 'cmpa\r'
        par_dict = {
            'CurrentPHlmax': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '5776'
                CurrentPHlmax = float(value) / 100
                par_dict['CurrentPHlmax'].append(silaFW_pb2.Real(value=CurrentPHlmax))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetPHlmax_Responses(**par_dict)
            logging.debug('Executed command GetPHlmax in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHlmax']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetPHlmax_Responses(
                **default_dict['GetPHlmax_Responses']
            )
    
        return return_value

    def SetPHlmax(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetPHlmax_Responses:
        """
        Executes the unobservable command "Set PH lmax"
            Set the value of the first calibration point (lmax) of the pH sensor. Default = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.SetPHlmax (Setpoint PH lmax):
            The set value of the first calibration point (lmax) of the pH sensor up to two decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHlmax (Current PH lmax): The current value of the first calibration point (lmax) of the
            pH sensor.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentPHlmax': []
        }
        command = 'cmpaXXXX\r'
        print(request.SetPHlmax)
        print(len(request.SetPHlmax))
        if len(request.SetPHlmax) == self.TotalBars:

            for i in range(0, len(request.SetPHlmax), 1):
                value = str(int(float(request.SetPHlmax[i].value) * 100))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cmpa%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHlmax'].append(silaFW_pb2.Real(value=request.SetPHlmax[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHlmax'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentPHlmax'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetPHlmax_Responses(**par_dict)
            logging.debug('Executed command SetPHlmax in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHlmax']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetPHlmax), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetPHlmax_Responses(
                **default_dict['SetPHlmax_Responses']
            )
    
        return return_value

    def GetPHlmin(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetPHlmin_Responses:
        """
        Executes the unobservable command "Get PH lmin"
            Get the given value of the second calibration point (lmin) of the pH sensor. Default = -.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHlmin (Current PH lmin): The current value of the second calibration point (lmin) of
            the pH sensor.
        """
    
        # initialise the return value
        return_value = None

        command = 'cmpb\r'
        par_dict = {
            'CurrentPHlmin': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '2492'
                CurrentPHlmin = float(value) / 100
                par_dict['CurrentPHlmin'].append(silaFW_pb2.Real(value=CurrentPHlmin))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetPHlmin_Responses(**par_dict)
            logging.debug('Executed command GetPHlmin in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHlmin']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetPHlmin_Responses(
                **default_dict['GetPHlmin_Responses']
            )
    
        return return_value

    def SetPHlmin(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetPHlmin_Responses:
        """
        Executes the unobservable command "Set PH lmin"
            Set the value of the second calibration point (lmin) of the pH sensor. Default = -.
    
        :param request: gRPC request containing the parameters passed:
            request.SetPHlmin (Setpoint PH lmin):
            The set value of the second calibration point (lmin) of the pH sensor up to two decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHlmin (Current PH lmin): The current value of the second calibration point (lmin) of the
            pH sensor.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentPHlmin': []
        }
        command = 'cmpbXXXX\r'
        if len(request.SetPHlmin) == self.TotalBars:

            for i in range(0, len(request.SetPHlmin), 1):
                value = str(int(float(request.SetPHlmin[i].value) * 100))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cmpb%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHlmin'].append(silaFW_pb2.Real(value=request.SetPHlmin[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHlmin'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentPHlmin'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetPHlmin_Responses(**par_dict)
            logging.debug('Executed command SetPHlmin in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHlmin']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetPHlmin), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetPHlmin_Responses(
                **default_dict['SetPHlmin_Responses']
            )
    
        return return_value

    def GetPHpH0(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetPHpH0_Responses:
        """
        Executes the unobservable command "Get PH pH0"
            Get the given value of the third calibration point (pH0) of the pH sensor. Default = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHpH0 (Current PH pH0): The current value of the third calibration point (pH0) of the pH
            sensor.
        """
    
        # initialise the return value
        return_value = None

        command = 'cmpc\r'
        par_dict = {
            'CurrentPHpH0': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '0714'
                CurrentPHpH0 = float(value) / 100
                par_dict['CurrentPHpH0'].append(silaFW_pb2.Real(value=CurrentPHpH0))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetPHpH0_Responses(**par_dict)
            logging.debug('Executed command GetPHpH0 in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHpH0']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetPHpH0_Responses(
                **default_dict['GetPHpH0_Responses']
            )
    
        return return_value

    def SetPHpH0(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetPHpH0_Responses:
        """
        Executes the unobservable command "Set PH pH0"
            Set the value of the third calibration point (pH0) of the pH sensor. Default = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.SetPHpH0 (Setpoint PH pH0):
            The set value of the third calibration point (pH0) of the pH sensor up to two decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHpH0 (Current PH pH0): The current value of the third calibration point (pH0) of the
            pH sensor.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentPHpH0': []
        }
        command = 'cmpcXXXX\r'
        if len(request.SetPHpH0) == self.TotalBars:

            for i in range(0, len(request.SetPHpH0), 1):
                value = str(int(float(request.SetPHpH0[i].value) * 100))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cmpc%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHpH0'].append(silaFW_pb2.Real(value=request.SetPHpH0[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHpH0'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentPHpH0'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetPHpH0_Responses(**par_dict)
            logging.debug('Executed command SetPHpH0 in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHpH0']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetPHpH0), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetPHpH0_Responses(
                **default_dict['SetPHpH0_Responses']
            )
    
        return return_value

    def GetPHdpH(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetPHdpH_Responses:
        """
        Executes the unobservable command "Get PH dpH"
            Get the given value of the fourth calibration point (dpH) of the pH sensor. Default = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHdpH (Current PH dpH): The current value of the fourth calibration point (dpH) of the
            pH sensor.
        """
    
        # initialise the return value
        return_value = None

        command = 'cmpd\r'
        par_dict = {
            'CurrentPHdpH': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '0046'
                CurrentPHdpH = float(value) / 100
                par_dict['CurrentPHdpH'].append(silaFW_pb2.Real(value=CurrentPHdpH))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetPHdpH_Responses(**par_dict)
            logging.debug('Executed command GetPHdpH in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHdpH']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetPHdpH_Responses(
                **default_dict['GetPHdpH_Responses']
            )
    
        return return_value

    def SetPHdpH(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetPHdpH_Responses:
        """
        Executes the unobservable command "Set PH dpH"
            Set the value of the fourth calibration point (dpH) of the pH sensor. Default = 0.
    
        :param request: gRPC request containing the parameters passed:
            request.SetPHdpH (Setpoint PH dpH):
            The set value of the fourth calibration point (dpH) of the pH sensor up to two decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHdpH (Current PH dpH): The current value of the fourth calibration point (dpH) of the
            pH sensor.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentPHdpH': []
        }
        command = 'cmpdXXXX\r'
        if len(request.SetPHdpH) == self.TotalBars:

            for i in range(0, len(request.SetPHdpH), 1):
                value = str(int(float(request.SetPHdpH[i].value) * 100))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cmpd%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHdpH'].append(silaFW_pb2.Real(value=request.SetPHdpH[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHdpH'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentPHdpH'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetPHdpH_Responses(**par_dict)
            logging.debug('Executed command GetReport in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHdpH']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetPHdpH), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetPHdpH_Responses(
                **default_dict['SetPHdpH_Responses']
            )
    
        return return_value

    def GetPHCalTemp(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetPHCalTemp_Responses:
        """
        Executes the unobservable command "Get PH Calibration Temperature"
            Get the value of the pH sensor calibration temperature in degree Celsius. Default = 20.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHCalTemp (Current PH Calibration Temperature): The current value of the pH calibration
             temperature in degree Celsius.
        """
    
        # initialise the return value
        return_value = None

        command = 'cpht\r'
        par_dict = {
            'CurrentPHCalTemp': []
        }
        j: int = 0
        try:
            for j in range(0, self.TotalBars, 1):
                value = '0370'
                CurrentPHCalTemp = float(value) / 10
                par_dict['CurrentPHCalTemp'].append(silaFW_pb2.Real(value=CurrentPHCalTemp))
            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.GetPHCalTemp_Responses(**par_dict)
            logging.debug('Executed command GetPHCalTemp in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHCalTemp']))
        except ConnectionError:
            logging.exception(
                'Communication failed executing the command: {command} with bar: {Bar}'.format(command=command, Bar=j))
            time.sleep(0.2)
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetPHCalTemp_Responses(
                **default_dict['GetPHCalTemp_Responses']
            )
    
        return return_value

    def SetPHCalTemp(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetPHCalTemp_Responses:
        """
        Executes the unobservable command "Set PH Calibration Temperature"
            Set the value of the pH calibration temperature in degree Celsius. Default = 20.
    
        :param request: gRPC request containing the parameters passed:
            request.SetPHCalTemp (Setpoint PH Calibration Temperature):
            The pH calibration temperature value to be set in degree Celsius up to one decimal point.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPHCalTemp (Current PH Calibration Temperature): The current value of the pH sensor
            calibration temperature in degree Celsius.
        """
    
        # initialise the return value
        return_value = None

        par_dict = {
            'CurrentPHCalTemp': []
        }
        command = 'cphtXXXX\r'
        if len(request.SetPHCalTemp) == self.TotalBars:

            for i in range(0, len(request.SetPHCalTemp), 1):
                value = str(int(float(request.SetPHCalTemp[i].value) * 10))
                if len(value) < 4:
                    value = value.zfill(4)
                if len(value) == 4:
                    try:
                        command = 'cpht%s\r' % value
                        logging.info('Writing command: {command} to bar/port: {bar}/{port}'
                                     .format(command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHCalTemp'].append(silaFW_pb2.Real(value=request.SetPHCalTemp[i].value))
                    except ConnectionError:
                        logging.exception(
                            'Communication failed executing the command: {command} with bar/port: {bar}/{port}'.format(
                                command=command, bar=i, port="COM%s" % i))
                        par_dict['CurrentPHCalTemp'].append(silaFW_pb2.Real(value=99.99))
                        time.sleep(0.2)
                else:
                    logging.error(
                        'Command length inadequate for serial communication protocol. Command {command} must be of '
                        'length 8, but is of length {length}'.format(command=command, length=(4 + len(value))))
                    par_dict['CurrentPHCalTemp'].append(silaFW_pb2.Real(value=99.99))

            time.sleep(0.2)
            return_value = CalibrationServicer_pb2.SetPHCalTemp_Responses(**par_dict)
            logging.debug('Executed command SetPHCalTemp in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=par_dict['CurrentPHCalTemp']))
        else:
            logging.error(
                'Dimensions do not match. Input dimension is: ({len_request}). Dimension required (TotalChannels) is: '
                '({num_channels})'.format(len_request=len(request.SetPHCalTemp), num_channels=self.TotalChannels))
    
        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetPHCalTemp_Responses(
                **default_dict['SetPHCalTemp_Responses']
            )
    
        return return_value
