"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Device Servicer*

:details: DeviceServicer:
    General device software and hardware information can be retrieved and changed within this function (Reglo DC
    (digital control) pump).
    Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 15.03.2021

:file:    DeviceServicer_simulation.py
:authors: Lukas Bromig

:date: (creation)          2021-03-16T12:40:00.317212
:date: (last modification) 2021-03-16T12:40:00.317212

.. note:: Code generated by sila2codegenerator 0.3.3-dev

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only
import inspect      # used for status determination
import serial

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import DeviceServicer_pb2 as DeviceServicer_pb2
# from .gRPC import DeviceServicer_pb2_grpc as DeviceServicer_pb2_grpc

# import default arguments
from .DeviceServicer_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class DeviceServicerSimulation:
    """
    Implementation of the *Device Servicer* in *Simulation* mode
        This is a RegloDC Service
    """

    def __init__(self, properties):
        """Class initialiser"""

        self.properties = properties
        logging.debug('Started server in mode: {mode}'.format(mode='Simulation'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    @staticmethod
    def _get_function_name():
        return inspect.stack()[1][3]

    def GetLog(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.CommandConfirmation:
        """
        Executes the observable command "Get Log"
            Get the current status of the device from the state machine of the SiLA server.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A command confirmation object with the following information:
            commandId: A command id with which this observable command can be referenced in future calls
            lifetimeOfExecution: The (maximum) lifetime of this command call.
        """
    
        # initialise default values
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        lifetime_of_execution: silaFW_pb2.Duration = None
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        # TODO:
        #   Execute the actual command
        #   Optional: Generate a lifetime_of_execution
    
        # respond with UUID and lifetime of execution
        command_uuid = silaFW_pb2.CommandExecutionUUID(value=str(uuid.uuid4()))
        if lifetime_of_execution is not None:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid,
                lifetimeOfExecution=lifetime_of_execution
            )
        else:
            return silaFW_pb2.CommandConfirmation(
                commandExecutionUUID=command_uuid
            )
    
    def GetLog_Info(self, request, context: grpc.ServicerContext) \
            -> silaFW_pb2.ExecutionInfo:
        """
        Returns execution information regarding the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            commandId: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: An ExecutionInfo response stream for the command with the following fields:
            commandStatus: Status of the command (enumeration)
            progressInfo: Information on the progress of the command (0 to 1)
            estimatedRemainingTime: Estimate of the remaining time required to run the command
            updatedLifetimeOfExecution: An update on the execution lifetime
        """
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        # Get the UUID of the command
        command_uuid = request.value
    
        # Get the current state
        execution_info = self._get_command_state(command_uuid=command_uuid)
    
        # construct the initial return dictionary in case while is not executed
        return_values = {'commandStatus': execution_info.commandStatus}
        if execution_info.HasField('progressInfo'):
            return_values['progressInfo'] = execution_info.progressInfo
        if execution_info.HasField('estimatedRemainingTime'):
            return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
        if execution_info.HasField('updatedLifetimeOfExecution'):
            return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
        # we loop only as long as the command is running
        while execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.waiting \
                or execution_info.commandStatus == silaFW_pb2.ExecutionInfo.CommandStatus.running:
            # TODO:
            #   Evaluate the command status --> command_status. Options:
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.running
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedSuccessfully
            #       command_stats = silaFW_pb2.ExecutionInfo.CommandStatus.finishedWithError
            #   Optional:
            #       * Determine the progress (progressInfo)
            #       * Determine the estimated remaining time
            #       * Update the Lifetime of execution
    
            # Update all values
            execution_info = self._get_command_state(command_uuid=command_uuid)
    
            # construct the return dictionary
            return_values = {'commandStatus': execution_info.commandStatus}
            if execution_info.HasField('progressInfo'):
                return_values['progressInfo'] = execution_info.progressInfo
            if execution_info.HasField('estimatedRemainingTime'):
                return_values['estimatedRemainingTime'] = execution_info.estimatedRemainingTime
            if execution_info.HasField('updatedLifetimeOfExecution'):
                return_values['updatedLifetimeOfExecution'] = execution_info.updatedLifetimeOfExecution
    
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
            # we add a small delay to give the client a chance to keep up.
            time.sleep(0.5)
        else:
            # one last time yield the status
            yield silaFW_pb2.ExecutionInfo(**return_values)
    
    def GetLog_Result(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetLog_Responses:
        """
        Returns the final result of the command call :meth:`~.GetLog`.
    
        :param request: A request object with the following properties
            CommandExecutionUUID: The UUID of the command executed.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentLogLevel (Current Log Level): The current log level of the latest logs , retrieved from the
            SiLA server log file.
            request.CurrentLogTimestamp (Current Log Timestamp): The current log timestamp of the latest logs ,
            retrieved from the SiLA server log file.
            request.CurrentLogMessage (Current Log Level): The current log level of the latest logs , retrieved from
            the SiLA server log file.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.GetLog_Responses = None
    
        # Get the UUID of the command
        command_uuid = request.value
    
        # TODO:
        #   Add implementation of Simulation for command GetLog here and write the resulting response
        #   in return_value
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetLog_Responses(
                **default_dict['GetLog_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetPumpAddress(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.SetPumpAddress_Responses:
        """
        Executes the unobservable command "Set Pump Address"
            Set the address of the pump (1-8).
    
        :param request: gRPC request containing the parameters passed:
            request.Address (Address):
            Set the address of the pump (1-8).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpAddressSet (Pump Address Set): Pump Address succeeded to Set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'@{request.Address.value}\r\n'
        try:
            read = '*'
            par_dict = {
                'PumpAddressSet': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.SetPumpAddress_Responses(**par_dict)
            self.properties.pump_address_sim = request.Address.value
            logging.debug('Executed command SetPumpAddress in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.SetPumpAddress_Responses(
                **default_dict['SetPumpAddress_Responses']
            )

        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetPumpStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetPumpStatus_Responses:
        """
        Executes the unobservable command "Get Pump Status"
            Get pump status. +=running, -=stopped/standby.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPumpStatus (Current Pump Status): Current pump status. +=running, -=stopped/standby.
        """
    
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}E\r\n'
        try:
            read = read = '+'
            par_dict = {
                'CurrentPumpStatus': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.GetPumpStatus_Responses(**par_dict)
            logging.debug('Executed command GetPumpStatus in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetPumpStatus_Responses(
                **default_dict['GetPumpStatus_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetVersionType(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetVersionType_Responses:
        """
        Executes the unobservable command "Get Version Type"
            Get pump information. Response is string of model description (variable length), software version
            (3 digits) and pump head model type code (3 digits).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVersionType (Current Version Type): Current pump information. Response is string of model
            description (variable length), software version (3 digits) and pump head model type code (3 digits).
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}#\r\n'
        try:
            read = 'REGLO DIGITAL 300 206'
            par_dict = {
                'CurrentVersionType': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.GetVersionType_Responses(**par_dict)
            logging.debug('Executed command GetVersionType in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetVersionType_Responses(
                **default_dict['GetVersionType_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetVersionSoftware(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetVersionSoftware_Responses:
        """
        Executes the unobservable command "Current Version Software"
            Get pump software version. Response is string.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVersionSoftware (Current Version Software): Current pump software version. Response is
            string.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}(\r\n'
        try:
            read = '0300'
            par_dict = {
                'CurrentVersionSoftware': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.GetVersionSoftware_Responses(**par_dict)
            logging.debug('Executed command GetVersionSoftware in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetVersionSoftware_Responses(
                **default_dict['GetVersionSoftware_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetPumpID(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetPumpID_Responses:
        """
        Executes the unobservable command "Get Pump ID"
            Get pump head identification number.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ID (Identification Number): Pump head identification number query response.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim})\r\n'
        try:
            read = '0206'
            par_dict = {
                'ID': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.GetPumpID_Responses(**par_dict)
            logging.debug('Executed command GetPumpID in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetPumpID_Responses(
                **default_dict['GetPumpID_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetPumpID(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.SetPumpID_Responses:
        """
        Executes the unobservable command "Set Pump ID"
            Set pump head identification number.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpID (Pump ID): The new pump head identification number
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ID (Identification Number): Pump head identification number query response.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = str(request.PumpID.value)
        user_input = user_input.zfill(4)  # If to short, pad to length 4
        user_input = user_input[:4]  # If to long, strictly limit to 4 digits

        command = f'{self.properties.pump_address_sim}){user_input}\r\n'
        try:
            read = '*'
            par_dict = {
                'ID': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.SetPumpID_Responses(**par_dict)
            logging.debug('Executed command SetPumpID in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.SetPumpID_Responses(
                **default_dict['SetPumpID_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def ResetToDefault(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.ResetToDefault_Responses:
        """
        Executes the unobservable command "Reset To Default"
            Resets all user configurable data to default values.
    
        :param request: gRPC request containing the parameters passed:
            request.Reset (Reset):
            Resets all user configurable data to default values.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ResetStatus (Reset Status): Reset of pump succeeded.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}0\r\n'
        try:
            read = '*'
            par_dict = {
                'ResetStatus': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.ResetToDefault_Responses(**par_dict)
            logging.debug('Executed command ResetToDefault in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.ResetToDefault_Responses(
                **default_dict['ResetToDefault_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetTotalVolume(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.GetTotalVolume_Responses:
        """
        Executes the unobservable command "Get Total Volume"
            Get total volume pumped since last reset, in muL, mL, or L.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TotalVolume (Total Volume): The total volume pumped since last reset, in muL, mL, or L
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}:\r\n'
        try:
            read = '4.983'
            par_dict = {
                'TotalVolume': silaFW_pb2.Real(value=float(read)),
            }
            return_value = DeviceServicer_pb2.GetTotalVolume_Responses(**par_dict)
            logging.debug('Executed command GetTotalVolume in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.GetTotalVolume_Responses(
                **default_dict['GetTotalVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def ResetTotalVolume(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.ResetTotalVolume_Responses:
        """
        Executes the unobservable command "Reset Total Volume"
            Reset the total dispensed volume since the last reset.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ResetResponse (ResetResponse): Successfully reset the total dispensed volume.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}W\r\n'
        try:
            read = '*'
            par_dict = {
                'ResetResponse': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.ResetTotalVolume_Responses(**par_dict)
            logging.debug('Executed command ResetTotalVolume in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.ResetTotalVolume_Responses(
                **default_dict['ResetTotalVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def UnlockControlPanel(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.UnlockControlPanel_Responses:
        """
        Executes the unobservable command "Unlock Control Panel"
            Switch control panel to manual operation.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ControlPanelUnlocked (Control Panel Unlocked): Control panel switched to manual operation.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}A\r\n'
        try:
            read = '*'
            par_dict = {
                'ControlPanelUnlocked': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.UnlockControlPanel_Responses(**par_dict)
            logging.debug('Executed command UnlockControlPanel in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.UnlockControlPanel_Responses(
                **default_dict['UnlockControlPanel_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def LockControlPanel(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.LockControlPanel_Responses:
        """
        Executes the unobservable command "Lock Control Panel"
            Set control panel to inactive (Input via control keys is not possible).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ControlPanelLocked (Control Panel Locked): The control panel has been locked.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}B\r\n'
        try:
            read = '*'
            par_dict = {
                'ControlPanelLocked': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.LockControlPanel_Responses(**par_dict)
            logging.debug('Executed command LockControlPanel in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.LockControlPanel_Responses(
                **default_dict['LockControlPanel_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDisplayNumbers(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.SetDisplayNumbers_Responses:
        """
        Executes the unobservable command "Set Display Numbers"
            Write numbers to the pump to display while under external control - float of length 5 including +/- and
            decimal points.
    
        :param request: gRPC request containing the parameters passed:
            request.DisplayNumbers (Display Numbers):
            Write numbers to the pump to display while under external control - float of length 5 including +/- and
            decimal points.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.DisplayNumbersSet (Display Numbers Set): Display Numbers succeeded to Set.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        user_input = str(request.DisplayNumbers.value)[::-1].zfill(5)[::-1]  # If to short, pad to length 5
        user_input = user_input[:5]  # If to long, strictly limit to 5 digits
        command = f'{self.properties.pump_address_sim}D{user_input}\r\n'

        try:
            read = '*'
            par_dict = {
                'DisplayNumbersSet': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.SetDisplayNumbers_Responses(**par_dict)
            logging.debug('Executed command SetDisplayNumbers in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.SetDisplayNumbers_Responses(
                **default_dict['SetDisplayNumbers_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDisplayLetters(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.SetDisplayLetters_Responses:
        """
        Executes the unobservable command "Set Display Letters"
            Write letters to the pump to display while under external control - string of length 4.
    
        :param request: gRPC request containing the parameters passed:
            request.DisplayLetters (Display Letters):
            Write letters to the pump to display while under external control - string of length 4.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.DisplayLettersSet (Display Letters Set): Display Letters succeeded to Set.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        user_input = str(request.DisplayLetters.value)[::-1].zfill(4)[::-1]  # If to short, pad to length 4
        user_input = user_input[:4]  # If to long, strictly limit to 4 letters
        command = f'{self.properties.pump_address_sim}DA{user_input}\r\n'

        try:
            read = '*'
            par_dict = {
                'DisplayLettersSet': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.SetDisplayLetters_Responses(**par_dict)
            logging.debug('Executed command SetDisplayLetters in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None

        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.SetDisplayLetters_Responses(
                **default_dict['SetDisplayLetters_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetCommunicationPort(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.SetCommunicationPort_Responses:
        """
        Executes the unobservable command "Set Pump ID"
            Set pump head identification number.
    
        :param request: gRPC request containing the parameters passed:
            request.PortName (Port Name): The name of the serial port.
            request.BaudRate (Baud Rate): The baud rate of the serial port connection.
            request.Parity (Parity):  Enable parity checking. Possible values: NONE, EVEN, ODD, MARK, SPACE. Default is
            NONE
            request.StopBits (Stop Bits):  Number of stop bits. Possible values: ONE, ONE_POINT_FIVE, TWO. Default is
            ONE
            request.Timeout (Timeout): Set a read timeout value. Default is 1.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetCommunicationPortStatus (Set Communication Port Status): The status of the set command.
        """
    
        # initialise the return value
        return_value = None

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        parity_options: dict = {
            '': serial.PARITY_NONE,
            'NONE': serial.PARITY_NONE,
            'EVEN': serial.PARITY_EVEN,
            'ODD': serial.PARITY_ODD,
            'MARK': serial.PARITY_MARK,
            'SPACE': serial.PARITY_SPACE
        }
        stop_bits_options: dict = {
            '': serial.STOPBITS_ONE,
            'ONE': serial.STOPBITS_ONE,
            'ONE_POINT_FIVE': serial.STOPBITS_ONE_POINT_FIVE,
            'TWO': serial.STOPBITS_TWO
        }
        warning_msg_parity = ''
        warning_msg_stop_bit = ''
        try:
            self.properties.port_name_sim = request.PortName.value
            self.properties.baud_rate_sim = request.BaudRate.value
            if request.Parity.value in parity_options.keys():
                self.properties.parity_sim = parity_options[request.Parity.value]
            else:
                logging.warning(f'The supplied value for parameter \'Parity\' does not match the required format. '
                                f'Possible values (string): NONE, EVEN, ODD, MARK, SPACE. Default is NONE. Current value '
                                f'remains at {self.properties.parity_sim}')
                warning_msg_parity = 'Parity not updated.'
            if request.StopBits.value in stop_bits_options.keys():
                self.properties.stop_bits_sim = stop_bits_options[request.StopBits.value]
            else:
                logging.warning(f'The supplied value for parameter \'StopBits\' does not match the required format.'
                                f'Possible values (string): ONE, ONE_POINT_FIVE, TWO. Default is ONE. Current value '
                                f'remains at {self.properties.stop_bits_sim}')
                warning_msg_stop_bit = 'StopBit not updated.'
            self.properties.timeout_sim = request.Timeout.value

            if warning_msg_parity == '' and warning_msg_stop_bit == '':
                read = 'Updated'
            elif warning_msg_parity != '' or warning_msg_stop_bit != '':
                read = f'Bad format: {warning_msg_parity} {warning_msg_stop_bit}'
            else:
                read = f'Update unsuccessful'
        except Exception as e:
            logging.exception(e)
            read = 'Values not updated'

        par_dict = {
            'SetCommunicationPortStatus': silaFW_pb2.String(value=read),
        }
        return_value = DeviceServicer_pb2.SetCommunicationPort_Responses(**par_dict)
        logging.debug('Executed command SetCommunicationPort in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.SetCommunicationPort_Responses(
                **default_dict['SetCommunicationPort_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def ConnectDevice(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.ConnectDevice_Responses:
        """
        Executes the unobservable command "Connect Device"
            Sets up a serial connection with the device using the specified connection details.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ConnectionStatus (ConnectionStatus): The connection status. If connection was unsuccessful, the
            error will be returned.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        read = self.properties.connect()
        par_dict = {
            'ConnectionStatus': silaFW_pb2.String(value=read),
        }
        return_value = DeviceServicer_pb2.ConnectDevice_Responses(**par_dict)
        logging.debug('Executed command ConnectDevice in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.ConnectDevice_Responses(
                **default_dict['ConnectDevice_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def ResetOverload(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.ResetOverload_Responses:
        """
        Executes the unobservable command "Reset Overload"
            Reset the device command input buffer overload.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ResetOverloadStatus (Reset Overload Status): The reset overload status. If reset was unsuccessful,
            the error will be returned.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address_sim}-\r\n'

        try:
            read = '*'
            par_dict = {
                'ResetOverloadStatus': silaFW_pb2.String(value=read),
            }
            return_value = DeviceServicer_pb2.ResetOverload_Responses(**par_dict)
            logging.debug('Executed command ResetOverload in mode: {mode} with response: {response}'
                          .format(mode='Simulation', response=read))
        except (ValueError, IndexError):
            logging.exception('Parsing of the following command response failed: {command}, {response}'
                              .format(command=command, response=read))
        except ConnectionError:
            logging.exception(f'Communication failed executing the command: {command}')
            return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.ResetOverload_Responses(
                **default_dict['ResetOverload_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def Subscribe_CurrentStatus(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Subscribe_CurrentStatus_Responses:
        """
        Requests the observable property Current Status
            Get the current status of the device from the internal state machine of the SiLA server.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.CurrentStatus (Current Status): Get the current status of the device from the internal state
            machine of the SiLA server.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Subscribe_CurrentStatus_Responses = None
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        # we could use a timeout here if we wanted
        while True:
            # TODO:
            #   Add implementation of Simulation for property CurrentStatus here and write the resulting
            #   response in return_value
    
            # create the default value
            if return_value is None:
                return_value = DeviceServicer_pb2.Subscribe_CurrentStatus_Responses(
                    **default_dict['Subscribe_CurrentStatus_Responses']
                )
    
    
            yield return_value

    def Get_PortName(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_PortName_Responses:
        """
        Requests the unobservable property Port Name
            The name of the serial port. 
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.PortName (Port Name): The name of the serial port.
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Get_PortName_Responses = None

        read = self.properties.port_name_sim
        par_dict = {
            'PortName': silaFW_pb2.String(value=read),
        }
        return_value = DeviceServicer_pb2.Get_PortName_Responses(**par_dict)
        logging.debug('Executed command Get_PortName in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_PortName_Responses(
                **default_dict['Get_PortName_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def Get_BaudRate(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_BaudRate_Responses:
        """
        Requests the unobservable property Baud Rate
            The baud rate of the serial port connection. 
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.BaudRate (Baud Rate): The baud rate of the serial port connection.
        """

        # initialise the return value
        return_value: DeviceServicer_pb2.Get_BaudRate_Responses = None

        read = self.properties.baud_rate_sim
        par_dict = {
            'BaudRate': silaFW_pb2.Integer(value=read),
        }
        return_value = DeviceServicer_pb2.Get_BaudRate_Responses(**par_dict)
        logging.debug('Executed command Get_BaudRate in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_BaudRate_Responses(
                **default_dict['Get_BaudRate_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def Get_Parity(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_Parity_Responses:
        """
        Requests the unobservable property Parity
            Enable parity checking. Possible values: NONE, EVEN, ODD, MARK, SPACE. Default is NONE
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.Parity (Parity):  Enable parity checking. Possible values: NONE, EVEN, ODD, MARK, SPACE. Default is
            NONE
        """
    
        # initialise the return value
        return_value: DeviceServicer_pb2.Get_Parity_Responses = None

        read = self.properties.parity_sim
        par_dict = {
            'Parity': silaFW_pb2.String(value=read),
        }
        return_value = DeviceServicer_pb2.Get_Parity_Responses(**par_dict)
        logging.debug('Executed command Get_Parity in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_Parity_Responses(
                **default_dict['Get_Parity_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def Get_StopBits(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_StopBits_Responses:
        """
        Requests the unobservable property Stop Bits
            Number of stop bits. Possible values: ONE, ONE_POINT_FIVE, TWO. Default is ONE
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.StopBits (Stop Bits):  Number of stop bits. Possible values: ONE, ONE_POINT_FIVE, TWO. Default is
            ONE
        """

        # initialise the return value
        return_value: DeviceServicer_pb2.Get_StopBits_Responses = None

        read = self.properties.stop_bits_sim
        par_dict = {
            'StopBits': silaFW_pb2.String(value=str(read)),
        }
        return_value = DeviceServicer_pb2.Get_StopBits_Responses(**par_dict)
        logging.debug('Executed command Get_StopBits in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_StopBits_Responses(
                **default_dict['Get_StopBits_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def Get_Timeout(self, request, context: grpc.ServicerContext) \
            -> DeviceServicer_pb2.Get_Timeout_Responses:
        """
        Requests the unobservable property Timeout
            Set a read timeout value. Default is 1.
    
        :param request: An empty gRPC request object (properties have no parameters)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: A response object with the following fields:
            request.Timeout (Timeout): Set a read timeout value. Default is 1.
        """

        # initialise the return value
        return_value: DeviceServicer_pb2.Get_Timeout_Responses = None

        read = self.properties.timeout_sim
        par_dict = {
            'Timeout': silaFW_pb2.Real(value=read),
        }
        return_value = DeviceServicer_pb2.Get_Timeout_Responses(**par_dict)
        logging.debug('Executed command Get_Timeout in mode: {mode} with response: {response}'
                      .format(mode='Simulation', response=read))
    
        # fallback to default
        if return_value is None:
            return_value = DeviceServicer_pb2.Get_Timeout_Responses(
                **default_dict['Get_Timeout_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value
