"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Parameter Control Servicer*

:details: ParameterControlServicer:
    Set and retrieve information regarding the parameter settings of the Reglo DC (digital control) pump.
    Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 15.03.2021.

:file:    ParameterControlServicer_real.py
:authors: Lukas Bromig

:date: (creation)          2021-03-16T12:40:00.497289
:date: (last modification) 2021-03-16T12:40:00.497289

.. note:: Code generated by sila2codegenerator 0.3.3-dev

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only
import inspect      # used for status determination

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import ParameterControlServicer_pb2 as ParameterControlServicer_pb2
# from .gRPC import ParameterControlServicer_pb2_grpc as ParameterControlServicer_pb2_grpc

# import default arguments
from .ParameterControlServicer_default_arguments import default_dict


# noinspection PyPep8Naming,PyUnusedLocal
class ParameterControlServicerReal:
    """
    Implementation of the *Parameter Control Servicer* in *Real* mode
        This is a RegloDC Service
    """

    def __init__(self, properties):
        """Class initialiser"""

        self.properties = properties
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    @staticmethod
    def _get_function_name():
        return inspect.stack()[1][3]

    def SetRPMMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetRPMMode_Responses:
        """
        Executes the unobservable command "Set RPM Mode"
            Set the pump mode to RPM mode (MODE PUMP% rpm).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): RPM pump mode successfully set.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}L\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetRPMMode_Responses(**par_dict)
                logging.debug('Executed command SetRPMMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetRPMMode_Responses(
                **default_dict['SetRPMMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetFlowRateMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetFlowRateMode_Responses:
        """
        Executes the unobservable command "Set Flow Rate Mode"
            Set the pump mode to Flow Rate mode (MODE PUMP% Flow rate).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Flow rate pump mode successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}M\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetFlowRateMode_Responses(**par_dict)
                logging.debug('Executed command SetFlowRateMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetFlowRateMode_Responses(
                **default_dict['SetFlowRateMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseVolumeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseVolumeMode_Responses:
        """
        Executes the unobservable command "Set Dispense Volume Mode"
            Set the pump mode to dispense volume mode (MODE DISP Volume).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Dispense volume pump mode successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}N\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseVolumeMode_Responses(**par_dict)
                logging.debug('Executed command SetDispenseVolumeMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseVolumeMode_Responses(
                **default_dict['SetDispenseVolumeMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseTimeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseTimeMode_Responses:
        """
        Executes the unobservable command "Set Dispense Time Mode"
            Set the pump mode to dispense time mode (MODE DISP Time).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Dispense time pump mode successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}O\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseTimeMode_Responses(**par_dict)
                logging.debug('Executed command SetDispenseTimeMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseTimeMode_Responses(
                **default_dict['SetDispenseTimeMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetPauseTimeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetPauseTimeMode_Responses:
        """
        Executes the unobservable command "Set Pause Time Mode"
            Set the current pump mode to pause-time mode (MODE PAUSE Time).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pause time pump mode successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}]\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetPauseTimeMode_Responses(**par_dict)
                logging.debug('Executed command SetPauseTimeMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetPauseTimeMode_Responses(
                **default_dict['SetPauseTimeMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseTimePauseTimeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseTimePauseTimeMode_Responses:
        """
        Executes the unobservable command "Set Time Mode"
            Set the pump mode to dispense time/ pause time mode (DISP Time + PAUSE Time).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Dispense time/ pause time pump mode successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}P\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseTimePauseTimeMode_Responses(**par_dict)
                logging.debug('Executed command SetDispenseTimePauseTimeMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseTimePauseTimeMode_Responses(
                **default_dict['SetDispenseTimePauseTimeMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseVolumePauseTimeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseVolumePauseTimeMode_Responses:
        """
        Executes the unobservable command "Set Dispense Volume Pause Time Mode"
            Set the pump mode to dispense volume/ pause time mode (DISP Volume + PAUSE Time).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Dispense volume/ pause time pump mode successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}Q\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'PumpModeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseVolumePauseTimeMode_Responses(**par_dict)
                logging.debug('Executed command SetDispenseVolumePauseTimeMode in mode: {mode} with response: {response}'
                              .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseVolumePauseTimeMode_Responses(
                **default_dict['SetDispenseVolumePauseTimeMode_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetDispenseVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetDispenseVolume_Responses:
        """
        Executes the unobservable command "Get Dispense Volume"
            Get the current setting for the dispensing volume in mL/min.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.DispenseVolume (Dispense Volume): Current setting for the dispensing volume in mL/min.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}v\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'DispenseVolume': silaFW_pb2.Real(value=float(read)),
                }
                return_value = ParameterControlServicer_pb2.GetDispenseVolume_Responses(**par_dict)
                logging.debug(
                    'Executed command GetDispenseVolume in mode: {mode} with response: {response}'
                    .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetDispenseVolume_Responses(
                **default_dict['GetDispenseVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseVolume_Responses:
        """
        Executes the unobservable command "Set Dispense Volume"
            Set the dispensing volume in mL.
    
        :param request: gRPC request containing the parameters passed:
            request.Volume (Volume):
            The volume to be set in mL.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VolumeSet (Volume Set): Volume successfully set
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = '{:.0e}'.format(request.Volume.value)
        mantisse, exponent = user_input.split('e')
        e_sig, e_val = exponent[:1], exponent[-1]
        mantisse = mantisse.zfill(4)  # If to short, pad to length 5
        mantisse = mantisse[:4]  # If to long, strictly limit to 5 digits

        command = f'{self.properties.pump_address}v{mantisse}{e_sig}{e_val}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'VolumeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseVolume_Responses(**par_dict)
                logging.debug(
                    'Executed command SetDispenseVolume in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseVolume_Responses(
                **default_dict['SetDispenseVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseVolumeModeVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseVolumeModeVolume_Responses:
        """
        Executes the unobservable command "Set Dispense Volume Mode Volume"
            Set the dispensing volume for the mode DISP Volume in mL (The entered dispensing volume is rounded down to complete roller steps).
    
        :param request: gRPC request containing the parameters passed:
            request.Volume (Volume):
            The dispensing volume  to be set in mL for mode DISP Volume.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VolumeSet (Volume Set): Volume successfully set
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = str(request.Volume.value)
        user_input = user_input.zfill(5)  # If to short, pad to length 5
        user_input = user_input[:5]  # If to long, strictly limit to 5 digits

        command = f'{self.properties.pump_address}[{user_input}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'VolumeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseVolumeModeVolume_Responses(**par_dict)
                logging.debug(
                    'Executed command SetDispenseVolumeModeVolume in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseVolumeModeVolume_Responses(
                **default_dict['SetDispenseVolumeModeVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetFlowRate(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetFlowRate_Responses:
        """
        Executes the unobservable command "Get Flow Rate"
            Get current flow rate (mL/min).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFlowRate (Current Flow Rate): Current flow rate of the pump.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}f\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentFlowRate': silaFW_pb2.Real(value=float(read)),
                }
                return_value = ParameterControlServicer_pb2.GetFlowRate_Responses(**par_dict)
                logging.debug(
                    'Executed command GetFlowRate in mode: {mode} with response: {response}'
                    .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetFlowRate_Responses(
                **default_dict['GetFlowRate_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetFlowRate(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetFlowRate_Responses:
        """
        Executes the unobservable command "Set Flow Rate"
            Set current flow rate (mL/min).
    
        :param request: gRPC request containing the parameters passed:
            request.FlowRate (Flow Rate):
            Set flow rate in mL/min.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetFlowRateSet (Set Flow Rate): Flow rate in mL/min successfully set
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = '{:.0e}'.format(request.FlowRate.value)
        mantisse, exponent = user_input.split('e')
        e_sig, e_val = exponent[:1], exponent[-1]
        mantisse = mantisse.zfill(4)  # If to short, pad to length 5
        mantisse = mantisse[:4]  # If to long, strictly limit to 5 digits

        command = f'{self.properties.pump_address}f{mantisse}{e_sig}{e_val}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'SetFlowRateSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetFlowRate_Responses(**par_dict)
                logging.debug(
                    'Executed command SetFlowRate in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetFlowRate_Responses(
                **default_dict['SetFlowRate_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetRPM(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetRPM_Responses:
        """
        Executes the unobservable command "Get RPM"
            Get current rpm.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentRPM (Current RPM): Current rpm of the pump.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}S\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentRPM': silaFW_pb2.Real(value=float(read)),
                }
                return_value = ParameterControlServicer_pb2.GetRPM_Responses(**par_dict)
                logging.debug(
                    'Executed command GetRPM in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetRPM_Responses(
                **default_dict['GetRPM_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetRPM(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetRPM_Responses:
        """
        Executes the unobservable command "Set RPM"
            Set pump speed in rpm (100 - 10000 (4 channel type); 160 - 16000 (2 channel type)).
    
        :param request: gRPC request containing the parameters passed:
            request.RPM (RPM):
            Set pump speed in rpm (0.1 - 1000 (4 channel type); 0.16 - 1600 (2 channel type)).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetRPMSet (Set RPM): RPM successfully set
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = str(request.RPM.value)
        if '.' in user_input:
            integer_part, float_part = user_input.split('.')
            integer_part = integer_part.zfill(4)
            float_part = float_part.ljust(2, '0')
            user_input = integer_part[:4] + float_part[:2]
        else:
            user_input = user_input.zfill(4)  # If to short, pad to length 4
            user_input = user_input[:4] + '00'  # If to long, strictly limit to 4 digits

        command = f'{self.properties.pump_address}S{user_input}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'SetRPMSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetRPM_Responses(**par_dict)
                logging.debug(
                    'Executed command SetRPM in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetRPM_Responses(
                **default_dict['SetRPM_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetPauseTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetPauseTime_Responses:
        """
        Executes the unobservable command "Get Pause Time"
            Get pause time in s.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPauseTime (Current Pause Pause Time): Current pause time in s.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}T\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentPauseTime': silaFW_pb2.Real(value=float(read)*10),
                }
                return_value = ParameterControlServicer_pb2.GetPauseTime_Responses(**par_dict)
                logging.debug(
                    'Executed command GetPauseTime in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetPauseTime_Responses(
                **default_dict['GetPauseTime_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetPauseTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetPauseTime_Responses:
        """
        Executes the unobservable command "Set Pause Time"
            Set pause time in s, min, and h.
    
        :param request: gRPC request containing the parameters passed:
            request.PauseTimeS (Pause Time S):
            Set pumping pause time in s. (0 - 999.9).
            request.PauseTimeMin (Pause Time Min):
            Set pumping pause time in min. (0 - 899).
            request.PauseTimeH (Pause Time H):
            Set pumping pause time in H. (0 - 999).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetPauseTimeSet (Set Pause Time): Pumping pause time successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    if request.PauseTimeS.value != 0.0:
                        user_input_s = str(request.PauseTimeS.value * 10)
                        user_input_s = user_input_s.zfill(4)  # If to short, pad to length 4
                        user_input_s = user_input_s[:4]  # If to long, strictly limit to 4 digits
                        command = f'{self.properties.pump_address}T{user_input_s}\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    elif request.PauseTimeS.value != 0:
                        user_input_m = str(request.PauseTimeMin.value)
                        user_input_m = user_input_m.zfill(3)  # If to short, pad to length 3
                        user_input_m = user_input_m[:3]  # If to long, strictly limit to 3 digits
                        command = f'{self.properties.pump_address}TM{user_input_m}\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    elif request.PauseTimeS.value != 0:
                        user_input_h = str(request.PauseTimeH.value)
                        user_input_h = user_input_h.zfill(3)  # If to short, pad to length 3
                        user_input_h = user_input_h[:3]  # If to long, strictly limit to 3 digits
                        command = f'{self.properties.pump_address}TH{user_input_h}\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    else:
                        command = f'{self.properties.pump_address_sim}T0000\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'SetPauseTimeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetPauseTime_Responses(**par_dict)
                logging.debug(
                    'Executed command SetPauseTime in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=(command), response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {(command)}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetPauseTime_Responses(
                **default_dict['SetPauseTime_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetDispenseCycles(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetDispenseCycles_Responses:
        """
        Executes the unobservable command "Get Dispense Cycle"
            Get number of dispense cycles.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.DispenseCycles (Dispense Cycles): Number of dispense cycles.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}"\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'DispenseCycles': silaFW_pb2.Integer(value=int(read)),
                }
                return_value = ParameterControlServicer_pb2.GetDispenseCycles_Responses(**par_dict)
                logging.debug(
                    'Executed command GetDispenseCycles in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetDispenseCycles_Responses(
                **default_dict['GetDispenseCycles_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseCycles(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseCycles_Responses:
        """
        Executes the unobservable command "Set Dispense Cycles"
            Set number of dispensing cycles (integer).
    
        :param request: gRPC request containing the parameters passed:
            request.DispenseCycles (Dispense Cycles):
            The number of dispense cycles to be set.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetDispenseCycleSet (Set Dispense Cycle): Number of dispense cycles successfully set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = str(request.DispenseCycles.value)
        user_input = user_input.zfill(4)  # If to short, pad to length 4
        user_input = user_input[:4]  # If to long, strictly limit to 4 digits

        command = f'{self.properties.pump_address}"{user_input}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'SetDispenseCycleSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseCycles_Responses(**par_dict)
                logging.debug(
                    'Executed command GetDispenseCycles in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseCycles_Responses(
                **default_dict['SetDispenseCycles_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetDispenseTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetDispenseTime_Responses:
        """
        Executes the unobservable command "Get Dispense Time"
            Get the time to dispense in s.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDispenseTime (CurrentDispenseTime): CurrentDispenseTime in s.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}V\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentDispenseTime': silaFW_pb2.Real(value=float(read)/10),
                }
                return_value = ParameterControlServicer_pb2.GetDispenseTime_Responses(**par_dict)
                logging.debug(
                    'Executed command GetDispenseTime in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetDispenseTime_Responses(
                **default_dict['GetDispenseTime_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetDispenseTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetDispenseTime_Responses:
        """
        Executes the unobservable command "Set Dispense Time"
            Set the time to dispense in s, min, and h.
    
        :param request: gRPC request containing the parameters passed:
            request.CurrentDispenseTimeS (Current Dispense Time S): Current dispense time in s (0 - 999).
            request.CurrentDispenseTimeMin (Current Dispense Time Min): Current dispense time in min (0 - 899).
            request.CurrentDispenseTimeH (Current Dispense Time H): Current dispense time in h (0 - 999).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetDispenseTimeSet (Set Dispense Time S): The dispense time in s was set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():

                    if request.CurrentDispenseTimeS.value != 0.0:
                        user_input_s = str(request.CurrentDispenseTimeS.value * 10)
                        user_input_s = user_input_s.zfill(4)  # If to short, pad to length 4
                        user_input_s = user_input_s[:4]  # If to long, strictly limit to 4 digits
                        command = f'{self.properties.pump_address}V{user_input_s}\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    elif request.CurrentDispenseTimeMin.value != 0:
                        user_input_m = str(request.CurrentDispenseTimeMin.value)
                        user_input_m = user_input_m.zfill(3)  # If to short, pad to length 3
                        user_input_m = user_input_m[:3]  # If to long, strictly limit to 3 digits
                        command = f'{self.properties.pump_address}VM{user_input_m}\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    elif request.CurrentDispenseTimeH.value != 0:
                        user_input_h = str(request.CurrentDispenseTimeH.value)
                        user_input_h = user_input_h.zfill(3)  # If to short, pad to length 3
                        user_input_h = user_input_h[:3]  # If to long, strictly limit to 3 digits
                        command = f'{self.properties.pump_address}VH{user_input_h}\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    else:
                        command = f'{self.properties.pump_address_sim}V0000\r\n'
                        self.properties.ser.write(str.encode(command))
                        read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'SetDispenseTimeSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetDispenseTime_Responses(**par_dict)
                logging.debug(
                    'Executed command SetDispenseTime in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=(command), response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {(command)}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetDispenseTime_Responses(
                **default_dict['SetDispenseTime_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetRollerSteps(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetRollerSteps_Responses:
        """
        Executes the unobservable command "Get Roller Steps"
            Get roller steps for mode DISP Volume.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentRollerSteps (Current Roller Steps): Current roller steps for mode DISP Volume.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}U\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentRollerSteps': silaFW_pb2.Integer(value=int(read)),
                }
                return_value = ParameterControlServicer_pb2.GetRollerSteps_Responses(**par_dict)
                logging.debug(
                    'Executed command GetRollerSteps in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetRollerSteps_Responses(
                **default_dict['GetRollerSteps_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetRollerSteps(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetRollerSteps_Responses:
        """
        Executes the unobservable command "Set Roller Steps"
            Set the roller steps for mode DISP Volume.
    
        :param request: gRPC request containing the parameters passed:
            request.CurrentRollerSteps (Current Roller Steps): Current roller steps for mode DISP Volume.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetRollerStepsSet (Set Current Roller Steps Set): The set roller steps for mode DISP Volume.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = request.CurrentRollerSteps.value
        u = '0000'
        U = '0000'
        if user_input <= 65535:
            U = str(user_input).zfill(4)  # If to short, pad to length 4
            U = U[:4]  # If to long, strictly limit to 4 digits
            u = '0000'
        elif user_input > 65535:
            u = str(int(user_input / 65535)).zfill(4)  # If to short, pad to length 4
            u = u[:4]  # If to long, strictly limit to 4 digits
            U = str(user_input % 65535).zfill(4)   # If to short, pad to length 4
            U = U[:4]  # If to long, strictly limit to 4 digits

        command_1 = f'{self.properties.pump_address}U{U}\r\n'
        command_2 = f'{self.properties.pump_address}u{u}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command_1))
                    read_1 = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    self.properties.ser.write(str.encode(command_2))
                    read_2 = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))

                if read_1 == '*' and read_2 == '*':
                    read = int(u) * 65536 + int(U)
                else:
                    read = 999999
                    # read = f'U:{read_1},u:{read_2}'
                par_dict = {
                    'SetRollerStepsSet': silaFW_pb2.Integer(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetRollerSteps_Responses(**par_dict)
                logging.debug(
                    'Executed command SetRollerSteps in mode: {mode} with response: {response}'
                    .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=(command_1+' '+command_2), response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command_1} {command_2}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetRollerSteps_Responses(
                **default_dict['SetRollerSteps_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetRollerStepVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetRollerStepVolume_Responses:
        """
        Executes the unobservable command "Get Roller Step Volume"
            Get roller step volume in nano-liters (nL).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentRollerStepVolume (Current Roller Step Volume): Current roller step volume in nano-liters (nL).
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}r\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentRollerStepVolume': silaFW_pb2.Integer(value=int(float(read))),
                }
                return_value = ParameterControlServicer_pb2.GetRollerStepVolume_Responses(**par_dict)
                logging.debug(
                    'Executed command GetRollerStepVolume in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetRollerStepVolume_Responses(
                **default_dict['GetRollerStepVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetRollerStepVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetRollerStepVolume_Responses:
        """
        Executes the unobservable command "Set Roller Step Volume"
            Set the roller step volume in nano-liters (nL).
    
        :param request: gRPC request containing the parameters passed:
            request.CurrentRollerStepVolume (Current Roller Step Volume): Current roller step volume in nano-liters (nL).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetRollerStepVolumeSet (Set Current Roller Step Volume Set): The set roller step volume in nano-liters (nL).
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = '{:.0e}'.format(request.CurrentRollerStepVolume.value)
        mantisse, exponent = user_input.split('e')
        e_sig, e_val = exponent[:1], exponent[-1]
        mantisse = mantisse.zfill(4)  # If to short, pad to length 5
        mantisse = mantisse[:4]  # If to long, strictly limit to 5 digits

        command = f'{self.properties.pump_address}r{mantisse}{e_sig}{e_val}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    if 'I' in read:
                        read.replace('I', '-')
                par_dict = {
                    'SetRollerStepVolumeSet': silaFW_pb2.Integer(value=int(float(read))),
                }
                return_value = ParameterControlServicer_pb2.SetRollerStepVolume_Responses(**par_dict)
                logging.debug(
                    'Executed command SetRollerStepVolume in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetRollerStepVolume_Responses(
                **default_dict['SetRollerStepVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def ResetRollerStepVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.ResetRollerStepVolume_Responses:
        """
        Executes the unobservable command "Reset Roller Step Volume"
            Reset the roller step volume back to default.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ResetRollerStepVolumeSet (Reset Current Roller Step Volume Set): The roller step volume is set to default.
        """
    
        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}r000000\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'ResetRollerStepVolumeSet': silaFW_pb2.Integer(value=int(float(read))),
                }
                return_value = ParameterControlServicer_pb2.ResetRollerStepVolume_Responses(**par_dict)
                logging.debug(
                    'Executed command ResetRollerStepVolume in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.ResetRollerStepVolume_Responses(
                **default_dict['ResetRollerStepVolume_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetRollerBackSteps(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetRollerBackSteps_Responses:
        """
        Executes the unobservable command "Get Roller Back Steps"
            Get roller back-steps (0 - 100).
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentRollerBackSteps (Current Roller Back Step): Current roller back-steps (0 - 100).
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}%\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'CurrentRollerBackSteps': silaFW_pb2.Integer(value=int(read)),
                }
                return_value = ParameterControlServicer_pb2.GetRollerBackSteps_Responses(**par_dict)
                logging.debug(
                    'Executed command GetRollerBackSteps in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetRollerBackSteps_Responses(
                **default_dict['GetRollerBackSteps_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetRollerBackSteps(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetRollerBackSteps_Responses:
        """
        Executes the unobservable command "Set Roller Back Steps"
            Set the roller back-steps (1 - 100).
    
        :param request: gRPC request containing the parameters passed:
            request.CurrentRollerBackSteps (Current Roller Back Step): Current roller back-steps (0 - 100).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetRollerBackStepsSet (Set Current Roller Back Steps Set): The set roller back-steps (0 - 100).
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = str(request.CurrentRollerBackSteps.value)
        user_input = user_input.zfill(4)  # If to short, pad to length 4
        user_input = user_input[:4]  # If to long, strictly limit to 4 digits

        command = f'{self.properties.pump_address}%{user_input}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    # Todo: Implement a correct return when the code is generated again. The return should be string.
                par_dict = {
                    'SetRollerBackStepsSet': silaFW_pb2.Integer(value=0),
                }
                return_value = ParameterControlServicer_pb2.SetRollerBackSteps_Responses(**par_dict)
                logging.debug(
                    'Executed command SetRollerBackSteps in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetRollerBackSteps_Responses(
                **default_dict['SetRollerBackSteps_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SaveApplicationParameters(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SaveApplicationParameters_Responses:
        """
        Executes the unobservable command "Save Application Parameters"
            Save changes made to the application parameters in this feature.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SaveResponse (Save Response): The response of this command
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}*\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'SaveResponse': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SaveApplicationParameters_Responses(**par_dict)
                logging.debug(
                    'Executed command SaveApplicationParameters in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SaveApplicationParameters_Responses(
                **default_dict['SaveApplicationParameters_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def GetTubingDiameter(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetTubingDiameter_Responses:
        """
        Executes the unobservable command "Get Tubing Diameter"
            Get the inside diameter of the tubing in mm.
    
        :param request: gRPC request containing the parameters passed:
            request.EmptyParameter (Empty Parameter): An empty parameter data type used if no parameter is required.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentTubingDiameter (Current Tubing Diameter): Current the inside diameter of the tubing in mm.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')

        command = f'{self.properties.pump_address}+\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                    read = read.split('mm')[0]
                par_dict = {
                    'CurrentTubingDiameter': silaFW_pb2.Real(value=float(read)),
                }
                return_value = ParameterControlServicer_pb2.GetTubingDiameter_Responses(**par_dict)
                logging.debug(
                    'Executed command GetTubingDiameter in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetTubingDiameter_Responses(
                **default_dict['GetTubingDiameter_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value

    def SetTubingDiameter(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetTubingDiameter_Responses:
        """
        Executes the unobservable command "Set Tubing Inside Diameter Value"
            Set the inside diameter of the tubing in mm.
    
        :param request: gRPC request containing the parameters passed:
            request.TubingDiameter (Tubing Diameter):
            The inside diameter of the tubing in mm. (must be one of following list:
            [0.13,0.19,0.25,0.38,0.44,0.51,0.57,0.64,0.76,0.89,0.95,1.02,1.09,1.14,1.22,1.3,1.42,1.52,1.65,1.75,1.85,2.06,2.29,2.54,2.79,3.17])
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TubingDiameterSet (Tubing Diameter Set): Tubing Diameter succeeded to Set.
        """

        # initialise the return value
        return_value = None
        read = None
        self.properties.status = f'{self._get_function_name()}'
        logging.debug(f'New status is: {self.properties.status}')
        user_input = str(int(request.TubingDiameter.value*100))
        user_input = user_input.zfill(4)  # If to short, pad to length 4
        user_input = user_input[:4]  # If to long, strictly limit to 4 digits

        command = f'{self.properties.pump_address}+{user_input}\r\n'
        for i in range(0, 3, 1):
            try:
                with self.properties.acquire_timeout_lock():
                    self.properties.ser.write(str.encode(command))
                    read = str(bytes.decode(self.properties.ser.read_until(str.encode('\r'))))
                par_dict = {
                    'TubingDiameterSet': silaFW_pb2.String(value=read),
                }
                return_value = ParameterControlServicer_pb2.SetTubingDiameter_Responses(**par_dict)
                logging.debug(
                    'Executed command SetTubingDiameter in mode: {mode} with response: {response}'
                        .format(mode='Real', response=read))
                break
            except (ValueError, IndexError):
                logging.exception('Parsing of the following command response failed: {command}, {response}'
                                  .format(command=command, response=read))
                logging.warning(f'Resending the command (#{i + 1} try)')
                if i == 2:
                    return_value = None
                continue
            except ConnectionError:
                logging.exception(f'Communication failed executing the command: {command}')
                return_value = None
    
        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetTubingDiameter_Responses(
                **default_dict['SetTubingDiameter_Responses']
            )
    
        # update the server status
        self.properties.status = f'{self._get_function_name()}'
    
        return return_value
