"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Calibration Servicer*

:details: CalibrationServicer:
    Set and retrieve the control parameter values for calibration of the Reglo ICC pump.
    By Valeryia Sidarava, Institute of Biochemical Engineering, Technical University of Munich, 31.07.2019,
    Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 15.04.2020.

:file:    CalibrationServicer_real.py
:authors: Valeryia Sidarava & Lukas Bromig

:date: (creation)          2020-04-16T10:19:57.982604
:date: (last modification) 2020-04-16T10:19:57.982604

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import CalibrationServicer_pb2 as CalibrationServicer_pb2
# from .gRPC import CalibrationServicer_pb2_grpc as CalibrationServicer_pb2_grpc

# import default arguments
from .CalibrationServicer_default_arguments import default_dict

# open serial connection via usb adapter
import serial
import serial.tools.list_ports
#Not available on BeagleBone/Debian
ports = list(serial.tools.list_ports.comports(include_links = True))


# noinspection PyPep8Naming,PyUnusedLocal
class CalibrationServicerReal:
    """
    Implementation of the *Calibration Servicer* in *Real* mode
        This is a RegloICC Service
    """

    def __init__(self, ser):
        """Class initialiser"""
        
        self.ser = ser
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def StartCalibration(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.StartCalibration_Responses:
        """
        Executes the unobservable command "Start Calibration on a Channel"
            Starts calibration on a channel.

        :param request: gRPC request containing the parameters passed:
            request.Calibrate (Calibrate):
            Starts calibration on a channel.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.StartCalibrationStatus (Start Calibration Status): Calibration succeeded to start.
        """

        # initialise the return value
        return_value = None
        channel=request.Calibrate.value

        command = '%sxY\r' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("Error writing Start command")
        logging.debug('Calibration started on channel %s: %s' % (channel, read))
        return_value = CalibrationServicer_pb2.StartCalibration_Responses(StartCalibrationStatus=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.StartCalibration_Responses(
                **default_dict['StartCalibration_Responses']
            )

        return return_value


    def CancelCalibration(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.CancelCalibration_Responses:
        """
        Executes the unobservable command "Cancel Calibration on a Channel"
            Cancels calibration on a channel.

        :param request: gRPC request containing the parameters passed:
            request.Calibrate (Calibrate):
            Cancels calibration on a channel.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.CancelCalibrationStatus (Cancel Calibration Status): Calibration succeeded to cancel.
        """

        # initialise the return value
        return_value = None
        channel = request.Calibrate.value

        command = '%sxZ\r' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("Error writing Cancel command")
        logging.debug('Calibration canceled on channel %s: %s' % (channel, read))
        return_value = CalibrationServicer_pb2.CancelCalibration_Responses(CancelCalibrationStatus=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.CancelCalibration_Responses(
                **default_dict['CancelCalibration_Responses']
            )

        return return_value


    def SetTargetVolume(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetTargetVolume_Responses:
        """
        Executes the unobservable command "Set Target Volume Value"
            Set target volume to pump for calibrating in mL.

        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.TargetVolume (Target Volume):
            Set target volume to pump for calibrating in mL.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.TargetVolumeSet (Target Volume Set): Target Volume succeeded to Set.
        """

        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value_user = request.TargetVolume.value

        value = format(float(value_user), '.4E')              # formats the input to scientific notation
        value = value[0:1]+value[2:5]+value[7:8]+value[9:]    # extracts 4 digits and +-E in pump input format
        command = '%sxU%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            read = str(float(read[:4])*0.001*(10**(float(read[5:]))))  # translates answer from pump format to "human"
        except:
            print("ERROR")
            read = "Error setting target volume"

        logging.debug('Target Volume set on channel %s to: %s' % (channel, read))
        return_value = CalibrationServicer_pb2.SetTargetVolume_Responses(TargetVolumeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetTargetVolume_Responses(
                **default_dict['SetTargetVolume_Responses']
            )

        return return_value


    def SetActualVolume(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetActualVolume_Responses:
        """
        Executes the unobservable command "Set Actual Volume Value"
            Set the actual volume measured during calibration in mL.

        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.ActualVolume (Actual Volume):
            The actual volume measured during calibration in mL.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.ActualVolumeSet (Actual Volume Set): Actual Volume succeeded to Set.
        """

        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value_user = request.ActuaVolume.value

        value = format(float(value_user), '.4E')                  # formats the input to scientific notation
        value = value[0:1]+value[2:5]+value[7:8]+value[9:]        # extracts 4 digits and +-E in pump input format
        command = '%sxV%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            read=str(float(read[:4])*0.001*(10**(float(read[5:]))))  # translates answer from pump format to "human"
        except:
            print("ERROR")
            read = "Error setting actual volume"
        logging.debug('Target Volume set on channel %s to: %s'%(channel, read))
        return_value = CalibrationServicer_pb2.SetActualVolume_Responses(ActualVolumeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetActualVolume_Responses(
                **default_dict['SetActualVolume_Responses']
            )

        return return_value


    def GetTargetVolume(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetTargetVolume_Responses:
        """
        Executes the unobservable command "Get Target Volume"
            Get target volume to pump for calibrating in mL.

        :param request: gRPC request containing the parameters passed:
            request.TargetVolume (Target Volume):
            Get target volume of pump for calibrating in mL.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.CurrentTargetVolume (Current Target Volume): Current target volume to pump for calibrating in mL.
        """

        # initialise the return value
        return_value = None
        channel = request.TargetVolume.value

        command = '%sxU\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                read = None
                logging.error("Error reading current volume setting. NONE")
            else:
                read = float(read[:4])*0.001*(10**(float(read[5:])))   # translates answer from pump format to "human"
        except:
            read = "Error reading target volume setting"
        logging.debug("Current target volume setting on channel %s is: %s mL" %(channel,read))
        return_value = CalibrationServicer_pb2.GetTargetVolume_Responses(CurrentTargetVolume=silaFW_pb2.Real(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetTargetVolume_Responses(
                **default_dict['GetTargetVolume_Responses']
            )

        return return_value


    def SetDirectionCalibration(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetDirectionCalibration_Responses:
        """
        Executes the unobservable command "Set Direction Flow for Calibration"
            Set direction flow for calibration J or K using DIRECTION format.

        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.Direction (Direction):
            Set direction flow for calibration J or K using DIRECTION format.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.SetDirectionCalibrationSet (Set Direction for Calibration Set): Direction flow for Calibration
            succeeded to Set.
        """

        # initialise the return value
        return_value = None
        rotation = None
        channel = request.Channel.value
        value = request.Direction.value

        command = '%sxR%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting rotation direction"
        if value == 'J':
            rotation = 'clockwise'
        elif value == 'K':
            rotation = 'counter-clockwise'
        logging.debug('Calibration Direction on channel %s set to: %s' % (channel, rotation))
        return_value = CalibrationServicer_pb2.SetDirectionCalibration_Responses(
            SetDirectionCalibrationSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetDirectionCalibration_Responses(
                **default_dict['SetDirectionCalibration_Responses']
            )

        return return_value


    def GetDirectionCalibration(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetDirectionCalibration_Responses:
        """
        Executes the unobservable command "Get Direction Flow for Calibration"
            Get direction flow for calibration (J or K).

        :param request: gRPC request containing the parameters passed:
            request.Direction (Direction):
            Get direction flow for calibration (J or K).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.CurrentDirectionCalibration (Current Direction for Calibration): Current Direction flow for Calibration.
        """

        # initialise the return value
        return_value = None
        rotation = None
        channel = request.Direction.value

        command = '%sxR\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                read = "Error retrieving channel rotation direction. NONE"
        except:
            read = "Error retrieving channel rotation direction"
        if read == 'J':
            return_val = 'clockwise'
        elif read == 'K':
            rotation = 'counter-clockwise'
        logging.debug('Rotation direction of channel %s is: %s' % (channel, rotation))
        return_value = CalibrationServicer_pb2.GetDirectionCalibration_Responses(
            CurrentDirectionCalibration=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetDirectionCalibration_Responses(
                **default_dict['GetDirectionCalibration_Responses']
            )

        return return_value


    def SetCalibrationTime(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.SetCalibrationTime_Responses:
        """
        Executes the unobservable command "Set Calibration Time"
            Set the current calibration time using Time Type 2 format.

        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.Time (Time):
            Set the current calibration time using Time Type 2 format.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.SetCalibrationTimeSet (Set Calibration Time Set): Calibration Time succeeded to Set.
        """

        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value_user = request.Time.value

        value = int(round(float(value_user)*10, 1))            # eliminates '.'
        value = '{:0>8}'.format(value)                        # formats the input to 8 digits
        command = '%sxW%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting calibration time"
        logging.debug('Calibration time on channel %s set to %s: %s' % (channel, value_user, read))
        return_value = CalibrationServicer_pb2.SetCalibrationTime_Responses(
            SetCalibrationTimeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.SetCalibrationTime_Responses(
                **default_dict['SetCalibrationTime_Responses']
            )

        return return_value


    def GetCalibrationTime(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetCalibrationTime_Responses:
        """
        Executes the unobservable command "Get Calibration Time"
            Get the current calibration time.

        :param request: gRPC request containing the parameters passed:
            request.Time (Time):
            Get the current calibration time.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.CurrentCalibrationTime (Current Calibration Time): Current Calibration Time.
        """

        # initialise the return value
        return_value = None
        channel = request.Time.value

        command = '%sxW\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            if read is None:
                read = None
                logging.error("Error reading calibration time. NONE")
            else:
                read = int(round(float(read), 2))        # answer is in s
        except:
            read = None
            logging.error("Error getting current calibration time")
        logging.debug("Current calibration time on channel %s is: %s seconds" % (channel, read))
        return_value = CalibrationServicer_pb2.GetCalibrationTime_Responses(
            CurrentCalibrationTime=silaFW_pb2.Integer(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetCalibrationTime_Responses(
                **default_dict['GetCalibrationTime_Responses']
            )

        return return_value


    def GetRunTimeCalibration(self, request, context: grpc.ServicerContext) \
            -> CalibrationServicer_pb2.GetRunTimeCalibration_Responses:
        """
        Executes the unobservable command "Get Run Time since last Calibration"
            Get channel run time since last calibration.

        :param request: gRPC request containing the parameters passed:
            request.Time (Time):
            Get channel run time since last calibration.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information

        :returns: The return object defined for the command with the following fields:
            request.CurrentRunTimeCalibration (Current Run Time since Calibration): Channel run time since last calibration..
        """

        # initialise the return value
        return_value = None
        channel = request.Time.value

        command = '%sxX\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            if read is None:
                read = None
                logging.error("Error reading run time since last calibration. NONE")
            else:
                read = int(round(float(read), 2))   # answer is in s
        except:
            read = None
            logging.error("Error getting current run time since last calibration")
        logging.debug("Current run time since last calibration on channel %s is: %s seconds" % (channel, read))
        return_value = CalibrationServicer_pb2.GetRunTimeCalibration_Responses(
            CurrentRunTimeCalibration=silaFW_pb2.Integer(value=read))

        # fallback to default
        if return_value is None:
            return_value = CalibrationServicer_pb2.GetRunTimeCalibration_Responses(
                **default_dict['GetRunTimeCalibration_Responses']
            )

        return return_value

    
