"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Configuration Servicer*

:details: ConfigurationServicer:
    Set and retrieve the control parameter values for the configuration of the Reglo ICC pump.
    By Valeryia Sidarava, Institute of Biochemical Engineering, Technical University of Munich, 31.07.2019,
    Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 15.04.2020.

:file:    ConfigurationServicer_real.py
:authors: Valeryia Sidarava & Lukas Bromig

:date: (creation)          2020-04-16T10:19:58.098094
:date: (last modification) 2020-04-16T10:19:58.098094

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import ConfigurationServicer_pb2 as ConfigurationServicer_pb2
# from .gRPC import ConfigurationServicer_pb2_grpc as ConfigurationServicer_pb2_grpc

# import default arguments
from .ConfigurationServicer_default_arguments import default_dict

# open serial connection via usb adapter
import serial
import serial.tools.list_ports

# noinspection PyPep8Naming,PyUnusedLocal
class ConfigurationServicerReal:
    """
    Implementation of the *Configuration Servicer* in *Real* mode
        This is a RegloICC Service
    """

    def __init__(self,ser):
        """Class initialiser"""

        self.ser = ser
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def SetTubingDiameter(self, request, context: grpc.ServicerContext) \
            -> ConfigurationServicer_pb2.SetTubingDiameter_Responses:
        """
        Executes the unobservable command "Set Tubing Inside Diameter Value"
            Set the inside diameter of the tubing in mm.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.TubingDiameter (Tubing Diameter):
            The inside diameter of the tubing in mm. (must be one of following list:
            [0.13,0.19,0.25,0.38,0.44,0.51,0.57,0.64,0.76,0.89,0.95,1.02,1.09,1.14,1.22,1.3,1.42,1.52,1.65,1.75,1.85,2.06,2.29,2.54,2.79,3.17])
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.TubingDiameterSet (Tubing Diameter Set): Tubing Diameter succeeded to Set.
        """
    
        # initialise the return value
        return_value = None
        channel=request.Channel.value
        value_user=request.TubingDiameter.value
        allowed_tubing_diameters = [0.13, 0.19, 0.25, 0.38, 0.44, 0.51, 0.57, 0.64, 0.76, 0.89, 0.95, 1.02, 1.09, 1.14,
                                    1.22, 1.3, 1.42, 1.52, 1.65, 1.75, 1.85, 2.06, 2.29, 2.54, 2.79, 3.17]

        if float(value_user) not in allowed_tubing_diameters:
            logging.debug('Wrong tube ID, see manual p.16 (of 36)')

        value = '{:0>5}'.format(value_user)       # formats the input to 5 digits
        value = value[0:2]+value[3:]              # Extracts 4 digits
        command = '%s+%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting tubing diameter"
        logging.debug('Setting Tubing Diameter of channel %s to %s is: %s' % (channel, value_user, read))
        return_value = ConfigurationServicer_pb2.SetTubingDiameter_Responses(TubingDiameterSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ConfigurationServicer_pb2.SetTubingDiameter_Responses(
                **default_dict['SetTubingDiameter_Responses']
            )
    
        return return_value
    
    
    def ResetToDefault(self, request, context: grpc.ServicerContext) \
            -> ConfigurationServicer_pb2.ResetToDefault_Responses:
        """
        Executes the unobservable command "Reset To Default"
            Resets all user configurable data to default values.
    
        :param request: gRPC request containing the parameters passed:
            request.Reset (Reset):
            Resets all user configurable data to default values.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.ResetStatus (Reset Status): Reset of pump succeeded.
        """
    
        # initialise the return value
        return_value = None
        address = request.Reset.value

        command = '%s0\r' % address
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("Error writing Reset command")
        logging.debug('Resetting pump %s : %s' % (address, read))
        return_value = ConfigurationServicer_pb2.ResetToDefault_Responses(ResetStatus=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ConfigurationServicer_pb2.ResetToDefault_Responses(
                **default_dict['ResetToDefault_Responses']
            )
    
        return return_value
    
    
    def GetTubingDiameter(self, request, context: grpc.ServicerContext) \
            -> ConfigurationServicer_pb2.GetTubingDiameter_Responses:
        """
        Executes the unobservable command "Get Tubing Diameter"
            Get the inside diameter of the tubing in mm.
    
        :param request: gRPC request containing the parameters passed:
            request.CurrentTubingDiameter (Current Tubing Diameter):
            The inside diameter of the tubing in mm.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentTubingDiameter (Current Tubing Diameter): Current the inside diameter of the tubing in mm.
        """
    
        # initialise the return value
        return_value = None
        channel = request.CurrentTubingDiameter.value

        command = '%s+\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                read = None
                logging.error("Error reading tubing diameter. NONE")
            else:
                read = float(read)
        except:
            read = None
            logging.error("Error reading tubing diameter")
        logging.debug("Current tubing diameter setting is: %s" % read)
        return_value = ConfigurationServicer_pb2.GetTubingDiameter_Responses(CurrentTubingDiameter=silaFW_pb2.Real(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ConfigurationServicer_pb2.GetTubingDiameter_Responses(
                **default_dict['GetTubingDiameter_Responses']
            )
    
        return return_value
    
    
    def SetBacksteps(self, request, context: grpc.ServicerContext) \
            -> ConfigurationServicer_pb2.SetBacksteps_Responses:
        """
        Executes the unobservable command "Set Backsteps setting"
            Set the current backsteps setting using Discrete Type 2.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.BackstepsSetting (Backsteps Setting):
            Set the current backsteps setting using Discrete Type 2.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.BackstepsSettingSet (Backsteps Setting Set): Backsteps Setting succeeded to Set.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value_user = request.BackstepsSetting.value

        value = '{:0>5}'.format(value_user)             # formats the input to 5 digits
        value = value[0:2]+value[3:]                    # Extracts 4 digits
        command = '%s%%%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting current backsteps setting"
        logging.debug('Setting backsteps for channel %s to %s: %s ' % (channel, value_user, read))
        return_value = ConfigurationServicer_pb2.SetBacksteps_Responses(BackstepsSettingSet=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ConfigurationServicer_pb2.SetBacksteps_Responses(
                **default_dict['SetBacksteps_Responses']
            )
    
        return return_value
    
    
    def GetBacksteps(self, request, context: grpc.ServicerContext) \
            -> ConfigurationServicer_pb2.GetBacksteps_Responses:
        """
        Executes the unobservable command "Get Backsteps setting"
            Get the current backsteps setting.
    
        :param request: gRPC request containing the parameters passed:
            request.BackstepsSetting (Backsteps Setting):
            Get the current backsteps setting.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentBackstepsSetting (Current Backsteps Setting): Current Backsteps Setting.
        """
    
        # initialise the return value
        return_value = None
        channel = request.BackstepsSetting.value

        command = '%s%%\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                read = None
                logging.error("Error reading current backsteps setting. NONE")
        except:
            read = None
            logging.error("Error reading current backsteps setting")
        logging.debug("Current backsteps setting on channel %s is: %s" % (channel, read))
        return_value = ConfigurationServicer_pb2.GetBacksteps_Responses(CurrentBackstepsSetting=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ConfigurationServicer_pb2.GetBacksteps_Responses(
                **default_dict['GetBacksteps_Responses']
            )
    
        return return_value
