"""
________________________________________________________________________

:PROJECT: SiLA2_python

*Parameter Control Servicer*

:details: ParameterControlServicer:
    Set and retrieve information regarding the parameter settings of the Reglo ICC pump.
    By Valeryia Sidarava, Institute of Biochemical Engineering, Technical University of Munich, 31.07.2019,
    Lukas Bromig, Institute of Biochemical Engineering, Technical University of Munich, 15.04.2020.

:file:    ParameterControlServicer_real.py
:authors: Valeryia Sidarava & Lukas Bromig

:date: (creation)          2020-04-16T10:19:58.319877
:date: (last modification) 2020-04-16T10:19:58.319877

.. note:: Code generated by sila2codegenerator 0.2.0

________________________________________________________________________

**Copyright**:
  This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
  INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  For further Information see LICENSE file that comes with this distribution.
________________________________________________________________________
"""

__version__ = "1.0"

# import general packages
import logging
import time         # used for observables
import uuid         # used for observables
import grpc         # used for type hinting only

# import SiLA2 library
import sila2lib.framework.SiLAFramework_pb2 as silaFW_pb2

# import gRPC modules for this feature
from .gRPC import ParameterControlServicer_pb2 as ParameterControlServicer_pb2
# from .gRPC import ParameterControlServicer_pb2_grpc as ParameterControlServicer_pb2_grpc

# import default arguments
from .ParameterControlServicer_default_arguments import default_dict

# open serial connection via usb adapter
import serial
import serial.tools.list_ports
#Not available on BeagleBone/Debian
ports = list(serial.tools.list_ports.comports(include_links = True))


# noinspection PyPep8Naming,PyUnusedLocal
class ParameterControlServicerReal:
    """
    Implementation of the *Parameter Control Servicer* in *Real* mode
        This is a RegloICC Service
    """

    def __init__(self, ser):
        """Class initialiser"""

        self.ser = ser
        logging.debug('Started server in mode: {mode}'.format(mode='Real'))

    def _get_command_state(self, command_uuid: str) -> silaFW_pb2.ExecutionInfo:
        """
        Method to fill an ExecutionInfo message from the SiLA server for observable commands

        :param command_uuid: The uuid of the command for which to return the current state

        :return: An execution info object with the current command state
        """

        #: Enumeration of silaFW_pb2.ExecutionInfo.CommandStatus
        command_status = silaFW_pb2.ExecutionInfo.CommandStatus.waiting
        #: Real silaFW_pb2.Real(0...1)
        command_progress = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_estimated_remaining = None
        #: Duration silaFW_pb2.Duration(seconds=<seconds>, nanos=<nanos>)
        command_lifetime_of_execution = None

        # TODO: check the state of the command with the given uuid and return the correct information

        # just return a default in this example
        return silaFW_pb2.ExecutionInfo(
            commandStatus=command_status,
            progressInfo=(
                command_progress if command_progress is not None else None
            ),
            estimatedRemainingTime=(
                command_estimated_remaining if command_estimated_remaining is not None else None
            ),
            updatedLifetimeOfExecution=(
                command_lifetime_of_execution if command_lifetime_of_execution is not None else None
            )
        )

    def SetFlowRate(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetFlowRate_Responses:
        """
        Executes the unobservable command "Set Flow Rate value "
            Set the desired flow rate of the pump, mL/min
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.SetFlowRate (Set FlowRate):
            Set the desired RPM flow rate in volume/time mode, mL/min
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetFlowRateSet (Set Flow Rate Set): Set Flow Rate succeeded to Set.
        """
    
        # initialise the return value
        return_value = None
        value_user = request.SetFlowRate.value
        channel = request.Channel.value
        value = format(float(value_user), '.4E')  # formats in scientific notation
        value = value[0:1]+value[2:5]+value[7:8]+value[9:]  # extracts 4 digits and +-E for pump input format
        command = '%sf%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)

            if read != '#':
                return_val = float(read[:4])*0.001*(10**(float(read[5:])))   # translates answer from pump format to "human"
            else: return_val = read
        except:
            print("ERROR")
            read = "Error setting flow rate"
            return_val = read
        return_value = ParameterControlServicer_pb2.SetFlowRate_Responses(SetFlowRateSet=silaFW_pb2.Real(
            value=return_val))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetFlowRate_Responses(
                **default_dict['SetFlowRate_Responses']
            )
        logging.debug("\nSetting of flow rate: %s mL/min on channel %s" % (return_val, channel))
        return return_value
    
    
    def SetRPMMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetRPMMode_Responses:
        """
        Executes the unobservable command "Set RPM Mode"
            Set the current channel/pump mode to RPM mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel = request.PumpMode.value

        command = '%sL\r\n'%channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting RPM mode"

        return_value = ParameterControlServicer_pb2.SetRPMMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetRPMMode_Responses(
                **default_dict['SetRPMMode_Responses'])

        logging.debug("Setting of RPM mode on channel %s: %s " %(channel, read))
        return return_value
    
    
    def SetFlowRateMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetFlowRateMode_Responses:
        """
        Executes the unobservable command "Set Flow Rate Mode"
            Set the current channel/pump mode to Flow Rate mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel = request.PumpMode.value
        command = '%sM\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting Flow Rate mode"
        return_value=ParameterControlServicer_pb2.SetFlowRateMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetFlowRateMode_Responses(
                **default_dict['SetFlowRateMode_Responses']
            )
        logging.debug("Setting of Flow Rate mode on channel %s: %s " % (channel, read))
        return return_value
    
    
    def SetVolumeRateMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetVolumeRateMode_Responses:
        """
        Executes the unobservable command "Set Volume Rate Mode"
            Set the current channel/pump mode to Volume (at Rate) mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel=request.PumpMode.value

        command = '%sO\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting Volume at Rate mode"
        return_value=ParameterControlServicer_pb2.SetVolumeRateMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))


        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetVolumeRateMode_Responses(
                **default_dict['SetVolumeRateMode_Responses']
            )
        logging.debug("Setting of Volume at Rate mode on channel %s: %s " % (channel, read))
        return return_value
    
    
    def SetVolumeTimeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetVolumeTimeMode_Responses:
        """
        Executes the unobservable command "Set Volume Time Mode"
            Set the current channel/pump mode to Volume (over Time) mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel = request.PumpMode.value

        command = '%sG\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting Volume over Time mode"
        return_value = ParameterControlServicer_pb2.SetVolumeTimeMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))


        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetVolumeTimeMode_Responses(
                **default_dict['SetVolumeTimeMode_Responses']
            )
        logging.debug("Setting of Volume over Time mode on channel %s: %s " % (channel, read))
        return return_value
    
    
    def SetVolumePauseMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetVolumePauseMode_Responses:
        """
        Executes the unobservable command "Set Volume Pause Mode"
            Set the current channel/pump mode to Volume+Pause mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel = request.PumpMode.value

        command = '%sQ\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting Volume + Pause mode"
        return_value=ParameterControlServicer_pb2.SetVolumePauseMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetVolumePauseMode_Responses(
                **default_dict['SetVolumePauseMode_Responses']
            )
        logging.debug("Setting of Volume + Pause mode on channel %s: %s " % (channel, read))
        return return_value


    def SetTimeMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetTimeMode_Responses:
        """
        Executes the unobservable command "Set Time Mode"
            Set the current channel/pump mode to Time mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel = request.PumpMode.value

        command = '%sN\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting Time mode"
        return_value = ParameterControlServicer_pb2.SetTimeMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))


        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetTimeMode_Responses(
                **default_dict['SetTimeMode_Responses']
            )

        logging.debug("Setting of Time mode on channel %s: %s " % (channel, read))
        return return_value
    
    
    def SetTimePauseMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetTimePauseMode_Responses:
        """
        Executes the unobservable command "Set Time Pause Mode"
            Set the current channel/pump mode to Time+Pause mode.
    
        :param request: gRPC request containing the parameters passed:
            request.PumpMode (Pump Mode):
            Current channel or pump mode: L=RPM / M=Flow Rate / O=Volume (at Rate) / G=Volume (over Time) / Q=Volume+Pause / N=Time / P=Time+Pause
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.PumpModeSet (Pump Mode Set): Pump mode succeeded to Set.
        """
    
        # initialise the return value
        return_value = None

        channel = request.PumpMode.value

        command = '%sP\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting Time + Pause mode"
        return_value = ParameterControlServicer_pb2.SetTimePauseMode_Responses(PumpModeSet=silaFW_pb2.String(value=read))


        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetTimePauseMode_Responses(
                **default_dict['SetTimePauseMode_Responses']
            )

        logging.debug("Setting of Time + Pause mode on channel %s: %s " % (channel, read))
        return return_value
    
    
    def SetVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetVolume_Responses:
        """
        Executes the unobservable command "Set Volume"
            Set the current setting for volume in mL. Volume Type 2.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.Volume (Volume):
            Set the current setting for volume in mL. Volume Type 2.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.VolumeSet (Volume Set): Volume succeeded to Set
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        volume = request.Volume.value

        value = format(float(volume), '.4E')                      # formats in scientific notation
        value = value[0:1]+value[2:5]+value[7:8]+value[9:]        # extracts 4 digits and +-E for pump input format
        command = '%sv%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read != '#':
                return_val = str(float(read[:4])*0.001*(10**(float(read[5:]))))   # translates answer from pump format to "human"
            else:
                return_val = read
        except:
            print("ERROR")
            read = "Error setting volume"
        return_value = ParameterControlServicer_pb2.SetVolume_Responses(VolumeSet=silaFW_pb2.String(value=return_val))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetVolume_Responses(
                **default_dict['SetVolume_Responses'])

        logging.debug("\nSetting of volume: %s mL on channel %s" % (return_val, channel))
        return return_value
    
    
    def GetFlowRate(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetFlowRate_Responses:
        """
        Executes the unobservable command "Get Flow Rate"
            Get current flow rate (mL/min).
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFlowRate (Current Flow Rate): Current flow rate of the channel.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%sf\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                return_val = "Error reading current flow rate. NONE"
            else:
                return_val = read
                return_val = float(return_val[:4])*0.001*(10**(float(return_val[5:])))    #translates answer from pump format to "human"
        except:
            read = "Error reading current flow rate"

        return_value = ParameterControlServicer_pb2.GetFlowRate_Responses(CurrentFlowRate=silaFW_pb2.Real(value=return_val))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetFlowRate_Responses(
                **default_dict['GetFlowRate_Responses'])

        logging.debug("Current flow rate is: %s mL/min" % return_val)
        return return_value
    
    
    def GetMode(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetMode_Responses:
        """
        Executes the unobservable command "Get Mode"
            Get the current channel or pump mode
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPumpMode (Current Pump Mode): Current channel or pump mode is retrieved.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        command = '%sxM\r\n'%channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                read = "Error reading current channel/pump mode. NONE"
        except:
            read = "Error reading current channel/pump mode"
        return_value = ParameterControlServicer_pb2.GetMode_Responses(CurrentPumpMode=silaFW_pb2.String(value=read))


        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetMode_Responses(
                **default_dict['GetMode_Responses']
            )
        logging.debug("Current channel/pump mode is: %s " % read)
        return return_value

    def GetVolume(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetVolume_Responses:
        """
        Executes the unobservable command "Get Volume"
            Get the current setting for volume in mL/min.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentVolume (Current Volume): Current setting for volume in mL/min.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%sv\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
            if read is None:
                read = "Error reading current volume setting. NONE"
            elif read !='#':
                read = float(read[:4])*0.001*(10**(float(read[5:])))    #translates answer from pump format to "human"
        except:
            read = 0
            logging.debug("Error reading current volume setting" % read)
        logging.debug("Current volume setting is: %s mL" % read)
        return_value = ParameterControlServicer_pb2.GetVolume_Responses(CurrentVolume=silaFW_pb2.Real(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetVolume_Responses(
                **default_dict['GetVolume_Responses'])

        return return_value
    
    
    def GetMaximumFlowRate(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetMaximumFlowRate_Responses:
        """
        Executes the unobservable command "Get Maximum Flow Rate"
            Get maximum flow rate achievable with current settings, mL/min.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.MaximumFlowRate (Maximum Flow Rate): Maximum Flow Rate achievable with current settings, mL/min.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%s?\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = float(read)
            if read is None:
                read = 0
                logging.debug("Error reading maximum flow rate. NONE: %s" % read)

        except:
            read = 0
            logging.debug("Error getting maximum flow rate: %s" % read)

        logging.debug("Maximum flow rate is: %s" % read)
        return_value = ParameterControlServicer_pb2.GetMaximumFlowRate_Responses(MaximumFlowRate=silaFW_pb2.Real(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetMaximumFlowRate_Responses(
                **default_dict['GetMaximumFlowRate_Responses']
            )
    
        return return_value
    
    
    def GetMaximumFlowRateWithCalibration(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetMaximumFlowRateWithCalibration_Responses:
        """
        Executes the unobservable command "Get Maximum Flow Rate using Calibration"
            Get maximum flow rate achievable with current settings using calibration, mL/min.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.MaximumFlowRateWithCalibration (Maximum Flow Rate using Calibration): Maximum Flow Rate achievable with current settings using calibration, mL/min.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%s!\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = read.split(' ml')[0]
            read = float(read)
            print('HERE read', read)
            if read is None:
                read = 0
                logging.debug("Error reading maximum flow rate. NONE: %s" % read)
            else:
                read = float(read)
        except:
            read = 0
            logging.debug("Error getting maximum flow rate: %s" % read)

        logging.debug("Maximum flow rate with current settings using calibration is: %s" % read)
        return_value = ParameterControlServicer_pb2.GetMaximumFlowRateWithCalibration_Responses(
                                 MaximumFlowRateWithCalibration=silaFW_pb2.Real(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetMaximumFlowRateWithCalibration_Responses(
                **default_dict['GetMaximumFlowRateWithCalibration_Responses']
            )
    
        return return_value
    
    
    def GetSpeedSettingRPM(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetSpeedSettingRPM_Responses:
        """
        Executes the unobservable command "Get Speed Setting in RPM"
            Get the current speed setting in RPM.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSpeedSetting (Current Speed Setting): The current speed setting in RPM.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%sS\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = float(read)

            if read is None:
                read = 0
                logging.debug("Error reading maximum flow rate. NONE: %s" % read)

        except:
            read = 0
            logging.debug("Error getting maximum flow rate: %s" % read)

        logging.debug("Current speed setting is: %s RPM" % read)
        return_value = ParameterControlServicer_pb2.GetSpeedSettingRPM_Responses(CurrentSpeedSetting=silaFW_pb2.Real(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetSpeedSettingRPM_Responses(
                **default_dict['GetSpeedSettingRPM_Responses']
            )
    
        return return_value
    
    
    def SetSpeedSettingRPM(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetSpeedSettingRPM_Responses:
        """
        Executes the unobservable command "Set Speed Setting in RPM"
            Set speed setting in RPM (RPM Mode flow rate setting (0.01 RPM) Discrete Type 3).
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.Speed (Speed):
            Speed setting in RPM (RPM Mode flow rate setting (0.01 RPM) Discrete Type 3)
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentSpeedSetting (Current Speed Setting): RPM Mode flow rate setting.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value = request.Speed.value

        value = int(float(value)*100) # elimination of '.'
        value = '{:0>6}'.format(value)   # formats the input to 6 digits
        command = '%sS%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting volume"
        return_val = request.Speed.value    # Success check
        logging.debug("Set Speed Setting in RPM on channel %s to %s RPM: %s " % (channel, return_val, read))
        return_value = ParameterControlServicer_pb2.SetSpeedSettingRPM_Responses(CurrentSpeedSetting=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetSpeedSettingRPM_Responses(
                **default_dict['SetSpeedSettingRPM_Responses']
            )
    
        return return_value
    
    
    def SetCurrentRunTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetCurrentRunTime_Responses:
        """
        Executes the unobservable command "Set Current Run Time"
            Set current pump run time using Time Type 2 format (0 to 999h or 0 to 3596400s).
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.RunTime (Run Time):
            Set current pump run time using Time Type 2 format (0 to 999h or 0 to 3596400s).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetCurrentRunTimeSet (Set Current Run Time): Current Run Time succeeded to set.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value = request.RunTime.value

        value = int(float(value)*10)              # elimination of '.'
        value = '{:0>8}'.format(value)            # formats the input to 8 digits
        command = '%sxT%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting run time"
        return_val = request.RunTime.value        # Success check
        logging.debug("Setting of Current Run Time  on channel %s to %s seconds: %s " % (channel, return_val, read))
        return_value = ParameterControlServicer_pb2.SetCurrentRunTime_Responses(SetCurrentRunTimeSet=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetCurrentRunTime_Responses(
                **default_dict['SetCurrentRunTime_Responses']
            )
    
        return return_value
    
    
    def GetCurrentRunTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetCurrentRunTime_Responses:
        """
        Executes the unobservable command "Get Current Run Time"
            Get current pump run time using Time Type 2 format.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.GetCurrentRunTime (Get Current Run Time): Current Run Time of the pump.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%sxT\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            if read is None:
                read = 0
                logging.error("Error reading run time. NONE")
            else:
                read = int(read)    # answer is in s
        except:
            read = 0
            logging.error("Error getting current run time")
        logging.debug("Current run time on channel %s is: %s seconds" % (channel, read))
        return_value = ParameterControlServicer_pb2.GetCurrentRunTime_Responses(GetCurrentRunTime=silaFW_pb2.Integer(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetCurrentRunTime_Responses(
                **default_dict['GetCurrentRunTime_Responses']
            )
    
        return return_value
    
    def SetPumpingPauseTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetPumpingPauseTime_Responses:
        """
        Executes the unobservable command "Set Pumping Pause Time"
            Set pumping pause time using Time Type 2 format (0 to 999h or 0 to 3596400s).
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.PumpingPauseTime (Pumping Pause Time):
            Set pumping pause time using Time Type 2 format (0 to 999h or 0 to 3596400s).
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetPumpingPauseTimeSet (Set Pumping Pause Time Time): Pumping pause time succeeded to set.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value_user = request.PumpingPauseTime.value

        value = int(float(value_user)*10)       # elimination of '.'
        value = '{:0>8}'.format(value)          # formats the input to 8 digits
        command = '%sxP%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting pause time"
        logging.debug("Setting of Pumping Pause Time  on channel %s to %s seconds: %s " % (channel, value_user, read))
        return_value = ParameterControlServicer_pb2.SetPumpingPauseTime_Responses(
            SetPumpingPauseTimeSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetPumpingPauseTime_Responses(
                **default_dict['SetPumpingPauseTime_Responses'])

        return return_value
    
    
    def GetPumpingPauseTime(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetPumpingPauseTime_Responses:
        """
        Executes the unobservable command "Get Pumping Pause Time"
            Get pumping pause time.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Channel
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentPumpingPauseTime (Current Pumping Pause Time): Current Pumping Pause Time.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%sxP\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            if read is None:
                read = 0
                logging.error("Error reading pause time. NONE")
            else:
                read = float(read) * 0.1   # answer is in 0.1 s
        except:
            read = 0
            logging.error("Error getting current pause time")
        logging.debug("Current pause time on channel %s is: %s seconds" % (channel, read))
        return_value = ParameterControlServicer_pb2.GetPumpingPauseTime_Responses(
            CurrentPumpingPauseTime=silaFW_pb2.Real(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetPumpingPauseTime_Responses(
                **default_dict['GetPumpingPauseTime_Responses']
            )
    
        return return_value
    
    
    def GetCycleCount(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetCycleCount_Responses:
        """
        Executes the unobservable command "Get Cycle Count"
            Get pump cycle count.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Channel
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentCycleCount (Current Cycle Count): Current Cycle Count.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%s"\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            if read is None:
                read = 0
                logging.error("Error reading cycle count. NONE")
            else:
                read = float(read) * 0.1    # answer is in 0.1s
        except:
            read = 0
            logging.error("Error getting cycle count")

        logging.debug("Current cycle count on channel %s is: %s " % (channel, read))
        return_value = ParameterControlServicer_pb2.GetCycleCount_Responses(
            CurrentCycleCount=silaFW_pb2.Real(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetCycleCount_Responses(
                **default_dict['GetCycleCount_Responses']
            )
    
        return return_value
    
    
    def SetCycleCount(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetCycleCount_Responses:
        """
        Executes the unobservable command "Set Cycle Count"
            Set pump cycle count (Discrete Type 2).
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.CycleCount (Cycle Count):
            Set pump cycle count
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetCycleCountSet (Set Cycle Count): Cycle Count succeeded to set.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        value = request.CycleCount.value

        value = '{:0>4}'.format(value)        #formats the input to 4 digits
        command = '%s"%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting pause time"
        logging.debug('Cycle count of %s set on channel %s: %s' % (value, channel, read))
        return_value = ParameterControlServicer_pb2.SetCycleCount_Responses(
            SetCycleCountSet=silaFW_pb2.String(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetCycleCount_Responses(
                **default_dict['SetCycleCount_Responses']
            )
    
        return return_value
    
    
    def GetDispenseTimeMlMin(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetDispenseTimeMlMin_Responses:
        """
        Executes the unobservable command "Get Dispense Time at mL/min"
            Get time to dispense at a given volume at a given mL/min flow rate.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.Volume (Volume):
            Gets the volume, which must be dispensed
            request.FlowRate (FlowRate):
            Gets the flow rate, with which the volume must be dispensed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDispenseTime (CurrentDispenseTime): CurrentDispenseTime at a given volume at a given mL/min flow rate.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        volume_user = format(float(request.Volume.value), '.3E')
        flowrate_user = format(float(request.FlowRate.value), '.3E')

        volume = volume_user[0:1]+volume_user[2:5]+volume_user[6:7]+volume_user[8:9]  # extracts 4 digits and +-E into the pump input format
        flowrate = flowrate_user[0:1]+flowrate_user[2:5]+flowrate_user[6:7]+flowrate_user[8:]  # extracts 4 digits and +-E into the pump input format
        command = '%sxv%s|%s\r\n' % (channel, volume, flowrate)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)

            if read is None:
                read = 0
                logging.error("Error reading time to dispense. NONE")
            else:
                read = float(read) * 0.1  # answer in 0.1s

        except:
            read = 0
            logging.error("Error getting time to dispense")
        logging.debug("Current time to dispense %s mL at rate of %s mL/min on channel %s is: %s seconds" % (
            volume_user, flowrate_user, channel, read))
        return_value = ParameterControlServicer_pb2.GetDispenseTimeMlMin_Responses(CurrentDispenseTime=silaFW_pb2.Real(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetDispenseTimeMlMin_Responses(
                **default_dict['GetDispenseTimeMlMin_Responses']
            )
    
        return return_value
    
    
    def GetDispenseTimeRPM(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetDispenseTimeRPM_Responses:
        """
        Executes the unobservable command "Get Dispense Time at RPM"
            Get time to dispense at a given volume at a given RPM.
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Channel
            request.Volume (Volume):
            Gets the volume, which must be dispensed
            request.FlowRate (FlowRate):
            Gets the flow rate, with which the volume must be dispensed
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentDispenseTime (CurrentDispenseTime): CurrentDispenseTime at a given volume at a given RPM.
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        volume_user = format(float(request.Volume.value), '.3E')
        flowrate_user = request.FlowRate.value

        volume = volume_user[0:1]+volume_user[2:5]+volume_user[6:7]+volume_user[8:9]    # extracts 4 digits and +-E into the pump input format
        flowrate = '{:0>6}'.format(flowrate_user)   # extracts 6 digits into the pump input format
        command = '%sxw%s|%s\r\n' % (channel, volume, flowrate)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)

            if read is None:
                read = 0
                logging.error("Error reading time to dispense. NONE")
            else:
                read = float(format(float(read)*0.1, '.2f'))  # translates answer from pump format to "human"

        except:
            read = 0
            logging.error("Error getting time to dispense")

        logging.debug("Current time to dispense is %s mL at rate of %s RPM on channel %s: %s seconds" % (
            volume_user, flowrate_user, channel, read))
        return_value = ParameterControlServicer_pb2.GetDispenseTimeRPM_Responses(
            CurrentDispenseTime=silaFW_pb2.Real(value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetDispenseTimeRPM_Responses(
                **default_dict['GetDispenseTimeRPM_Responses']
            )
    
        return return_value
    
    
    def GetFlowRateAtModes(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.GetFlowRateAtModes_Responses:
        """
        Executes the unobservable command "Get Flow Rate when mode is not RPM or Flow Rate"
            Get flow rate from RPM (S) or flow rate (f) when mode is not RPM or Flow Rate.: 0=RPM, 1=Flow Rate
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Channel
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.CurrentFlowRate (Current Flow Rate): CurrentDispenseTime at a given volume at a given RPM. : 0=RPM,
            1=Flow Rate
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value

        command = '%sxf\r\n' % channel
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = bool(int(read))
            if read is None:
                read = False
                logging.debug("Error reading current flow rate. NONE")
            else:
                read = bool(read)
        except:
            read = None
            logging.error("Error reading current flow rate")
        if read is False:
            return_val = 'RPM'
        elif read is True:
            return_val = 'Flow Rate'
        logging.debug("Current flow rate is from: %s mode " % return_val)
        return_value = ParameterControlServicer_pb2.GetFlowRateAtModes_Responses(CurrentFlowRate=silaFW_pb2.Boolean(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.GetFlowRateAtModes_Responses(
                **default_dict['GetFlowRateAtModes_Responses']
            )
    
        return return_value
    
    
    def SetFlowRateAtModes(self, request, context: grpc.ServicerContext) \
            -> ParameterControlServicer_pb2.SetFlowRateAtModes_Responses:
        """
        Executes the unobservable command "Set Flow Rate when mode is not RPM or Flow Rate"
            Set RPM flow rate not in RPM or Flow Rate mode. (Discrete Type 3)
    
        :param request: gRPC request containing the parameters passed:
            request.Channel (Channel):
            Gets the channel, on which the command will be executed
            request.FlowRate (Flow Rate):
            Set RPM flow rate not in RPM or Flow Rate mode.
        :param context: gRPC :class:`~grpc.ServicerContext` object providing gRPC-specific information
    
        :returns: The return object defined for the command with the following fields:
            request.SetFlowRateSet (Set Flow Rate): RPM flow rate not in RPM or Flow Rate mode succeeded to set
        """
    
        # initialise the return value
        return_value = None
        channel = request.Channel.value
        flowrate = request.FlowRate.value
        value = int(round((float(flowrate)*100), 1))     # elimination of '.'
        value = '{:0>6}'.format(value)                   # formatis the input to 6 digits
        command = '%sxf%s\r\n' % (channel, value)
        command_as_bytes = str.encode(command)
        try:
            self.ser.write(command_as_bytes)
            read = self.ser.readline().rstrip()
            read = bytes.decode(read)
            read = str(read)
        except:
            print("ERROR")
            read = "Error setting flow rate"
        logging.debug("Set RPM Flow Rate not in RPM or flow rate mode to %s: %s" % (flowrate, read))
        return_value = ParameterControlServicer_pb2.SetFlowRateAtModes_Responses(SetFlowRateSet=silaFW_pb2.String(
            value=read))

        # fallback to default
        if return_value is None:
            return_value = ParameterControlServicer_pb2.SetFlowRateAtModes_Responses(
                **default_dict['SetFlowRateAtModes_Responses']
            )
    
        return return_value
