import astropy.io.fits as fits
import psutil
import logging
import numpy as np
import collections
import os
import inspect
import pickle

LOG = logging.getLogger('pipeline_parallel.utils')


def update_dict(d, u):
    """
    Recursively merge or update dict-like objects.
    i.e, change a value to a key that already exists or
    add a (key, value) that did not previously existed

    source: https://stackoverflow.com/questions/3232943/update-value-of-a-nested-dictionary-of-varying-depth

    :param dict d: Original dictionnary
    :param dict u: dictionnary of updates to apply to 'd'
    :return dict d: Return updated version of 'd'
    """

    for k, v in u.items():
        if isinstance(v, collections.Mapping):
            d[k] = update_dict(d.get(k, {}), v)
        else:
            d[k] = v
    return d


def init_log(log="pipeline.log", stdout_loglevel="INFO", file_loglevel="DEBUG", extra_config=None):
    """

    :param str log: filename where to store logs. By default "pipeline.log"
    :param str stdout_loglevel: log level for standard output (ERROR, WARNING, INFO, DEBUG)
    :param str file_loglevel: log level for log file (ERROR, WARNING, INFO, DEBUG)
    :param dict extra_config: [optional] Set of extra properties to be added to the dict_config for logging
    :return:
    :rtype:
    """

    import logging.config

    log_config = {
        "version": 1,
        "formatters":
            {
                "form01":
                    {
                        "format": "%(asctime)s %(levelname)-8s %(message)s",
                        "datefmt": "%H:%M:%S"
                    },
                "form02":
                    {
                        "format": "%(asctime)s [%(processName)s/%(name)s] %(levelname)s - %(message)s",
                        "datefmt": "%H:%M:%S"
                    },
            },
        "handlers":
            {
                "console":
                    {
                        "class": "logging.StreamHandler",
                        "formatter": "form01",
                        "level": stdout_loglevel,
                        "stream": "ext://sys.stdout",
                    },
                "file":
                    {
                        "class": "logging.FileHandler",
                        "formatter": "form02",
                        "level": file_loglevel,
                        "filename": log,
                        "mode": "w",  # Overwrite file if it exists
                    },
            },
        "loggers":
            {
                "":
                    {
                        "level": "NOTSET",
                        "handlers": ["console", "file"],
                    },
            },
        "disable_existing_loggers": False,
    }

    if extra_config is not None:
        log_config = update_dict(log_config, extra_config)

    logging.config.dictConfig(log_config)


def get_jwst_miri_expected_ram(filenames):
    """
    Based on a formulae (generated by scripts/fit_trends.py
    on data obtained on sample level 1B from MIRISim)

    We assume that when NINTS > 1, the RAM
    value for NINTS=1 is an upper limit, which was true for our tests

    Tests were done for FULL array. A correcting factor might be
    needed for downscale in case of a sub-array

    :param filenames: list of FITS args (and path)
    :type filenames: list(str)

    :return: list of expected RAM consumption for each file
    :rtype: list(float)
    """

    rams = []
    for filename in filenames:
        with fits.open(filename) as hdulist:
            header = hdulist[0].header

        if header["INSTRUME"] != "MIRI":
            raise ValueError("Error: Only works for MIRI")

        frames = header['NGROUPS']
        integrations = header['NINTS']

        # Reference value for FULL ARRAY
        ram = 0.05 * frames * integrations + 1.7

        # Correct for detector size (compared to FULL ARRAY)
        corr_factor = (header["SUBSIZE1"] * header["SUBSIZE2"]) / (1032. * 1024.)

        ram *= corr_factor

        rams.append(ram)

    return rams


def get_process_ram(process):
    """
    Given a multiprocess.Process object, will return the RAM used in GB

    :param multiprocess.Process process: A running process

    :return float: The Current RAM used by that process in GB
    """
    try:
        p = psutil.Process(process.pid)
    except psutil._exceptions.NoSuchProcess:
        LOG.warning("Unable to retrieve memory info from {}, process not found".format(process.name))
        return np.nan

    return p.memory_info().rss / 1024. ** 3


def force_single_thread():
    """
    By Default, Numpy and Scipy will use multi-threading furiously
    without even checking if other processes are present (i.e 24 cores means 24 threads for Numpy)

    This function force Numpy to run in a single thread. It must be used before any package attempt to import Numpy.

    """

    # Prevent multi-threads to avoid exploding the context switches
    os.environ["NUMEXPR_NUM_THREADS"] = '1'
    os.environ["OMP_NUM_THREADS"] = '1'

    # Better than setting MKL_NUM_THREADS=1 (source: https://software.intel.com/en-us/node/528380)
    os.environ["MKL_THREADING_LAYER"] = "sequential"


def strtime(sec):
    """
    Format a time interval in seconds.
    Return a string like:
    42 days 15h25m54s

    Note: The number of days is omitted if it's 0

    Return "N/A" if not a float/int or NaN

    :param float sec: Time interval in seconds
    :return str: Formatted string to display this time nicely
    """

    if not isinstance(sec, (int, float)) or np.isnan(sec):
        return "N/A"

    (d, r) = divmod(sec, 86400)
    (h, r) = divmod(r, 3600)
    (m, s) = divmod(r, 60)

    if d:
        string = "{:.0f} days ".format(d)
    else:
        string = ""

    string += "{:02.0f}h{:02.0f}m{:02.0f}s".format(h, m, s)

    return string


def generate_example_script(scriptname="example_pool.py", overwrite=False):
    """
    Create a script named by default "example_pool.py" that show the bare minimum needed to launch a Pool

    :param str scriptname: (optional) named of the file where the example will be written
    :param bool overwrite: (optional) If True, will force overwrite a file if it exists. By default, False

    """

    EXAMPLE = """
    #!/usr/bin/env python
    #  -*- coding: utf-8 -*-
    
    import pipeline_parallel
    
    # Must be before All critical package imports, especially including Numpy
    pipeline_parallel.force_single_thread()
    
    import time
    import logging
    import os
    
    LOG = logging.getLogger("main")
    
    pipeline_parallel.init_log(log="pipeline.log", stdout_loglevel="INFO", file_loglevel="DEBUG")
    
    def my_function(arg):
    
        print("arg")
        
    
    def my_pool():
        arg_pools = ["a", "b", "c", "d", "e", "f"]
        
        # Estimate RAM needed by each run
        ram = [1.] * len(arg_pools)
    
        lvl1s = pipeline_parallel.ArgList(args=arg_pools, ram=ram)
    
        pool = pipeline_parallel.ProcessPool(func=my_function, params=lvl1s)
        pool.run()
    
    
    # Launch the pool
    my_pool()
    
    """

    # Get rid of ugly extra indent (needed because the code needs to be pretty too!
    EXAMPLE = inspect.cleandoc(EXAMPLE)

    if overwrite:
        filemode = "w"
    else:
        filemode = "x"

    obj = open(scriptname, filemode)
    obj.write(EXAMPLE)
    obj.close()


def correct_restart(folder):
    """
    Up to v1.2.2, there was a bug in the pool.restart file. This function is here to check for this bug and correct the
    file, to ensure you can restart your pool, even if the file was corrupted initially.

    If you apply this on a perfectly fine file, nothing will happen and the file will not be overwritten

    In case an unexpected problem occurs after problems are detected in the file, the original one is
    renamed pool.restart.check.

    :param str folder: Folder where the pool.restart file is supposed to be.
    """
    filename = os.path.join(folder, "pool.restart")
    obj = open(filename, "rb")
    func_name, args, rams = pickle.load(obj)
    obj.close()

    nb_items = len(args)

    start_id = next((i for i in range(nb_items) if not isinstance(args[i], tuple)), None)

    if start_id is not None:
        nb_wrong = nb_items - start_id
        print(f"Folder: {folder} ; nb_wrong = {nb_wrong}")

        # Repair args and rams list
        tmp = args[start_id:]
        args[start_id:] = rams[start_id:]
        rams[start_id:] = tmp

        # Copy previous file just in case
        bkp_filename = f"{filename}.check"
        os.rename(filename, bkp_filename)

        # Write pickle file
        out_filename = f"{filename}"
        with open(out_filename, 'wb') as obj:
            pickle.dump((func_name, args, rams), obj, protocol=pickle.HIGHEST_PROTOCOL)
    else:
        print("No problems")
