# -*- coding: utf-8 -*-
"""MPT.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/15Yxgi-Htb8GDHhRcZpQyhNO2b6XeFMHZ
"""




import numpy as np
import datetime as dt
import yfinance as yf
import scipy.optimize as sc
import pandas as pd
import plotly.graph_objects as go

#Importing Data from YFin
def getData(stocks, start, end):
  stockData = yf.download(stocks, start = start, end = end, adjusted = True)
  returns = stockData['Adj Close'].pct_change().dropna()
  meanReturns = returns.mean()
  covMatrix = returns.cov()
  return meanReturns, covMatrix

#Defining Portfolio Performance 
def portfolioPerformance(weights, meanReturns, covMatrix):
  returns = np.sum(meanReturns * weights) * 252   #No. of Trading Day
  std = np.sqrt(np.dot(weights.T, np.dot(covMatrix, weights))) * np.sqrt(252)    
  #Converting equation above to matrix form. Variance is add-able and we use 252 as it is the number of trading days.
  return returns, std


def negativeSR(weights, meanReturns, covMatrix, riskFreeRate = 0):
  pReturns, pStd = portfolioPerformance(weights, meanReturns, covMatrix)
  return -(pReturns - riskFreeRate)/pStd  
  #Returning NSharpe Ratio as -(Excess Return over Market/Risk Taken)

def maxSR(meanReturns, covMatrix, riskFreeRate = 0, constraintSet = (0,1)):
  # Minimise the NegativeSR by altering the weights of the portfolio
  numAssets = len(meanReturns)
  args = (meanReturns, covMatrix, riskFreeRate)
  constraints = ({'type':'eq', 'fun': lambda x : np.sum(x) - 1})
  bound = constraintSet
  bounds = list(bound for asset in range(numAssets))
  result = sc.minimize(negativeSR, numAssets * [1./ numAssets], args = args, method = 'SLSQP', bounds = bounds, constraints = constraints)

  return result

def portfolioVariance(weights, meanReturns, covMatrix):
  return portfolioPerformance(weights, meanReturns, covMatrix)[1]

def portfolioReturn(weights, meanReturns, covMatrix):
  return portfolioPerformance(weights, meanReturns, covMatrix)[0]

def minimizeVariance(meanReturns, covMatrix, riskFreeRate = 0, constraintSet = (0,1)):
  #Minimise the portfolio variance by altering the weights/allocation of assets in the portfolio
  numAssets = len(meanReturns)
  args = (meanReturns, covMatrix)
  constraints = ({'type':'eq', 'fun': lambda x : np.sum(x) - 1})
  bound = constraintSet
  bounds = list(bound for asset in range(numAssets))
  result = sc.minimize(portfolioVariance, numAssets * [1./ numAssets], args = args, method = 'SLSQP', bounds = bounds, constraints = constraints)

  return result

def efficientOpt(meanReturns, covMatrix, returnTarget, constraintSet = (0,1)):
  'For each returnTarget, we want to optimise the portfolio for minimum variance.'
  numAssets = len(meanReturns)
  args = (meanReturns, covMatrix)
  constraints = ({'type':'eq', 'fun':lambda x: portfolioReturn(x, meanReturns, covMatrix)- returnTarget},{'type':'eq', 'fun': lambda x : np.sum(x) - 1})  #Can be an inequality(>=)
  bound = constraintSet
  bounds = tuple(bound for asset in range(numAssets))
  effOpt = sc.minimize(portfolioVariance, numAssets * [1./numAssets], args = args, constraints = constraints, method = 'SLSQP', bounds = bounds)

  return effOpt

stockList = ['CBA', 'BHP', 'TLS']
stocks = [stock + '.AX' for stock in stockList]

endDate = dt.datetime.now()
startDate = endDate - dt.timedelta(days = 365)

weights = np.array([0.33, 0, 0.67])

meanReturns, covMatrix = getData(stocks, start = startDate, end = endDate)
returns, std = portfolioPerformance(weights, meanReturns, covMatrix)

MaxSRResult = maxSR(meanReturns, covMatrix)
MaxSR, MaxSRWeights = MaxSRResult['fun'], MaxSRResult['x']
print(MaxSR, MaxSRWeights)

MinVarResult = minimizeVariance(meanReturns, covMatrix)
minVar, minVarWeights = MinVarResult['fun'], MinVarResult['x']
print(minVar, minVarWeights)

def calculatedResults(meanReturns, covMatrix, riskFreeRate = 0, constraintSet = (0,1)):
  '''Function to read Mean, CovMatrix and other Financial Information
  Output Maximum Sharpe Ratio, Min Volatility and Efficient Frontier'''

  #For Maximum Sharpe Ratio
  maxSR_Portfolio = maxSR(meanReturns,covMatrix)
  maxSR_Returns, maxSR_std = portfolioPerformance(maxSR_Portfolio['x'], meanReturns, covMatrix)
  maxSR_allocation = pd.DataFrame(maxSR_Portfolio['x'], index = meanReturns.index, columns = ['Allocation'])
  maxSR_allocation.Allocation = [round(i*100,0) for i in maxSR_allocation.Allocation]

  #For Min Volatitlity Portfolio
  minVol_Portfolio = minimizeVariance(meanReturns,covMatrix)
  minVol_Returns, minVol_std = portfolioPerformance(minVol_Portfolio['x'], meanReturns, covMatrix)
  minVol_allocation = pd.DataFrame(minVol_Portfolio['x'], index = meanReturns.index, columns = ['Allocation'])
  minVol_allocation.Allocation = [round(i*100,0) for i in minVol_allocation.Allocation]

  
  #Efficient Frontier
  efficientList = []
  targetReturns = np.linspace(minVol_Returns, maxSR_Returns, 100)
  for target in targetReturns:
    efficientList.append(efficientOpt(meanReturns, covMatrix, target)['fun'])

  maxSR_Returns, maxSR_std = round(maxSR_Returns * 100,2), round(maxSR_std*100,2)
  minVol_Returns, minVol_std = round(minVol_Returns * 100,2), round(minVol_std*100,2)

  return maxSR_Returns, maxSR_std, maxSR_allocation, minVol_Returns, minVol_std, minVol_allocation, efficientList, targetReturns

print(calculatedResults(meanReturns, covMatrix))

def EF_graph(meanReturns, covMatrix, riskFreeRate = 0, constraintSet = (0,1)):
  'Return a graph plotting the Min. Volatility, MaxSR and EfficientFrontier'
  maxSR_Returns, maxSR_std, maxSR_allocation, minVol_Returns, minVol_std, minVol_allocation, efficientList, targetReturns = calculatedResults(meanReturns, covMatrix)
  
  #Max SR
  MaxSharpeRatio = go.Scatter(
      name = 'Maximum Sharpe Ratio',
      mode = 'markers',
      x = [maxSR_std],
      y = [maxSR_Returns],
      marker = dict(color = 'red', size = 14, line = dict(width = 3, color = 'black'))
  )

  #Min Vol.   
  MinVol = go.Scatter(
      name = 'Minimum Volatility',
      mode = 'markers',
      x = [minVol_std],
      y = [minVol_Returns],
      marker = dict(color = 'green', size = 14, line = dict(width = 3, color = 'black'))   
  )

  #Efficient Frontier   
  EF_curve = go.Scatter(
      name = 'Efficent Frontier',
      mode = 'lines',
      x = [round(ef_std*100, 2) for ef_std in efficientList],
      y = [round(target*100, 2) for target in targetReturns],
      line = dict(color = 'blue', width = 4, dash = 'dashdot')   
  )

  data = [MaxSharpeRatio, MinVol, EF_curve]

  layout = go.Layout(
      title = 'Portfolio Optimization with Efficient Frontier',
      yaxis = dict(title = 'Annualised Return(%)'),
      xaxis = dict(title = 'Annualised Volatility(%)'),
      width = 800,
      height = 600,      
      showlegend = True,
      legend = dict(
          x = 0.75, y = 0, traceorder = 'normal',
          bgcolor = '#E2E2E2',
          bordercolor = 'black',
          borderwidth = 2),
          )

  fig = go.Figure(data = data,layout = layout)
  return fig.show()
  


EF_graph(meanReturns, covMatrix)

