"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const PACKAGE_LINKS = env_lambda_shared_1.requireEnv('PACKAGE_LINKS');
    const PACKAGE_TAGS = env_lambda_shared_1.requireEnv('PACKAGE_TAGS');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const assembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            constructFramework = detectConstructFramework(assembly);
            const { license, name, version } = assembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        // Add custom links content to metdata for display on the frontend
        const allowedLinks = JSON.parse(PACKAGE_LINKS);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        // Add computed tags to metadata
        const packageTagsConfig = JSON.parse(PACKAGE_TAGS);
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a, _b;
    let name;
    let majorVersion;
    let majorVersionAmbiguous = false;
    for (const depName of Object.keys((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
        if (depName.startsWith('@aws-cdk/')) {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning undefined...
                return undefined;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (depName.startsWith('@cdktf/')) {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning undefined...
                return undefined;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (depName === 'cdk8s' || depName === 'cdk8s-plus') {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning undefined...
                return undefined;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            continue;
        }
        const depVersion = (_b = assembly.dependencies) === null || _b === void 0 ? void 0 : _b[depName];
        if (depVersion) {
            const major = new semver_1.SemVer(depVersion).major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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