"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'awscdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const db = Nano(NPM_REPLICA_REGISTRY_URL).db.use('registry');
    const initialMarker = await loadLastTransactionMarker(db);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
                metrics.putMetric("LastSeq" /* LAST_SEQ */, updatedMarker, aws_embedded_metrics_1.Unit.None);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param registry a Nano database corresponding to the Npmjs.com CouchDB instance.
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(registry) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: constants_lambda_shared_1.MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            const dbUpdateSeq = (await registry.info()).update_seq;
            if (dbUpdateSeq < marker) {
                console.warn(`Current DB update_seq (${dbUpdateSeq}) is lower than marker (CouchDB instance was likely replaced), resetting to 0!`);
                return 0;
            }
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.warn(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from scratch`);
            return 0;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(constants_lambda_shared_1.MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    ko(new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`));
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a, _b, _c, _d;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)))
            || Object.keys((_b = infos.dependencies) !== null && _b !== void 0 ? _b : {}).some(isConstructFrameworkPackage)
            || Object.keys((_c = infos.devDependencies) !== null && _c !== void 0 ? _c : {}).some(isConstructFrameworkPackage)
            || Object.keys((_d = infos.peerDependencies) !== null && _d !== void 0 ? _d : {}).some(isConstructFrameworkPackage);
    }
    /**
     * Package is one of the known construct framework's first party packages:
     * - @aws-cdk/*
     * - @cdktf/*
     * - cdk8s or cdk8s-plus
     */
    function isConstructFrameworkPackage(name) {
        // NOTE: Prefix matching should only be used for @scope/ names.
        return name.startsWith('@aws-cdk/')
            || name.startsWith('@cdktf/')
            || name === 'cdk8s'
            || name === 'cdk8s-plus';
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibnBtLWpzLWZvbGxvd2VyLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wYWNrYWdlLXNvdXJjZXMvbnBtanMvbnBtLWpzLWZvbGxvd2VyLmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxtQ0FBbUM7QUFDbkMsK0JBQStCO0FBQy9CLDZCQUEwQjtBQUUxQiwrREFBdUY7QUFFdkYsaUVBQWlFO0FBQ2pFLDZCQUE4QjtBQUM5Qix1RkFBOEU7QUFDOUUsMEZBQW9GO0FBQ3BGLDhEQUE4RDtBQUM5RCw4RUFBb0U7QUFFcEUsMEZBQXlFO0FBQ3pFLHVFQUF5RztBQUN6RyxpRUFBaUU7QUFDakUsTUFBTSxvQkFBb0IsR0FBRyxPQUFPLENBQUMsNkJBQTZCLENBQUMsQ0FBQztBQUVwRSxNQUFNLG9CQUFvQixHQUFHLEtBQU0sQ0FBQztBQUNwQyxNQUFNLGtCQUFrQixHQUF3QixJQUFJLEdBQUcsQ0FBQyxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO0FBQ3hHLE1BQU0sd0JBQXdCLEdBQUcsOEJBQThCLENBQUM7QUFFaEUsb0NBQW9DO0FBQ3BDLG9DQUFhLENBQUMsU0FBUyxHQUFHLDJDQUFpQixDQUFDO0FBRTVDOzs7Ozs7Ozs7R0FTRztBQUNJLEtBQUssVUFBVSxPQUFPLENBQUMsS0FBcUIsRUFBRSxPQUFnQjtJQUNuRSxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUV4RCxNQUFNLGFBQWEsR0FBRyw4QkFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hELE1BQU0sUUFBUSxHQUFHLDhCQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7SUFFekMsTUFBTSxRQUFRLEdBQUcsTUFBTSxxQ0FBYyxDQUFDLFNBQVMsRUFBRSxDQUFDO0lBQ2xELE1BQU0sV0FBVyxHQUFHLE1BQU0sd0NBQWlCLENBQUMsU0FBUyxFQUFFLENBQUM7SUFFeEQsTUFBTSxFQUFFLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUU3RCxNQUFNLGFBQWEsR0FBRyxNQUFNLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRTFELE1BQU0sTUFBTSxHQUE4QjtRQUN4QyxXQUFXLEVBQUUsSUFBSTtRQUNqQiw4Q0FBOEM7UUFDOUMsSUFBSSxFQUFFLElBQUk7UUFDVixLQUFLLEVBQUUsYUFBYSxDQUFDLE9BQU8sRUFBRTtRQUM5QixxR0FBcUc7UUFDckcsK0ZBQStGO1FBQy9GLE9BQU8sRUFBRSxvQkFBb0I7UUFDN0IseUJBQXlCO1FBQ3pCLFFBQVEsRUFBRTtZQUNSLElBQUksRUFBRSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUU7U0FDcEI7UUFDRCxTQUFTLEVBQUUsRUFBRTtLQUNkLENBQUM7SUFFRiwyRUFBMkU7SUFDM0UseUJBQXlCO0lBQ3pCLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7UUFDNUIsSUFBSSxhQUFhLEdBQUcsYUFBYSxDQUFDO1FBRWxDLEVBQUUsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQzthQUN6QixFQUFFLENBQUMsT0FBTyxFQUFFLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxLQUF3QixFQUFFLEVBQUU7O1lBQ3ZFLCtIQUErSDtZQUMvSCxPQUFPLENBQUMsYUFBYSxFQUFFLENBQUM7WUFFeEIsT0FBTyxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUUsYUFBYSxDQUFDLENBQUM7WUFDL0MsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBRTdCLDhCQUE4QjtZQUM5QixLQUFLLE1BQU0sRUFBRSxHQUFHLEVBQUUsSUFBSSxLQUFLLEVBQUU7Z0JBQzNCLFVBQUksR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLElBQUksMENBQUUsUUFBUSxFQUFFO29CQUN2QixPQUFPLENBQUMsU0FBUywwQ0FFZixTQUFTLEdBQUcsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFDakQsMkJBQUksQ0FBQyxZQUFZLENBQ2xCLENBQUM7aUJBQ0g7YUFDRjtZQUVELElBQUk7Z0JBQ0YsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLE1BQU0sYUFBYSxDQUFDLENBQUM7Z0JBQzlELE9BQU8sQ0FBQyxTQUFTLG1DQUEwQixLQUFLLENBQUMsTUFBTSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ3JFLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDL0QsT0FBTyxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsYUFBYSxDQUFDLENBQUM7Z0JBRTdDLHdFQUF3RTtnQkFDeEUseUVBQXlFO2dCQUN6RSxNQUFNLFlBQVksR0FBRyx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxXQUFXLENBQUMsQ0FBQztnQkFDcEYsT0FBTyxDQUFDLEdBQUcsQ0FBQyxjQUFjLFlBQVksQ0FBQyxNQUFNLHFDQUFxQyxDQUFDLENBQUM7Z0JBQ3BGLE9BQU8sQ0FBQyxTQUFTLDREQUF1QyxZQUFZLENBQUMsTUFBTSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBRXpGLGdDQUFnQztnQkFDaEMsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFO29CQUNqRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUM7b0JBQzFCLE1BQU0scUJBQXFCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO29CQUM1QyxPQUFPLENBQUMsU0FBUyxtQ0FBMEIsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLE1BQU0sRUFBRSwyQkFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUNyRixDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUVKLHVDQUF1QztnQkFDdkMsTUFBTSx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDekMsYUFBYSxHQUFHLE9BQU8sQ0FBQztnQkFFeEIsc0dBQXNHO2dCQUN0RywyR0FBMkc7Z0JBQzNHLDJIQUEySDtnQkFDM0gsSUFBSSxPQUFPLENBQUMsd0JBQXdCLEVBQUUsSUFBSSxLQUFNLENBQUMsZ0JBQWdCLEVBQUU7b0JBQ2pFLE9BQU8sQ0FBQyxHQUFHLENBQUMsbURBQW1ELENBQUMsQ0FBQztvQkFDakUsMkZBQTJGO29CQUMxRixFQUFFLENBQUMsYUFBcUIsQ0FBQyxNQUFNLEVBQUUsQ0FBQztpQkFDcEM7cUJBQU07b0JBQ0wsT0FBTyxDQUFDLEdBQUcsQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO29CQUM1RCxFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxDQUFDO29CQUN4QixPQUFPLENBQUMsU0FBUyx1Q0FBNEIsT0FBTyxDQUFDLHdCQUF3QixFQUFFLEVBQUUsMkJBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztvQkFDcEcsRUFBRSxDQUFDLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7aUJBQ3RDO2FBQ0Y7WUFBQyxPQUFPLEdBQUcsRUFBRTtnQkFDWiwwRUFBMEU7Z0JBQzFFLE9BQU8sQ0FBQyxLQUFLLENBQUMscUJBQXFCLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQzFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7Z0JBQ3hCLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNUO29CQUFTO2dCQUNSLE9BQU8sQ0FBQyxTQUFTLG9EQUFtQyxJQUFJLENBQUMsR0FBRyxFQUFFLEdBQUcsU0FBUyxFQUFFLDJCQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7Z0JBQy9GLE9BQU8sQ0FBQyxTQUFTLDJCQUFzQixhQUFhLEVBQUUsMkJBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNsRTtRQUNILENBQUMsQ0FBQyxDQUFDO2FBQ0YsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUU7WUFDaEIsT0FBTyxDQUFDLEdBQUcsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1lBQ3BELEVBQUUsQ0FBQyxFQUFFLGFBQWEsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0lBQ1AsQ0FBQyxDQUFDLENBQUM7SUFFSCxpQ0FBaUM7SUFDakM7Ozs7OztPQU1HO0lBQ0gsS0FBSyxVQUFVLHlCQUF5QixDQUFDLFFBQXFDO1FBQzVFLElBQUk7WUFDRixNQUFNLFFBQVEsR0FBRyxNQUFNLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUM7Z0JBQ3hDLE1BQU0sRUFBRSxhQUFhO2dCQUNyQixHQUFHLEVBQUUsMENBQWdCO2FBQ3RCLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUNiLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLElBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDckUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsTUFBTSxFQUFFLENBQUMsQ0FBQztZQUV2RCxNQUFNLFdBQVcsR0FBRyxDQUFDLE1BQU0sUUFBUSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDO1lBQ3ZELElBQUksV0FBVyxHQUFHLE1BQU0sRUFBRTtnQkFDeEIsT0FBTyxDQUFDLElBQUksQ0FBQywwQkFBMEIsV0FBVyxnRkFBZ0YsQ0FBQyxDQUFDO2dCQUNwSSxPQUFPLENBQUMsQ0FBQzthQUNWO1lBRUQsT0FBTyxNQUFNLENBQUM7U0FDZjtRQUFDLE9BQU8sS0FBSyxFQUFFO1lBQ2QsSUFBSSxLQUFLLENBQUMsSUFBSSxLQUFLLFdBQVcsRUFBRTtnQkFDOUIsTUFBTSxLQUFLLENBQUM7YUFDYjtZQUNELE9BQU8sQ0FBQyxJQUFJLENBQUMsdUJBQXVCLGFBQWEsSUFBSSwwQ0FBZ0IseUNBQXlDLENBQUMsQ0FBQztZQUNoSCxPQUFPLENBQUMsQ0FBQztTQUNWO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxLQUFLLFVBQVUseUJBQXlCLENBQUMsUUFBZ0I7UUFDdkQsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1Q0FBdUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUMvRCxPQUFPLFNBQVMsQ0FBQywwQ0FBZ0IsRUFBRSxRQUFRLENBQUMsT0FBTyxFQUFFLEVBQUUsRUFBRSxXQUFXLEVBQUUsMkJBQTJCLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZHLENBQUM7SUFDRCxZQUFZO0lBRVosd0JBQXdCO0lBQ3hCLEtBQUssVUFBVSxxQkFBcUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsR0FBRyxFQUFrQixFQUFFLE9BQXNCO1FBQ25HLElBQUk7WUFDRix1QkFBdUI7WUFDdkIsTUFBTSxPQUFPLEdBQUcsTUFBTSxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUVsRCw0Q0FBNEM7WUFDNUMsNEZBQTRGO1lBQzVGLDRGQUE0RjtZQUM1RixNQUFNLFVBQVUsR0FBRyxHQUFHLGlDQUE2QixHQUFHLElBQUksU0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsQ0FBQztZQUNySCxNQUFNLFNBQVMsQ0FBQyxVQUFVLEVBQUUsT0FBTyxFQUFFO2dCQUNuQyxXQUFXLEVBQUUsMEJBQTBCO2dCQUN2QyxRQUFRLEVBQUU7b0JBQ1IsYUFBYSxFQUFFLFFBQVEsQ0FBQyxXQUFXLEVBQUU7b0JBQ3JDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTTtvQkFDckMsWUFBWSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTztvQkFDaEMsVUFBVSxFQUFFLEdBQUcsQ0FBQyxPQUFPLEVBQUU7aUJBQzFCO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsb0NBQW9DO1lBQ3BDLE1BQU0sV0FBVyxHQUFHO2dCQUNsQixVQUFVLEVBQUUsUUFBUSxhQUFhLElBQUksVUFBVSxFQUFFO2dCQUNqRCxRQUFRLEVBQUU7b0JBQ1IsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTztvQkFDeEIsR0FBRyxFQUFFLEdBQUcsQ0FBQyxPQUFPLEVBQUU7aUJBQ25CO2dCQUNELElBQUksRUFBRSxRQUFRLENBQUMsV0FBVyxFQUFFO2FBQzdCLENBQUM7WUFDRixNQUFNLE9BQU8sR0FBbUI7Z0JBQzlCLEdBQUcsV0FBVztnQkFDZCxTQUFTLEVBQUUsbUNBQVMsQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDO2FBQzNDLENBQUM7WUFFRiwyQkFBMkI7WUFDM0IsTUFBTSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsV0FBVyxDQUFDO2dCQUMxQixXQUFXLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztnQkFDN0MsUUFBUSxFQUFFLFFBQVE7YUFDbkIsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBRWIsT0FBTyxDQUFDLFNBQVMsb0RBQW1DLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3BFO1FBQUMsT0FBTyxHQUFHLEVBQUU7WUFDWiwwRUFBMEU7WUFDMUUsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLEdBQUcsK0RBQStELEtBQUssQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBQzNILE9BQU8sQ0FBQyxTQUFTLG9EQUFtQyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNuRSxNQUFNLFNBQVMsQ0FBQyxHQUFHLGlDQUE2QixHQUFHLEdBQUcsRUFBRSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxHQUFHLEtBQUssRUFBRSw2QkFBNkIsRUFBRSxHQUFHLEVBQUUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUU7Z0JBQ25JLFdBQVcsRUFBRSxrQkFBa0I7Z0JBQy9CLFFBQVEsRUFBRTtvQkFDUixhQUFhLEVBQUUsUUFBUSxDQUFDLFdBQVcsRUFBRTtpQkFDdEM7YUFDRixDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFDRCxZQUFZO0lBRVosaUNBQWlDO0lBQ2pDOzs7Ozs7T0FNRztJQUNILFNBQVMsT0FBTyxDQUFDLEdBQVc7UUFDMUIsT0FBTyxJQUFJLE9BQU8sQ0FBUyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRTtZQUNwQyxLQUFLLENBQUMsR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDLFFBQVEsRUFBRSxFQUFFO2dCQUMxQixJQUFJLFFBQVEsQ0FBQyxVQUFVLEtBQUssR0FBRyxFQUFFO29CQUMvQixFQUFFLENBQUMsSUFBSSxLQUFLLENBQUMscUJBQXFCLFFBQVEsQ0FBQyxVQUFVLE1BQU0sUUFBUSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQztpQkFDdkY7Z0JBRUQsSUFBSSxJQUFJLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDM0IsUUFBUSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2pGLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2dCQUN2QyxRQUFRLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztZQUM3QixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0gsU0FBUyxTQUFTLENBQUMsR0FBVyxFQUFFLElBQWlCLEVBQUUsT0FBaUUsRUFBRTtRQUNwSCxPQUFPLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUM7WUFDeEIsTUFBTSxFQUFFLGFBQWE7WUFDckIsR0FBRyxFQUFFLEdBQUc7WUFDUixJQUFJLEVBQUUsSUFBSTtZQUNWLFFBQVEsRUFBRTtnQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7Z0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDckMsR0FBRyxJQUFJLENBQUMsUUFBUTthQUNqQjtZQUNELEdBQUcsSUFBSTtTQUNSLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNmLENBQUM7SUFDRCxZQUFZO0FBQ2QsQ0FBQztBQTFQRCwwQkEwUEM7QUFFRDs7Ozs7Ozs7OztHQVVHO0FBQ0gsU0FBUyx1QkFBdUIsQ0FDOUIsT0FBMEIsRUFDMUIsT0FBc0IsRUFDdEIsUUFBd0IsRUFDeEIsV0FBOEI7O0lBRzlCLE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFrQixDQUFDO0lBRTNDLEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1FBQzVCLHVFQUF1RTtRQUN2RSxrRUFBa0U7UUFDbEUsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyxTQUFTLEVBQUU7WUFDakMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLDBDQUEwQyxNQUFNLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUNuRixPQUFPLENBQUMsU0FBUyxtREFBa0MsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbEUsU0FBUztTQUNWO1FBRUQsMkdBQTJHO1FBQzNHLElBQUksb0JBQW9CLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsRUFBRTtZQUNsRCxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsaUVBQWlFLE1BQU0sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzFHLE9BQU8sQ0FBQyxTQUFTLG1EQUFrQyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNsRSxTQUFTO1NBQ1Y7UUFFRCxtRUFBbUU7UUFDbkUsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLFFBQVEsSUFBSSxJQUFJLEVBQUU7WUFDL0IsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLDhDQUE4QyxNQUFNLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUN2RixPQUFPLENBQUMsU0FBUyxtREFBa0MsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbEUsU0FBUztTQUNWO1FBRUQsc0VBQXNFO1FBQ3RFLElBQUksTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksSUFBSSxFQUFFO1lBQzNCLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRywwQ0FBMEMsTUFBTSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDbkYsT0FBTyxDQUFDLFNBQVMsbURBQWtDLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ2xFLFNBQVM7U0FDVjtRQUVELGlEQUFpRDtRQUNqRCxNQUFNLGFBQWEsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDO1lBQ25ELGdEQUFnRDthQUMvQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxHQUFHLEtBQUssU0FBUyxJQUFJLEdBQUcsS0FBSyxVQUFVLENBQUM7WUFDM0Qsb0RBQW9EO2FBQ25ELEdBQUcsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLE9BQU8sRUFBRSxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBVSxDQUFDO1lBQ25FLDJCQUEyQjthQUMxQixJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDckQsT0FBTyxDQUFDLFNBQVMsb0RBQW1DLGFBQWEsQ0FBQyxNQUFNLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV0RixJQUFJLGNBQWdDLENBQUM7UUFDckMsS0FBSyxNQUFNLENBQUMsT0FBTyxFQUFFLFFBQVEsQ0FBQyxJQUFJLGFBQWEsRUFBRTtZQUMvQyxJQUFJLGNBQWMsSUFBSSxJQUFJLElBQUksY0FBYyxDQUFDLE9BQU8sRUFBRSxLQUFLLFFBQVEsQ0FBQyxPQUFPLEVBQUUsRUFBRTtnQkFDN0UsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQzNDLElBQUksS0FBSyxJQUFJLElBQUksRUFBRTtvQkFDakIscURBQXFEO29CQUNyRCxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsOEJBQThCLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxJQUFJLE9BQU8seUJBQXlCLENBQUMsQ0FBQztpQkFDOUc7cUJBQU0sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsRUFBRTtvQkFFcEMsaUNBQWlDO29CQUNqQyxNQUFNLE1BQU0sR0FBRyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO29CQUMxRCxJQUFJLE1BQU0sRUFBRTt3QkFDVixPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcscUJBQXFCLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO3dCQUN6RSxPQUFPLENBQUMsU0FBUyw0Q0FBK0IsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7d0JBQy9ELFNBQVM7cUJBQ1Y7b0JBRUQsT0FBTyxDQUFDLFNBQVMsZ0RBQWlDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxRQUFRLENBQUMsT0FBTyxFQUFFLEVBQUUsMkJBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztvQkFDdEcsTUFBTSxVQUFVLEdBQUcsV0FBVyxDQUFDLE1BQU0sT0FBQyxLQUFLLENBQUMsT0FBTyxtQ0FBSSxZQUFZLENBQUMsSUFBSSxJQUFJLENBQUM7b0JBQzdFLE9BQU8sQ0FBQyxTQUFTLCtDQUFnQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQ2pGLElBQUksVUFBVSxFQUFFO3dCQUNkLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLEdBQUcsRUFBRSxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztxQkFDbkQ7eUJBQU07d0JBQ0wsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLGNBQWMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksT0FBTyx3Q0FBd0MsTUFBQSxLQUFLLENBQUMsT0FBTyxtQ0FBSSxZQUFZLEVBQUUsQ0FBQyxDQUFDO3FCQUM1STtpQkFDRjtxQkFBTTtvQkFDTCxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsZUFBZSxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSSxPQUFPLHFDQUFxQyxDQUFDLENBQUM7aUJBQzNHO2dCQUNELGNBQWMsR0FBRyxRQUFRLENBQUM7YUFDM0I7U0FDRjtLQUNGO0lBQ0QsT0FBTyxNQUFNLENBQUM7SUFFZCxTQUFTLGtCQUFrQixDQUFDLEtBQWtCOztRQUM1QyxJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxFQUFFO1lBQ3RCLE9BQU8sS0FBSyxDQUFDO1NBQ2Q7UUFDRCxPQUFPLEtBQUssQ0FBQyxJQUFJLEtBQUssV0FBVztlQUM1QixLQUFLLENBQUMsSUFBSSxLQUFLLGFBQWE7ZUFDNUIsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDO3NCQUNqQyxLQUFLLENBQUMsUUFBUSwwQ0FBRSxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBQztlQUN4RCxNQUFNLENBQUMsSUFBSSxPQUFDLEtBQUssQ0FBQyxZQUFZLG1DQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQywyQkFBMkIsQ0FBQztlQUN2RSxNQUFNLENBQUMsSUFBSSxPQUFDLEtBQUssQ0FBQyxlQUFlLG1DQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQywyQkFBMkIsQ0FBQztlQUMxRSxNQUFNLENBQUMsSUFBSSxPQUFDLEtBQUssQ0FBQyxnQkFBZ0IsbUNBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0gsU0FBUywyQkFBMkIsQ0FBQyxJQUFZO1FBQy9DLCtEQUErRDtRQUMvRCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDO2VBQzlCLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDO2VBQzFCLElBQUksS0FBSyxPQUFPO2VBQ2hCLElBQUksS0FBSyxZQUFZLENBQUM7SUFDN0IsQ0FBQztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb25zb2xlIGZyb20gJ2NvbnNvbGUnO1xuaW1wb3J0ICogYXMgaHR0cHMgZnJvbSAnaHR0cHMnO1xuaW1wb3J0IHsgVVJMIH0gZnJvbSAndXJsJztcblxuaW1wb3J0IHsgbWV0cmljU2NvcGUsIENvbmZpZ3VyYXRpb24sIE1ldHJpY3NMb2dnZXIsIFVuaXQgfSBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcyc7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQsIFNjaGVkdWxlZEV2ZW50IH0gZnJvbSAnYXdzLWxhbWJkYSc7XG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuaW1wb3J0IE5hbm8gPSByZXF1aXJlKCduYW5vJyk7XG5pbXBvcnQgeyBEZW55TGlzdENsaWVudCB9IGZyb20gJy4uLy4uL2JhY2tlbmQvZGVueS1saXN0L2NsaWVudC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IExpY2Vuc2VMaXN0Q2xpZW50IH0gZnJvbSAnLi4vLi4vYmFja2VuZC9saWNlbnNlLWxpc3QvY2xpZW50LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0ICogYXMgYXdzIGZyb20gJy4uLy4uL2JhY2tlbmQvc2hhcmVkL2F3cy5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IHJlcXVpcmVFbnYgfSBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9lbnYubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBJbmdlc3Rpb25JbnB1dCB9IGZyb20gJy4uLy4uL2JhY2tlbmQvc2hhcmVkL2luZ2VzdGlvbi1pbnB1dC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IGludGVncml0eSB9IGZyb20gJy4uLy4uL2JhY2tlbmQvc2hhcmVkL2ludGVncml0eS5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IE1ldHJpY05hbWUsIE1BUktFUl9GSUxFX05BTUUsIE1FVFJJQ1NfTkFNRVNQQUNFLCBTM0tleVByZWZpeCB9IGZyb20gJy4vY29uc3RhbnRzLmxhbWJkYS1zaGFyZWQnO1xuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIEB0eXBlc2NyaXB0LWVzbGludC9uby1yZXF1aXJlLWltcG9ydHNcbmNvbnN0IG5vcm1hbGl6ZU5QTU1ldGFkYXRhID0gcmVxdWlyZSgnbm9ybWFsaXplLXJlZ2lzdHJ5LW1ldGFkYXRhJyk7XG5cbmNvbnN0IFRJTUVPVVRfTUlMTElTRUNPTkRTID0gMTBfMDAwO1xuY29uc3QgQ09OU1RSVUNUX0tFWVdPUkRTOiBSZWFkb25seVNldDxzdHJpbmc+ID0gbmV3IFNldChbJ2NkaycsICdhd3MtY2RrJywgJ2F3c2NkaycsICdjZGs4cycsICdjZGt0ZiddKTtcbmNvbnN0IE5QTV9SRVBMSUNBX1JFR0lTVFJZX1VSTCA9ICdodHRwczovL3JlcGxpY2F0ZS5ucG1qcy5jb20vJztcblxuLy8gQ29uZmlndXJlIGVtYmVkZGVkIG1ldHJpY3MgZm9ybWF0XG5Db25maWd1cmF0aW9uLm5hbWVzcGFjZSA9IE1FVFJJQ1NfTkFNRVNQQUNFO1xuXG4vKipcbiAqIFRoaXMgZnVuY3Rpb24gdHJpZ2dlcnMgb24gYSBmaXhlZCBzY2hlZHVsZSBhbmQgcmVhZHMgYSBzdHJlYW0gb2YgY2hhbmdlcyBmcm9tIG5wbWpzIGNvdWNoZGIgX2NoYW5nZXMgZW5kcG9pbnQuXG4gKiBVcG9uIGludm9jYXRpb24gdGhlIGZ1bmN0aW9uIHN0YXJ0cyByZWFkaW5nIGZyb20gYSBzZXF1ZW5jZSBzdG9yZWQgaW4gYW4gczMgb2JqZWN0IC0gdGhlIGBtYXJrZXJgLlxuICogSWYgdGhlIG1hcmtlciBmYWlscyB0byBsb2FkIChvciBkbyBub3QgZXhpc3QpLCB0aGUgc3RyZWFtIHdpbGwgc3RhcnQgZnJvbSBgbm93YCAtIHRoZSBsYXRlc3QgY2hhbmdlLlxuICogRm9yIGVhY2ggY2hhbmdlOlxuICogIC0gdGhlIHBhY2thZ2UgdmVyc2lvbiB0YXJiYWxsIHdpbGwgYmUgY29waWVkIGZyb20gdGhlIG5wbSByZWdpc3RyeSB0byBhIHN0YXRpbmcgYnVja2V0LlxuICogIC0gYSBtZXNzYWdlIHdpbGwgYmUgc2VudCB0byBhbiBzcXMgcXVldWVcbiAqIG5wbSByZWdpc3RyeSBBUEkgZG9jczogaHR0cHM6Ly9naXRodWIuY29tL25wbS9yZWdpc3RyeS9ibG9iL21hc3Rlci9kb2NzL1JFR0lTVFJZLUFQSS5tZFxuICogQHBhcmFtIGNvbnRleHQgYSBMYW1iZGEgZXhlY3V0aW9uIGNvbnRleHRcbiAqL1xuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGhhbmRsZXIoZXZlbnQ6IFNjaGVkdWxlZEV2ZW50LCBjb250ZXh0OiBDb250ZXh0KSB7XG4gIGNvbnNvbGUubG9nKGBFdmVudDogJHtKU09OLnN0cmluZ2lmeShldmVudCwgbnVsbCwgMil9YCk7XG5cbiAgY29uc3Qgc3RhZ2luZ0J1Y2tldCA9IHJlcXVpcmVFbnYoJ0JVQ0tFVF9OQU1FJyk7XG4gIGNvbnN0IHF1ZXVlVXJsID0gcmVxdWlyZUVudignUVVFVUVfVVJMJyk7XG5cbiAgY29uc3QgZGVueUxpc3QgPSBhd2FpdCBEZW55TGlzdENsaWVudC5uZXdDbGllbnQoKTtcbiAgY29uc3QgbGljZW5zZUxpc3QgPSBhd2FpdCBMaWNlbnNlTGlzdENsaWVudC5uZXdDbGllbnQoKTtcblxuICBjb25zdCBkYiA9IE5hbm8oTlBNX1JFUExJQ0FfUkVHSVNUUllfVVJMKS5kYi51c2UoJ3JlZ2lzdHJ5Jyk7XG5cbiAgY29uc3QgaW5pdGlhbE1hcmtlciA9IGF3YWl0IGxvYWRMYXN0VHJhbnNhY3Rpb25NYXJrZXIoZGIpO1xuXG4gIGNvbnN0IGNvbmZpZzogTmFuby5DaGFuZ2VzUmVhZGVyT3B0aW9ucyA9IHtcbiAgICBpbmNsdWRlRG9jczogdHJ1ZSxcbiAgICAvLyBwYXVzZSB0aGUgY2hhbmdlcyByZWFkZXIgYWZ0ZXIgZWFjaCByZXF1ZXN0XG4gICAgd2FpdDogdHJ1ZSxcbiAgICBzaW5jZTogaW5pdGlhbE1hcmtlci50b0ZpeGVkKCksXG4gICAgLy8gYGNoYW5nZXNSZWFkZXIuZ2V0YCBzdG9wcyBvbmNlIGEgcmVzcG9uc2Ugd2l0aCB6ZXJvIGNoYW5nZXMgaXMgcmVjZWl2ZWQsIGhvd2V2ZXIgaXQgd2FpdHMgdG9vIGxvbmdcbiAgICAvLyAgc2luY2Ugd2Ugd2FudCB0byB0ZXJtaW5hdGUgdGhlIExhbWJkYSBmdW5jdGlvbiB3ZSBkZWZpbmUgYSB0aW1lb3V0IHNob3J0ZXIgdGhhbiB0aGUgZGVmYXVsdFxuICAgIHRpbWVvdXQ6IFRJTUVPVVRfTUlMTElTRUNPTkRTLFxuICAgIC8vIE9ubHkgaXRlbXMgd2l0aCBhIG5hbWVcbiAgICBzZWxlY3Rvcjoge1xuICAgICAgbmFtZTogeyAkZ3Q6IG51bGwgfSxcbiAgICB9LFxuICAgIGJhdGNoU2l6ZTogMzAsXG4gIH07XG5cbiAgLy8gV2UgbmVlZCB0byBtYWtlIGFuIGV4cGxpY2l0IFByb21pc2UgaGVyZSwgYmVjYXVzZSBvdGhlcndpc2UgTGFtYmRhIHdvbid0XG4gIC8vIGtub3cgd2hlbiBpdCdzIGRvbmUuLi5cbiAgcmV0dXJuIG5ldyBQcm9taXNlKChvaywga28pID0+IHtcbiAgICBsZXQgdXBkYXRlZE1hcmtlciA9IGluaXRpYWxNYXJrZXI7XG5cbiAgICBkYi5jaGFuZ2VzUmVhZGVyLmdldChjb25maWcpXG4gICAgICAub24oJ2JhdGNoJywgbWV0cmljU2NvcGUoKG1ldHJpY3MpID0+IGFzeW5jIChiYXRjaDogcmVhZG9ubHkgQ2hhbmdlW10pID0+IHtcbiAgICAgICAgLy8gQ2xlYXIgYXV0b21hdGljYWxseSBzZXQgZGltZW5zaW9ucyAtIHdlIGRvbid0IG5lZWQgdGhlbSAoc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3NsYWJzL2F3cy1lbWJlZGRlZC1tZXRyaWNzLW5vZGUvaXNzdWVzLzczKVxuICAgICAgICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcblxuICAgICAgICBtZXRyaWNzLnNldFByb3BlcnR5KCdTdGFydFNlcScsIHVwZGF0ZWRNYXJrZXIpO1xuICAgICAgICBjb25zdCBzdGFydFRpbWUgPSBEYXRlLm5vdygpO1xuXG4gICAgICAgIC8vIEVtaXQgbnBtLmpzIHJlcGxpY2F0aW9uIGxhZ1xuICAgICAgICBmb3IgKGNvbnN0IHsgZG9jIH0gb2YgYmF0Y2gpIHtcbiAgICAgICAgICBpZiAoZG9jPy50aW1lPy5tb2RpZmllZCkge1xuICAgICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoXG4gICAgICAgICAgICAgIE1ldHJpY05hbWUuTlBNSlNfQ0hBTkdFX0FHRSxcbiAgICAgICAgICAgICAgc3RhcnRUaW1lIC0gbmV3IERhdGUoZG9jLnRpbWUubW9kaWZpZWQpLmdldFRpbWUoKSxcbiAgICAgICAgICAgICAgVW5pdC5NaWxsaXNlY29uZHMsXG4gICAgICAgICAgICApO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIHRyeSB7XG4gICAgICAgICAgY29uc29sZS5sb2coYFJlY2VpdmVkIGEgYmF0Y2ggb2YgJHtiYXRjaC5sZW5ndGh9IGVsZW1lbnQocylgKTtcbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLkNIQU5HRV9DT1VOVCwgYmF0Y2gubGVuZ3RoLCBVbml0LkNvdW50KTtcbiAgICAgICAgICBjb25zdCBsYXN0U2VxID0gTWF0aC5tYXgoLi4uYmF0Y2gubWFwKChjaGFuZ2UpID0+IGNoYW5nZS5zZXEpKTtcbiAgICAgICAgICBtZXRyaWNzLnNldFByb3BlcnR5KCdFbmRTZXEnLCB1cGRhdGVkTWFya2VyKTtcblxuICAgICAgICAgIC8vIE9idGFpbiB0aGUgbW9kaWZpZWQgcGFja2FnZSB2ZXJzaW9uIGZyb20gdGhlIHVwZGF0ZSBldmVudCwgYW5kIGZpbHRlclxuICAgICAgICAgIC8vIG91dCBwYWNrYWdlcyB0aGF0IGFyZSBub3Qgb2YgaW50ZXJlc3QgdG8gdXMgKG5vdCBjb25zdHJ1Y3QgbGlicmFyaWVzKS5cbiAgICAgICAgICBjb25zdCB2ZXJzaW9uSW5mb3MgPSBnZXRSZWxldmFudFZlcnNpb25JbmZvcyhiYXRjaCwgbWV0cmljcywgZGVueUxpc3QsIGxpY2Vuc2VMaXN0KTtcbiAgICAgICAgICBjb25zb2xlLmxvZyhgSWRlbnRpZmllZCAke3ZlcnNpb25JbmZvcy5sZW5ndGh9IHJlbGV2YW50IHBhY2thZ2UgdmVyc2lvbiB1cGRhdGUocylgKTtcbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlJFTEVWQU5UX1BBQ0tBR0VfVkVSU0lPTlMsIHZlcnNpb25JbmZvcy5sZW5ndGgsIFVuaXQuQ291bnQpO1xuXG4gICAgICAgICAgLy8gUHJvY2VzcyBhbGwgcmVtYWluaW5nIHVwZGF0ZXNcbiAgICAgICAgICBhd2FpdCBQcm9taXNlLmFsbCh2ZXJzaW9uSW5mb3MubWFwKGFzeW5jIChpbmZvcykgPT4ge1xuICAgICAgICAgICAgY29uc3QgYmVmb3JlID0gRGF0ZS5ub3coKTtcbiAgICAgICAgICAgIGF3YWl0IHByb2Nlc3NVcGRhdGVkVmVyc2lvbihpbmZvcywgbWV0cmljcyk7XG4gICAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlNUQUdJTkdfVElNRSwgRGF0ZS5ub3coKSAtIGJlZm9yZSwgVW5pdC5NaWxsaXNlY29uZHMpO1xuICAgICAgICAgIH0pKTtcblxuICAgICAgICAgIC8vIFVwZGF0ZSB0aGUgdHJhbnNhY3Rpb24gbWFya2VyIGluIFMzLlxuICAgICAgICAgIGF3YWl0IHNhdmVMYXN0VHJhbnNhY3Rpb25NYXJrZXIobGFzdFNlcSk7XG4gICAgICAgICAgdXBkYXRlZE1hcmtlciA9IGxhc3RTZXE7XG5cbiAgICAgICAgICAvLyBJZiB3ZSBoYXZlIGVub3VnaCB0aW1lIGxlZnQgYmVmb3JlIHRpbWVvdXQsIHByb2NlZWQgd2l0aCB0aGUgbmV4dCBiYXRjaCwgb3RoZXJ3aXNlIHdlJ3JlIGRvbmUgaGVyZS5cbiAgICAgICAgICAvLyBTaW5jZSB0aGUgZGlzdHJpYnV0aW9uIG9mIHRoZSB0aW1lIGl0IHRha2VzIHRvIHByb2Nlc3MgZWFjaCBwYWNrYWdlL2JhdGNoIGlzIG5vbiB1bmlmb3JtLCB0aGlzIGlzIGEgYmVzdFxuICAgICAgICAgIC8vIGVmZm9ydCwgYW5kIHdlIGV4cGVjdCB0aGUgZnVuY3Rpb24gdG8gdGltZW91dCBpbiBzb21lIGludm9jYXRpb25zLCB3ZSByZWx5IG9uIHRoZSBkb3duc3RyZWFtIGlkZW1wb3RlbmN5IHRvIGhhbmRsZSB0aGlzLlxuICAgICAgICAgIGlmIChjb250ZXh0LmdldFJlbWFpbmluZ1RpbWVJbk1pbGxpcygpID49IDMwXzAwMCAvKiAzMCBzZWNvbmRzICovKSB7XG4gICAgICAgICAgICBjb25zb2xlLmxvZygnVGhlcmUgaXMgc3RpbGwgdGltZSwgcmVxdWVzdGluZyB0aGUgbmV4dCBiYXRjaC4uLicpO1xuICAgICAgICAgICAgLy8gTm90ZTogdGhlIGByZXN1bWVgIGZ1bmN0aW9uIGlzIG1pc3NpbmcgZnJvbSB0aGUgYG5hbm9gIHR5cGUgZGVmaW5pdGlvbnMsIGJ1dCBpcyB0aGVyZS4uLlxuICAgICAgICAgICAgKGRiLmNoYW5nZXNSZWFkZXIgYXMgYW55KS5yZXN1bWUoKTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgY29uc29sZS5sb2coJ1dlIGFyZSBhbG1vc3Qgb3V0IG9mIHRpbWUsIHNvIHN0b3BwaW5nIGhlcmUuJyk7XG4gICAgICAgICAgICBkYi5jaGFuZ2VzUmVhZGVyLnN0b3AoKTtcbiAgICAgICAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuUkVNQUlOSU5HX1RJTUUsIGNvbnRleHQuZ2V0UmVtYWluaW5nVGltZUluTWlsbGlzKCksIFVuaXQuTWlsbGlzZWNvbmRzKTtcbiAgICAgICAgICAgIG9rKHsgaW5pdGlhbE1hcmtlciwgdXBkYXRlZE1hcmtlciB9KTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgICAgIC8vIEFuIGV4Y2VwdGlvbiBidWJibGVkIG91dCwgd2hpY2ggbWVhbnMgdGhpcyBMYW1iZGEgZXhlY3V0aW9uIGhhcyBmYWlsZWQuXG4gICAgICAgICAgY29uc29sZS5lcnJvcihgVW5leHBlY3RlZCBlcnJvcjogJHtlcnJ9YCk7XG4gICAgICAgICAgZGIuY2hhbmdlc1JlYWRlci5zdG9wKCk7XG4gICAgICAgICAga28oZXJyKTtcbiAgICAgICAgfSBmaW5hbGx5IHtcbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLkJBVENIX1BST0NFU1NJTkdfVElNRSwgRGF0ZS5ub3coKSAtIHN0YXJ0VGltZSwgVW5pdC5NaWxsaXNlY29uZHMpO1xuICAgICAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuTEFTVF9TRVEsIHVwZGF0ZWRNYXJrZXIsIFVuaXQuTm9uZSk7XG4gICAgICAgIH1cbiAgICAgIH0pKVxuICAgICAgLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgY29uc29sZS5sb2coJ05vIG1vcmUgdXBkYXRlcyB0byBwcm9jZXNzLCBleGl0aW5nLicpO1xuICAgICAgICBvayh7IGluaXRpYWxNYXJrZXIsIHVwZGF0ZWRNYXJrZXIgfSk7XG4gICAgICB9KTtcbiAgfSk7XG5cbiAgLy8jcmVnaW9uIExhc3QgdHJhbnNhY3Rpb24gbWFya2VyXG4gIC8qKlxuICAgKiBMb2FkcyB0aGUgbGFzdCB0cmFuc2FjdGlvbiBtYXJrZXIgZnJvbSBTMy5cbiAgICpcbiAgICogQHBhcmFtIHJlZ2lzdHJ5IGEgTmFubyBkYXRhYmFzZSBjb3JyZXNwb25kaW5nIHRvIHRoZSBOcG1qcy5jb20gQ291Y2hEQiBpbnN0YW5jZS5cbiAgICpcbiAgICogQHJldHVybnMgdGhlIHZhbHVlIG9mIHRoZSBsYXN0IHRyYW5zYWN0aW9uIG1hcmtlci5cbiAgICovXG4gIGFzeW5jIGZ1bmN0aW9uIGxvYWRMYXN0VHJhbnNhY3Rpb25NYXJrZXIocmVnaXN0cnk6IE5hbm8uRG9jdW1lbnRTY29wZTx1bmtub3duPik6IFByb21pc2U8bnVtYmVyPiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgYXdzLnMzKCkuZ2V0T2JqZWN0KHtcbiAgICAgICAgQnVja2V0OiBzdGFnaW5nQnVja2V0LFxuICAgICAgICBLZXk6IE1BUktFUl9GSUxFX05BTUUsXG4gICAgICB9KS5wcm9taXNlKCk7XG4gICAgICBjb25zdCBtYXJrZXIgPSBOdW1iZXIucGFyc2VJbnQocmVzcG9uc2UuQm9keSEudG9TdHJpbmcoJ3V0Zi04JyksIDEwKTtcbiAgICAgIGNvbnNvbGUubG9nKGBSZWFkIGxhc3QgdHJhbnNhY3Rpb24gbWFya2VyOiAke21hcmtlcn1gKTtcblxuICAgICAgY29uc3QgZGJVcGRhdGVTZXEgPSAoYXdhaXQgcmVnaXN0cnkuaW5mbygpKS51cGRhdGVfc2VxO1xuICAgICAgaWYgKGRiVXBkYXRlU2VxIDwgbWFya2VyKSB7XG4gICAgICAgIGNvbnNvbGUud2FybihgQ3VycmVudCBEQiB1cGRhdGVfc2VxICgke2RiVXBkYXRlU2VxfSkgaXMgbG93ZXIgdGhhbiBtYXJrZXIgKENvdWNoREIgaW5zdGFuY2Ugd2FzIGxpa2VseSByZXBsYWNlZCksIHJlc2V0dGluZyB0byAwIWApO1xuICAgICAgICByZXR1cm4gMDtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIG1hcmtlcjtcbiAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgaWYgKGVycm9yLmNvZGUgIT09ICdOb1N1Y2hLZXknKSB7XG4gICAgICAgIHRocm93IGVycm9yO1xuICAgICAgfVxuICAgICAgY29uc29sZS53YXJuKGBNYXJrZXIgb2JqZWN0IChzMzovLyR7c3RhZ2luZ0J1Y2tldH0vJHtNQVJLRVJfRklMRV9OQU1FfSkgZG9lcyBub3QgZXhpc3QsIHN0YXJ0aW5nIGZyb20gc2NyYXRjaGApO1xuICAgICAgcmV0dXJuIDA7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgdGhlIGxhc3QgdHJhbnNhY3Rpb24gbWFya2VyIGluIFMzLlxuICAgKlxuICAgKiBAcGFyYW0gc2VxdWVuY2UgdGhlIGxhc3QgdHJhbnNhY3Rpb24gbWFya2VyIHZhbHVlXG4gICAqL1xuICBhc3luYyBmdW5jdGlvbiBzYXZlTGFzdFRyYW5zYWN0aW9uTWFya2VyKHNlcXVlbmNlOiBOdW1iZXIpIHtcbiAgICBjb25zb2xlLmxvZyhgVXBkYXRpbmcgbGFzdCB0cmFuc2FjdGlvbiBtYXJrZXIgdG8gJHtzZXF1ZW5jZX1gKTtcbiAgICByZXR1cm4gcHV0T2JqZWN0KE1BUktFUl9GSUxFX05BTUUsIHNlcXVlbmNlLnRvRml4ZWQoKSwgeyBDb250ZW50VHlwZTogJ3RleHQvcGxhaW47IGNoYXJzZXQ9VVRGLTgnIH0pO1xuICB9XG4gIC8vI2VuZHJlZ2lvblxuXG4gIC8vI3JlZ2lvbiBCdXNpbmVzcyBMb2dpY1xuICBhc3luYyBmdW5jdGlvbiBwcm9jZXNzVXBkYXRlZFZlcnNpb24oeyBpbmZvcywgbW9kaWZpZWQsIHNlcSB9OiBVcGRhdGVkVmVyc2lvbiwgbWV0cmljczogTWV0cmljc0xvZ2dlcik6IFByb21pc2U8dm9pZD4ge1xuICAgIHRyeSB7XG4gICAgICAvLyBEb3dubG9hZCB0aGUgdGFyYmFsbFxuICAgICAgY29uc3QgdGFyYmFsbCA9IGF3YWl0IGh0dHBHZXQoaW5mb3MuZGlzdC50YXJiYWxsKTtcblxuICAgICAgLy8gU3RvcmUgdGhlIHRhcmJhbGwgaW50byB0aGUgc3RhZ2luZyBidWNrZXRcbiAgICAgIC8vIC0gaW5mb3MuZGlzdC50YXJiYWxsID0+IGh0dHBzOi8vcmVnaXN0cnkubnBtanMub3JnLzxAc2NvcGU+LzxuYW1lPi8tLzxuYW1lPi08dmVyc2lvbj4udGd6XG4gICAgICAvLyAtIHN0YWdpbmdLZXkgICAgICAgICA9PiAgICAgICAgICAgICAgICAgICAgIHN0YWdlZC88QHNjb3BlPi88bmFtZT4vLS88bmFtZT4tPHZlcnNpb24+LnRnelxuICAgICAgY29uc3Qgc3RhZ2luZ0tleSA9IGAke1MzS2V5UHJlZml4LlNUQUdFRF9LRVlfUFJFRklYfSR7bmV3IFVSTChpbmZvcy5kaXN0LnRhcmJhbGwpLnBhdGhuYW1lfWAucmVwbGFjZSgvXFwvezIsfS9nLCAnLycpO1xuICAgICAgYXdhaXQgcHV0T2JqZWN0KHN0YWdpbmdLZXksIHRhcmJhbGwsIHtcbiAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9vY3RldC1zdHJlYW0nLFxuICAgICAgICBNZXRhZGF0YToge1xuICAgICAgICAgICdNb2RpZmllZC1BdCc6IG1vZGlmaWVkLnRvSVNPU3RyaW5nKCksXG4gICAgICAgICAgJ09yaWdpbi1JbnRlZ3JpdHknOiBpbmZvcy5kaXN0LnNoYXN1bSxcbiAgICAgICAgICAnT3JpZ2luLVVSSSc6IGluZm9zLmRpc3QudGFyYmFsbCxcbiAgICAgICAgICAnU2VxdWVuY2UnOiBzZXEudG9GaXhlZCgpLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFByZXBhcmUgU1FTIG1lc3NhZ2UgZm9yIGluZ2VzdGlvblxuICAgICAgY29uc3QgbWVzc2FnZUJhc2UgPSB7XG4gICAgICAgIHRhcmJhbGxVcmk6IGBzMzovLyR7c3RhZ2luZ0J1Y2tldH0vJHtzdGFnaW5nS2V5fWAsXG4gICAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgICAgZGlzdDogaW5mb3MuZGlzdC50YXJiYWxsLFxuICAgICAgICAgIHNlcTogc2VxLnRvRml4ZWQoKSxcbiAgICAgICAgfSxcbiAgICAgICAgdGltZTogbW9kaWZpZWQudG9VVENTdHJpbmcoKSxcbiAgICAgIH07XG4gICAgICBjb25zdCBtZXNzYWdlOiBJbmdlc3Rpb25JbnB1dCA9IHtcbiAgICAgICAgLi4ubWVzc2FnZUJhc2UsXG4gICAgICAgIGludGVncml0eTogaW50ZWdyaXR5KG1lc3NhZ2VCYXNlLCB0YXJiYWxsKSxcbiAgICAgIH07XG5cbiAgICAgIC8vIFNlbmQgdGhlIFNRUyBtZXNzYWdlIG91dFxuICAgICAgYXdhaXQgYXdzLnNxcygpLnNlbmRNZXNzYWdlKHtcbiAgICAgICAgTWVzc2FnZUJvZHk6IEpTT04uc3RyaW5naWZ5KG1lc3NhZ2UsIG51bGwsIDIpLFxuICAgICAgICBRdWV1ZVVybDogcXVldWVVcmwsXG4gICAgICB9KS5wcm9taXNlKCk7XG5cbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuU1RBR0lOR19GQUlMVVJFX0NPVU5ULCAwLCBVbml0LkNvdW50KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIC8vIFNvbWV0aGluZyBmYWlsZWQsIHN0b3JlIHRoZSBwYXlsb2FkIGluIHRoZSBwcm9ibGVtIHByZWZpeCwgYW5kIG1vdmUgb24uXG4gICAgICBjb25zb2xlLmVycm9yKGBbJHtzZXF9XSBGYWlsZWQgcHJvY2Vzc2luZywgbG9nZ2luZyBlcnJvciB0byBTMyBhbmQgcmVzdW1pbmcgd29yay4gJHtpbmZvcy5uYW1lfUAke2luZm9zLnZlcnNpb259OiAke2Vycn1gKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuU1RBR0lOR19GQUlMVVJFX0NPVU5ULCAxLCBVbml0LkNvdW50KTtcbiAgICAgIGF3YWl0IHB1dE9iamVjdChgJHtTM0tleVByZWZpeC5GQUlMRURfS0VZX1BSRUZJWH0ke3NlcX1gLCBKU09OLnN0cmluZ2lmeSh7IC4uLmluZm9zLCBfY29uc3RydWN0X2h1Yl9mYWlsdXJlX3JlYXNvbjogZXJyIH0sIG51bGwsIDIpLCB7XG4gICAgICAgIENvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vanNvbicsXG4gICAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICAgJ01vZGlmaWVkLUF0JzogbW9kaWZpZWQudG9JU09TdHJpbmcoKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuICAvLyNlbmRyZWdpb25cblxuICAvLyNyZWdpb24gQXN5bmNocm9ub3VzIFByaW1pdGl2ZXNcbiAgLyoqXG4gICAqIE1ha2VzIGFuIEhUVFAgR0VUIHJlcXVlc3QsIGFuZCByZXR1cm5zIHRoZSByZXN1bHRpbmcgcGF5bG9hZC5cbiAgICpcbiAgICogQHBhcmFtIHVybCB0aGUgVVJMIHRvIGdldC5cbiAgICpcbiAgICogQHJldHVybnMgYSBCdWZmZXIgY29udGFpbmluZyB0aGUgcmVjZWl2ZWQgZGF0YS5cbiAgICovXG4gIGZ1bmN0aW9uIGh0dHBHZXQodXJsOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gbmV3IFByb21pc2U8QnVmZmVyPigob2ssIGtvKSA9PiB7XG4gICAgICBodHRwcy5nZXQodXJsLCAocmVzcG9uc2UpID0+IHtcbiAgICAgICAgaWYgKHJlc3BvbnNlLnN0YXR1c0NvZGUgIT09IDIwMCkge1xuICAgICAgICAgIGtvKG5ldyBFcnJvcihgVW5zdWNjZXNzZnVsIEdFVDogJHtyZXNwb25zZS5zdGF0dXNDb2RlfSAtICR7cmVzcG9uc2Uuc3RhdHVzTWVzc2FnZX1gKSk7XG4gICAgICAgIH1cblxuICAgICAgICBsZXQgYm9keSA9IEJ1ZmZlci5hbGxvYygwKTtcbiAgICAgICAgcmVzcG9uc2Uub24oJ2RhdGEnLCAoY2h1bmspID0+IGJvZHkgPSBCdWZmZXIuY29uY2F0KFtib2R5LCBCdWZmZXIuZnJvbShjaHVuayldKSk7XG4gICAgICAgIHJlc3BvbnNlLm9uY2UoJ2Nsb3NlJywgKCkgPT4gb2soYm9keSkpO1xuICAgICAgICByZXNwb25zZS5vbmNlKCdlcnJvcicsIGtvKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFB1dHMgYW4gb2JqZWN0IGluIHRoZSBzdGFnaW5nIGJ1Y2tldCwgd2l0aCBzdGFuZGFyZGl6ZWQgb2JqZWN0IG1ldGFkYXRhLlxuICAgKlxuICAgKiBAcGFyYW0ga2V5ICB0aGUga2V5IGZvciB0aGUgb2JqZWN0IHRvIGJlIHB1dC5cbiAgICogQHBhcmFtIGJvZHkgdGhlIGJvZHkgb2YgdGhlIG9iamVjdCB0byBiZSBwdXQuXG4gICAqIEBwYXJhbSBvcHRzIGFueSBvdGhlciBvcHRpb25zIHRvIHVzZSB3aGVuIHNlbmRpbmcgdGhlIFMzIHJlcXVlc3QuXG4gICAqXG4gICAqIEByZXR1cm5zIHRoZSByZXN1bHQgb2YgdGhlIFMzIHJlcXVlc3QuXG4gICAqL1xuICBmdW5jdGlvbiBwdXRPYmplY3Qoa2V5OiBzdHJpbmcsIGJvZHk6IEFXUy5TMy5Cb2R5LCBvcHRzOiBPbWl0PEFXUy5TMy5QdXRPYmplY3RSZXF1ZXN0LCAnQnVja2V0JyB8ICdLZXknIHwgJ0JvZHknPiA9IHt9KSB7XG4gICAgcmV0dXJuIGF3cy5zMygpLnB1dE9iamVjdCh7XG4gICAgICBCdWNrZXQ6IHN0YWdpbmdCdWNrZXQsXG4gICAgICBLZXk6IGtleSxcbiAgICAgIEJvZHk6IGJvZHksXG4gICAgICBNZXRhZGF0YToge1xuICAgICAgICAnTGFtYmRhLUxvZy1Hcm91cCc6IGNvbnRleHQubG9nR3JvdXBOYW1lLFxuICAgICAgICAnTGFtYmRhLUxvZy1TdHJlYW0nOiBjb250ZXh0LmxvZ1N0cmVhbU5hbWUsXG4gICAgICAgICdMYW1iZGEtUnVuLUlkJzogY29udGV4dC5hd3NSZXF1ZXN0SWQsXG4gICAgICAgIC4uLm9wdHMuTWV0YWRhdGEsXG4gICAgICB9LFxuICAgICAgLi4ub3B0cyxcbiAgICB9KS5wcm9taXNlKCk7XG4gIH1cbiAgLy8jZW5kcmVnaW9uXG59XG5cbi8qKlxuICogT2J0YWlucyB0aGUgYFZlcnNpb25JbmZvYCBjb3JyZXNwb25kaW5nIHRvIHRoZSBtb2RpZmllZCB2ZXJzaW9uKHMpIGluIHRoZVxuICogcHJvdmlkZWQgYENoYW5nZWAgb2JqZWN0cywgZW5zdXJlcyB0aGV5IGFyZSByZWxldmFudCAoY29uc3RydWN0IGxpYnJhcmllcyksXG4gKiBhbmQgcmV0dXJucyB0aG9zZSBvbmx5LlxuICpcbiAqIEBwYXJhbSBjaGFuZ2VzIHRoZSBjaGFuZ2VzIHRvIGJlIHByb2Nlc3NlZC5cbiAqIEBwYXJhbSBtZXRyaWNzIHRoZSBtZXRyaWNzIGxvZ2dlciB0byB1c2UuXG4gKiBAcGFyYW0gZGVueUxpc3QgZGVueSBsaXN0IGNsaWVudFxuICpcbiAqIEByZXR1cm5zIGEgbGlzdCBvZiBgVmVyc2lvbkluZm9gIG9iamVjdHNcbiAqL1xuZnVuY3Rpb24gZ2V0UmVsZXZhbnRWZXJzaW9uSW5mb3MoXG4gIGNoYW5nZXM6IHJlYWRvbmx5IENoYW5nZVtdLFxuICBtZXRyaWNzOiBNZXRyaWNzTG9nZ2VyLFxuICBkZW55TGlzdDogRGVueUxpc3RDbGllbnQsXG4gIGxpY2Vuc2VMaXN0OiBMaWNlbnNlTGlzdENsaWVudCxcbik6IHJlYWRvbmx5IFVwZGF0ZWRWZXJzaW9uW10ge1xuXG4gIGNvbnN0IHJlc3VsdCA9IG5ldyBBcnJheTxVcGRhdGVkVmVyc2lvbj4oKTtcblxuICBmb3IgKGNvbnN0IGNoYW5nZSBvZiBjaGFuZ2VzKSB7XG4gICAgLy8gRmlsdGVyIG91dCBhbGwgZWxlbWVudHMgdGhhdCBkb24ndCBoYXZlIGEgXCJuYW1lXCIgaW4gdGhlIGRvY3VtZW50LCBhc1xuICAgIC8vIHRoZXNlIGFyZSBzY2hlbWFzLCB3aGljaCBhcmUgbm90IHJlbGV2YW50IHRvIG91ciBidXNpbmVzcyBoZXJlLlxuICAgIGlmIChjaGFuZ2UuZG9jLm5hbWUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgY29uc29sZS5lcnJvcihgWyR7Y2hhbmdlLnNlcX1dIENoYW5nZWQgZG9jdW1lbnQgY29udGFpbnMgbm8gJ25hbWUnOiAke2NoYW5nZS5pZH1gKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuVU5QUk9DRVNTQUJMRV9FTlRJVFksIDEsIFVuaXQuQ291bnQpO1xuICAgICAgY29udGludWU7XG4gICAgfVxuXG4gICAgLy8gVGhlIG5vcm1hbGl6ZSBmdW5jdGlvbiBjaGFuZ2UgdGhlIG9iamVjdCBpbiBwbGFjZSwgaWYgdGhlIGRvYyBvYmplY3QgaXMgaW52YWxpZCBpdCB3aWxsIHJldHVybiB1bmRlZmluZWRcbiAgICBpZiAobm9ybWFsaXplTlBNTWV0YWRhdGEoY2hhbmdlLmRvYykgPT09IHVuZGVmaW5lZCkge1xuICAgICAgY29uc29sZS5lcnJvcihgWyR7Y2hhbmdlLnNlcX1dIENoYW5nZWQgZG9jdW1lbnQgaW52YWxpZCwgbnBtIG5vcm1hbGl6ZSByZXR1cm5lZCB1bmRlZmluZWQ6ICR7Y2hhbmdlLmlkfWApO1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5VTlBST0NFU1NBQkxFX0VOVElUWSwgMSwgVW5pdC5Db3VudCk7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG5cbiAgICAvLyBTb21ldGltZXMsIHRoZXJlIGFyZSBubyB2ZXJzaW9ucyBpbiB0aGUgZG9jdW1lbnQuIFdlIHNraXAgdGhvc2UuXG4gICAgaWYgKGNoYW5nZS5kb2MudmVyc2lvbnMgPT0gbnVsbCkge1xuICAgICAgY29uc29sZS5lcnJvcihgWyR7Y2hhbmdlLnNlcX1dIENoYW5nZWQgZG9jdW1lbnQgY29udGFpbnMgbm8gJ3ZlcnNpb25zJzogJHtjaGFuZ2UuaWR9YCk7XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlVOUFJPQ0VTU0FCTEVfRU5USVRZLCAxLCBVbml0LkNvdW50KTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cblxuICAgIC8vIFNvbWV0aW1lcywgdGhlcmUgaXMgbm8gJ3RpbWUnIGVudHJ5IGluIHRoZSBkb2N1bWVudC4gV2Ugc2tpcCB0aG9zZS5cbiAgICBpZiAoY2hhbmdlLmRvYy50aW1lID09IG51bGwpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoYFske2NoYW5nZS5zZXF9XSBDaGFuZ2VkIGRvY3VtZW50IGNvbnRhaW5zIG5vICd0aW1lJzogJHtjaGFuZ2UuaWR9YCk7XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlVOUFJPQ0VTU0FCTEVfRU5USVRZLCAxLCBVbml0LkNvdW50KTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cblxuICAgIC8vIEdldCB0aGUgbGFzdCBtb2RpZmljYXRpb24gZGF0ZSBmcm9tIHRoZSBjaGFuZ2VcbiAgICBjb25zdCBzb3J0ZWRVcGRhdGVzID0gT2JqZWN0LmVudHJpZXMoY2hhbmdlLmRvYy50aW1lKVxuICAgICAgLy8gSWdub3JlIHRoZSBcImNyZWF0ZWRcIiBhbmQgXCJtb2RpZmllZFwiIGtleXMgaGVyZVxuICAgICAgLmZpbHRlcigoW2tleV0pID0+IGtleSAhPT0gJ2NyZWF0ZWQnICYmIGtleSAhPT0gJ21vZGlmaWVkJylcbiAgICAgIC8vIFBhcnNlIGFsbCB0aGUgZGF0ZXMgdG8gZW5zdXJlIHRoZXkgYXJlIGNvbXBhcmFibGVcbiAgICAgIC5tYXAoKFt2ZXJzaW9uLCBpc29EYXRlXSkgPT4gW3ZlcnNpb24sIG5ldyBEYXRlKGlzb0RhdGUpXSBhcyBjb25zdClcbiAgICAgIC8vIFNvcnQgYnkgZGF0ZSwgZGVzY2VuZGluZ1xuICAgICAgLnNvcnQoKFssIGxdLCBbLCByXSkgPT4gci5nZXRUaW1lKCkgLSBsLmdldFRpbWUoKSk7XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5QQUNLQUdFX1ZFUlNJT05fQ09VTlQsIHNvcnRlZFVwZGF0ZXMubGVuZ3RoLCBVbml0LkNvdW50KTtcblxuICAgIGxldCBsYXRlc3RNb2RpZmllZDogRGF0ZSB8IHVuZGVmaW5lZDtcbiAgICBmb3IgKGNvbnN0IFt2ZXJzaW9uLCBtb2RpZmllZF0gb2Ygc29ydGVkVXBkYXRlcykge1xuICAgICAgaWYgKGxhdGVzdE1vZGlmaWVkID09IG51bGwgfHwgbGF0ZXN0TW9kaWZpZWQuZ2V0VGltZSgpID09PSBtb2RpZmllZC5nZXRUaW1lKCkpIHtcbiAgICAgICAgY29uc3QgaW5mb3MgPSBjaGFuZ2UuZG9jLnZlcnNpb25zW3ZlcnNpb25dO1xuICAgICAgICBpZiAoaW5mb3MgPT0gbnVsbCkge1xuICAgICAgICAgIC8vIENvdWxkIGJlIHRoZSB2ZXJzaW9uIGluIHF1ZXN0aW9uIHdhcyB1bi1wdWJsaXNoZWQuXG4gICAgICAgICAgY29uc29sZS5sb2coYFske2NoYW5nZS5zZXF9XSBDb3VsZCBub3QgZmluZCBpbmZvIGZvciBcIiR7Y2hhbmdlLmRvYy5uYW1lfUAke3ZlcnNpb259XCIuIFdhcyBpdCB1bi1wdWJsaXNoZWQ/YCk7XG4gICAgICAgIH0gZWxzZSBpZiAoaXNDb25zdHJ1Y3RMaWJyYXJ5KGluZm9zKSkge1xuXG4gICAgICAgICAgLy8gc2tpcCBpZiB0aGlzIHBhY2thZ2UgaXMgZGVuaWVkXG4gICAgICAgICAgY29uc3QgZGVuaWVkID0gZGVueUxpc3QubG9va3VwKGluZm9zLm5hbWUsIGluZm9zLnZlcnNpb24pO1xuICAgICAgICAgIGlmIChkZW5pZWQpIHtcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKGBbJHtjaGFuZ2Uuc2VxfV0gUGFja2FnZSBkZW5pZWQ6ICR7SlNPTi5zdHJpbmdpZnkoZGVuaWVkKX1gKTtcbiAgICAgICAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuREVOWV9MSVNURURfQ09VTlQsIDEsIFVuaXQuQ291bnQpO1xuICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5QQUNLQUdFX1ZFUlNJT05fQUdFLCBEYXRlLm5vdygpIC0gbW9kaWZpZWQuZ2V0VGltZSgpLCBVbml0Lk1pbGxpc2Vjb25kcyk7XG4gICAgICAgICAgY29uc3QgaXNFbGlnaWJsZSA9IGxpY2Vuc2VMaXN0Lmxvb2t1cChpbmZvcy5saWNlbnNlID8/ICdVTkxJQ0VOU0VEJykgIT0gbnVsbDtcbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLklORUxJR0lCTEVfTElDRU5TRSwgaXNFbGlnaWJsZSA/IDAgOiAxLCBVbml0LkNvdW50KTtcbiAgICAgICAgICBpZiAoaXNFbGlnaWJsZSkge1xuICAgICAgICAgICAgcmVzdWx0LnB1c2goeyBpbmZvcywgbW9kaWZpZWQsIHNlcTogY2hhbmdlLnNlcSB9KTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgY29uc29sZS5sb2coYFske2NoYW5nZS5zZXF9XSBQYWNrYWdlIFwiJHtjaGFuZ2UuZG9jLm5hbWV9QCR7dmVyc2lvbn1cIiBkb2VzIG5vdCB1c2UgYWxsb3ctbGlzdGVkIGxpY2Vuc2U6ICR7aW5mb3MubGljZW5zZSA/PyAnVU5MSUNFTlNFRCd9YCk7XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIGNvbnNvbGUubG9nKGBbJHtjaGFuZ2Uuc2VxfV0gSWdub3JpbmcgXCIke2NoYW5nZS5kb2MubmFtZX1AJHt2ZXJzaW9ufVwiIGFzIGl0IGlzIG5vdCBhIGNvbnN0cnVjdCBsaWJyYXJ5LmApO1xuICAgICAgICB9XG4gICAgICAgIGxhdGVzdE1vZGlmaWVkID0gbW9kaWZpZWQ7XG4gICAgICB9XG4gICAgfVxuICB9XG4gIHJldHVybiByZXN1bHQ7XG5cbiAgZnVuY3Rpb24gaXNDb25zdHJ1Y3RMaWJyYXJ5KGluZm9zOiBWZXJzaW9uSW5mbyk6IGJvb2xlYW4ge1xuICAgIGlmIChpbmZvcy5qc2lpID09IG51bGwpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gICAgcmV0dXJuIGluZm9zLm5hbWUgPT09ICdjb25zdHJ1Y3QnXG4gICAgICB8fCBpbmZvcy5uYW1lID09PSAnYXdzLWNkay1saWInXG4gICAgICB8fCBpbmZvcy5uYW1lLnN0YXJ0c1dpdGgoJ0Bhd3MtY2RrJylcbiAgICAgIHx8IGluZm9zLmtleXdvcmRzPy5zb21lKChrdykgPT4gQ09OU1RSVUNUX0tFWVdPUkRTLmhhcyhrdykpXG4gICAgICB8fCBPYmplY3Qua2V5cyhpbmZvcy5kZXBlbmRlbmNpZXMgPz8ge30pLnNvbWUoaXNDb25zdHJ1Y3RGcmFtZXdvcmtQYWNrYWdlKVxuICAgICAgfHwgT2JqZWN0LmtleXMoaW5mb3MuZGV2RGVwZW5kZW5jaWVzID8/IHt9KS5zb21lKGlzQ29uc3RydWN0RnJhbWV3b3JrUGFja2FnZSlcbiAgICAgIHx8IE9iamVjdC5rZXlzKGluZm9zLnBlZXJEZXBlbmRlbmNpZXMgPz8ge30pLnNvbWUoaXNDb25zdHJ1Y3RGcmFtZXdvcmtQYWNrYWdlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQYWNrYWdlIGlzIG9uZSBvZiB0aGUga25vd24gY29uc3RydWN0IGZyYW1ld29yaydzIGZpcnN0IHBhcnR5IHBhY2thZ2VzOlxuICAgKiAtIEBhd3MtY2RrLypcbiAgICogLSBAY2RrdGYvKlxuICAgKiAtIGNkazhzIG9yIGNkazhzLXBsdXNcbiAgICovXG4gIGZ1bmN0aW9uIGlzQ29uc3RydWN0RnJhbWV3b3JrUGFja2FnZShuYW1lOiBzdHJpbmcpOiBib29sZWFuIHtcbiAgICAvLyBOT1RFOiBQcmVmaXggbWF0Y2hpbmcgc2hvdWxkIG9ubHkgYmUgdXNlZCBmb3IgQHNjb3BlLyBuYW1lcy5cbiAgICByZXR1cm4gbmFtZS5zdGFydHNXaXRoKCdAYXdzLWNkay8nKVxuICAgICAgfHwgbmFtZS5zdGFydHNXaXRoKCdAY2RrdGYvJylcbiAgICAgIHx8IG5hbWUgPT09ICdjZGs4cydcbiAgICAgIHx8IG5hbWUgPT09ICdjZGs4cy1wbHVzJztcbiAgfVxufVxuXG4vKipcbiAgKiBUaGUgc2NoZW1lIG9mIGEgcGFja2FnZSB2ZXJzaW9uIGluIHRoZSB1cGRhdGUuIEluY2x1ZGVzIHRoZSBwYWNrYWdlLmpzb24ga2V5cywgYXMgd2VsbCBhcyBzb21lIGFkZGl0aW9uYWwgbnBtIG1ldGFkYXRhXG4gICogQHNlZSBodHRwczovL2dpdGh1Yi5jb20vbnBtL3JlZ2lzdHJ5L2Jsb2IvbWFzdGVyL2RvY3MvUkVHSVNUUlktQVBJLm1kI3ZlcnNpb25cbiAgKi9cbmludGVyZmFjZSBWZXJzaW9uSW5mbyB7XG4gIHJlYWRvbmx5IGRlcGVuZGVuY2llcz86IHsgcmVhZG9ubHkgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuICByZWFkb25seSBkZXZEZXBlbmRlbmNpZXM/OiB7IHJlYWRvbmx5IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcmVhZG9ubHkgcGVlckRlcGVuZGVuY2llcz86IHsgcmVhZG9ubHkgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuICByZWFkb25seSBqc2lpOiB1bmtub3duO1xuICByZWFkb25seSBsaWNlbnNlPzogc3RyaW5nO1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IFtrZXk6IHN0cmluZ106IHVua25vd247XG4gIHJlYWRvbmx5IGtleXdvcmRzOiBzdHJpbmdbXTtcbiAgcmVhZG9ubHkgZGlzdDoge1xuICAgIHJlYWRvbmx5IHNoYXN1bTogc3RyaW5nO1xuICAgIHJlYWRvbmx5IHRhcmJhbGw6IHN0cmluZztcbiAgfTtcbiAgcmVhZG9ubHkgdmVyc2lvbjogc3RyaW5nO1xufVxuXG5pbnRlcmZhY2UgVXBkYXRlZFZlcnNpb24ge1xuICAvKipcbiAgICogVGhlIGBWZXJzaW9uSW5mb2AgZm9yIHRoZSBtb2RpZmllZCBwYWNrYWdlIHZlcnNpb24uXG4gICAqL1xuICByZWFkb25seSBpbmZvczogVmVyc2lvbkluZm87XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIGF0IHdoaWNoIHRoZSBgVmVyc2lvbkluZm9gIHdhcyBsYXN0IG1vZGlmaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9kaWZpZWQ6IERhdGU7XG5cbiAgLyoqXG4gICAqIFRoZSBDb3VjaERCIHRyYW5zYWN0aW9uIG51bWJlciBmb3IgdGhlIHVwZGF0ZS5cbiAgICovXG4gIHJlYWRvbmx5IHNlcTogbnVtYmVyO1xufVxuXG5pbnRlcmZhY2UgRG9jdW1lbnQge1xuXG4gIC8qKlxuICAgKiBhIExpc3Qgb2YgYWxsIFZlcnNpb24gb2JqZWN0cyBmb3IgdGhlIHBhY2thZ2VcbiAgICovXG4gIHJlYWRvbmx5IHZlcnNpb25zOiB7IFtrZXk6c3RyaW5nXTogVmVyc2lvbkluZm8gfCB1bmRlZmluZWQgfTtcblxuICAvKipcbiAgICogVGhlIHBhY2thZ2UncyBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaW1lc3RhbXBzIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGRvY3VtZW50LiBUaGUgdmFsdWVzIGFyZSBJU08tODYwMSBlbmNvZGVkXG4gICAqIHRpbWVzdGFtcHMuXG4gICAqL1xuICByZWFkb25seSB0aW1lOiB7XG4gICAgcmVhZG9ubHkgY3JlYXRlZDogc3RyaW5nO1xuICAgIHJlYWRvbmx5IG1vZGlmaWVkOiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgW3ZlcnNpb246IHN0cmluZ106IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIENoYW5nZSB7XG4gIHJlYWRvbmx5IHNlcTogbnVtYmVyO1xuICByZWFkb25seSBkb2M6IERvY3VtZW50O1xuICByZWFkb25seSBpZDogc3RyaW5nO1xuICByZWFkb25seSBkZWxldGVkOiBib29sZWFuO1xufVxuIl19