import datetime
from files_sdk.models.public_ip_address import PublicIpAddress
from files_sdk.api import Api
from files_sdk.exceptions import InvalidParameterError, MissingParameterError, NotImplementedError

class IpAddress:
    default_attributes = {
        'id': None,     # string - Unique label for list; used by Zapier and other integrations.
        'associated_with': None,     # string - The object that this public IP address list is associated with.
        'group_id': None,     # int64 - Group ID
        'ip_addresses': None,     # array - A list of IP addresses.
    }

    def __init__(self, attributes={}, options={}):
        self.set_attributes(attributes)
        self.options = options

    def set_attributes(self, attributes):
        for (attribute, default_value) in IpAddress.default_attributes.items():
            setattr(self, attribute, attributes.get(attribute, default_value))

    def get_attributes(self):
        return {k: getattr(self, k, None) for k in IpAddress.default_attributes if getattr(self, k, None) is not None}


# Parameters:
#   page - int64 - Current page number.
#   per_page - int64 - Number of records to show per page.  (Max: 10,000, 1,000 or less is recommended).
#   action - string - Deprecated: If set to `count` returns a count of matching records rather than the records themselves.
def list(params = {}, options = {}):
    if "page" in params and not isinstance(params["page"], int):
        raise InvalidParameterError("Bad parameter: page must be an int")
    if "per_page" in params and not isinstance(params["per_page"], int):
        raise InvalidParameterError("Bad parameter: per_page must be an int")
    if "action" in params and not isinstance(params["action"], str):
        raise InvalidParameterError("Bad parameter: action must be an str")
    response, options = Api.send_request("GET", "/ip_addresses", params, options)
    return [ IpAddress(entity_data, options) for entity_data in response.data ]

def all(params = {}, options = {}):
    list(params, options)

# Parameters:
#   page - int64 - Current page number.
#   per_page - int64 - Number of records to show per page.  (Max: 10,000, 1,000 or less is recommended).
#   action - string - Deprecated: If set to `count` returns a count of matching records rather than the records themselves.
def get_reserved(params = {}, options = {}):
    if "page" in params and not isinstance(params["page"], int):
        raise InvalidParameterError("Bad parameter: page must be an int")
    if "per_page" in params and not isinstance(params["per_page"], int):
        raise InvalidParameterError("Bad parameter: per_page must be an int")
    if "action" in params and not isinstance(params["action"], str):
        raise InvalidParameterError("Bad parameter: action must be an str")
    response, options = Api.send_request("GET", "/ip_addresses/reserved", params, options)
    return [ PublicIpAddress(entity_data, options) for entity_data in response.data ]

def new(*args, **kwargs):
    return IpAddress(*args, **kwargs)