import click
from odoo_backup.models import Server
import os
from odoo_backup.config import DATA_DIR
from odoo_backup.server import create_server


if not os.path.isdir(os.path.join(DATA_DIR)):
    os.mkdir(os.path.join(DATA_DIR))


@click.group()
def cli():
    """Simple Odoo backup manager"""
    pass


@cli.command()
def list():
    """Lists all available servers"""
    for server in Server.all():
        print("{} ({})".format(server.name, server.ip))


@cli.command()
@click.argument('server_name')
def check(server_name):
    """Checks a server (or all servers) connection and available backups"""
    if server_name == "all":
        for server in Server.all():
            server.check()
    else:
        Server.get(server_name).check()


@cli.command()
@click.argument('server_name')
@click.option('--database', default='all', help="database to backup")
def backup(server_name, database):
    """Backups a server (or all servers) databases (on only one using --database option)"""
    if server_name == "all":
        for server in Server.all():
            server.backup_all()
    else:
        server = Server.get(server_name)
        if database != "all":
            server.backup(database)
        else:
            server.backup_all()


@cli.command()
@click.argument('server_name')
@click.argument('database_name')
@click.option('--backup-date', default='latest', help='backup date (latest, oldest or yyyy-mm-dd format')
def download(server_name, database_name, backup_date):
    """Provides an URL to download specified backup"""
    server = Server.get(server_name)
    database = server.databases.get(database_name)
    if database is None:
        print("Database {server_name}:{database_name} doesn't exist in config file.")
        exit(1)
    file_name = server.get_backup_name(database_name, backup_date)
    if file_name is None:
        print('Unable to find {}:{}:{} backup.'.format(server_name, database_name, backup_date))
        exit(0)
    print('[ INFO ] Setting up web server for backup file download. Please use CTRL+C after downloading.')
    server = create_server(server_name=server_name, database_name=database_name, file_name=file_name)
    server.run()


if __name__ == "__main__":
    cli()
