"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const monitored_certificate_1 = require("../monitored-certificate");
const config_1 = require("./config");
const home_response_function_1 = require("./home-response-function");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const indexFunctionId = `IndexHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_DISABLED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        const indexBehaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_DISABLED,
            functionAssociations: [{
                    function: new home_response_function_1.HomeResponseFunction(this, indexFunctionId, {
                        functionName: indexFunctionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // The base URL is currently the custom DNS if any was used, or the distribution domain name.
        // This needs changing in case, for example, we add support for a custom URL prefix.
        this.baseUrl = `https://${props.domain ? props.domain.zone.zoneName : this.distribution.distributionDomainName}`;
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        const websiteOrigin = new origins.S3Origin(this.bucket);
        this.distribution.addBehavior('/index.html', websiteOrigin, indexBehaviorOptions);
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            cacheControl: [
                aws_s3_deployment_1.CacheControl.setPublic(),
                aws_s3_deployment_1.CacheControl.maxAge(core_1.Duration.hours(1)),
                aws_s3_deployment_1.CacheControl.mustRevalidate(),
                aws_s3_deployment_1.CacheControl.sMaxAge(core_1.Duration.minutes(5)),
                aws_s3_deployment_1.CacheControl.proxyRevalidate(),
            ],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            sources: [s3deploy.Source.asset(webappDir)],
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
            categories: props.categories,
        });
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            cacheControl: [
                aws_s3_deployment_1.CacheControl.setPublic(),
                aws_s3_deployment_1.CacheControl.maxAge(core_1.Duration.hours(1)),
                aws_s3_deployment_1.CacheControl.mustRevalidate(),
                aws_s3_deployment_1.CacheControl.sMaxAge(core_1.Duration.minutes(5)),
                aws_s3_deployment_1.CacheControl.proxyRevalidate(),
            ],
            sources: [s3deploy.Source.asset(config.file.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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