# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['glovo_api_python', 'glovo_api_python.constants', 'glovo_api_python.resources']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.22.0,<3.0.0']

setup_kwargs = {
    'name': 'glovo-api-python',
    'version': '2.0.1',
    'description': 'Glovo API Python SDK',
    'long_description': '[![Python Versions](https://img.shields.io/pypi/pyversions/glovo-api-python.svg?color=3776AB&logo=python&logoColor=white)](https://www.python.org/)\n[![PyPI Version](https://img.shields.io/pypi/v/glovo-api-python.svg?color=blue&logo=pypi&logoColor=white)](https://pypi.org/project/glovo-api-python/)\n[![PyPI Downloads](https://img.shields.io/pypi/dm/glovo-api-python.svg?color=blue&logo=pypi&logoColor=white)](https://pypi.org/project/glovo-api-python/)\n\n[![Build Status](https://travis-ci.org/softbutterfly/glovo-api-python.svg?branch=master)](https://travis-ci.org/softbutterfly/glovo-api-python)\n[![codecov](https://codecov.io/gh/softbutterfly/glovo-api-python/branch/master/graph/badge.svg)](https://codecov.io/gh/softbutterfly/glovo-api-python)\n[![Codacy Badge](https://api.codacy.com/project/badge/Grade/1787644ed8524433b9466f74d24b44d2)](https://www.codacy.com/gh/softbutterfly/glovo-api-python?utm_source=github.com&utm_medium=referral&utm_content=softbutterfly/glovo-api-python&utm_campaign=Badge_Grade)\n[![Codacy Badge](https://api.codacy.com/project/badge/Coverage/1787644ed8524433b9466f74d24b44d2)](https://www.codacy.com/gh/softbutterfly/glovo-api-python?utm_source=github.com&utm_medium=referral&utm_content=softbutterfly/glovo-api-python&utm_campaign=Badge_Coverage)\n[![Requirements Status](https://requires.io/github/softbutterfly/glovo-api-python/requirements.svg?branch=master)](https://requires.io/github/softbutterfly/glovo-api-python/requirements/?branch=master)\n\n[![Stars](https://img.shields.io/github/stars/softbutterfly/glovo-api-python?logo=github)](https://github.com/softbutterfly/glovo-api-python/)\n[![License](https://img.shields.io/pypi/l/glovo-api-python.svg?color=blue)](https://github.com/softbutterfly/glovo-api-python/blob/master/LICENSE.txt)\n\n# Glovo Python Business API\n\nCreate, retrieve and track your Glovo orders trough their Business API.\n\n## Requirements\n\nThis package requires at least\n\n- requests 2.21.0\n- python 3.7\n\nThis package was not tested with prior versions of these packages but it can works as well.\n\n## Install\n\nYou can install via pip. Run the following command:\n\n```bash\npip install glovo-api-python\n```\n\n## Credentials\n\nCreate an account in Glovo (it can be created from the App). This api **needs a credit card associated to your account**. You can add one from your app and it will be used automatically for any order. In order to get your API credentials you should login in the desired environment and go to _Credentials_ section on your profile.\n\n- [B2B Production](https://business.glovoapp.com/dashboard/profile)\n- [B2B Sandbox/Test](https://business.testglovo.com/dashboard/profile)\n\nExample `ApiKey` & `ApiSecret`:\n\n```python\napi_key = \'155761234946286\'\napi_secret = \'767b4e2f48e4412d95a6eb1234bdc78b\'\n```\n\n## Usage\n\nSimple usage looks like:\n\n### Initialize client\n\n```python\nfrom glovo_api_python.client import Glovo\n\napi_key = \'sample_api_key\'\napi_secret = \'sample_api_secret\'\n\nclient = Glovo(api_key, api_secret)\n```\n\n### Get working areas\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#getworkingareas)\n\n```python\nwoking_areas = client.working_area.list()\nprint(working_areas)\n# Will show this\n# {\n#     \'status\': 200,\n#     \'data\': {\n#         \'workingAreas\': [\n#             {\n#                 \'code\': \'ABJ\',\n#                 \'polygons\': [\n#                     \'<ENCODED POLYLINE>\',\n#                     \'<ENCODED POLYLINE>\'\n#                 ],\n#                 \'workingTime\': {\n#                     \'from\': \'09:00\',\n#                     \'duration\': 120\n#                 }\n#             }\n#             <OTHER WORKING AREAS>\n#         ]\n#     }\n# }\n```\n\n### Estimate order price\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#estimateorderprice)\n\n```python\npickup_address = {\n    "lat": -12.0563673,\n    "lon": -76.9733736,\n    "type": "PICKUP",\n    "label": "Avenida los Cipreses, 140",\n    "details": "Edificio Orbes, Piso 3, Oficina de Productos Angel Breña",\n    "contactPhone": None,\n    "contactPerson": None\n}\n\ndelivery_address = {\n    "lat": -12.055013,\n    "lon": -77.03845849999999,\n    "type": "DELIVERY",\n    "label": "Avenida Inca Garcilaso de la Vega, 1250",\n    "details": "Oficina 511",\n    "contactPhone": None,\n    "contactPerson": None\n}\n\nestimates_order_price = client.order.estimate({\n    "scheduleTime": None,\n    "description": "Some useful description",\n    "addresses": [\n        pickup_address,\n        delivery_address\n    ]\n})\nprint(estimates_order_price)\n# Will show this\n# {\n#     \'status\': 200,\n#     \'data\': {\n#         \'total\': {\n#             \'amount\': 1260,\n#             \'currency\': \'PEN\'\n#         }\n#     }\n# }\n```\n\n### Create order\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#createorder)\n\n```python\npickup_address = {\n    "lat": -12.0563673,\n    "lon": -76.9733736,\n    "type": "PICKUP",\n    "label": "Avenida los Cipreses, 140",\n    "details": "Edificio Orbes, Piso 3, Oficina de Productos Angel Breña",\n    "contactPhone": None,\n    "contactPerson": None\n}\n\ndelivery_address = {\n    "lat": -12.055013,\n    "lon": -77.03845849999999,\n    "type": "DELIVERY",\n    "label": "Avenida Inca Garcilaso de la Vega, 1250",\n    "details": "Oficina 511",\n    "contactPhone": None,\n    "contactPerson": None\n}\n\nplaced_order = client.order.create({\n    "scheduleTime": 12344566, # Set to None for immediate order\n    "description": "Some useful description",\n    "addresses": [\n        pickup_address,\n        delivery_address\n    ]\n})\n\nprint(placed_order)\n# Will show this\n# {\n#     \'status\': 200,\n#     \'data\': {\n#         "id": 123456789,\n#         "state": "SCHEDULED",\n#         "scheduleTime": 12344566,\n#         "description": "A 30cm by 30cm box",\n#         "addresses": [\n#             <PICKUP ADDRESS>,\n#             <DELIVERY ADDRESS>,\n#         ]\n#     }\n# }\n```\n\n### Retrieve order\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#retrieveorder)\n\n```python\norder_id = 32678866\nplaced_order = client.order.read(order_id)\n\nprint(placed_order)\n# Will show this\n# {\n#     \'status\': 200,\n#     \'data\': {\n#         \'scheduleTime\': None,\n#         \'description\': \'Necesito enviar una llave\',\n#         \'addresses\': [\n#             <PICKUP ADDRESS>,\n#             <DELIVERY ADDRESS>,\n#         ],\n#         \'id\': \'32678866\',\n#         \'state\': \'DELIVERED\',\n#         \'reference\': None\n#     }\n# }\n```\n\n### Get order tracking\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#getordertracking)\n\n```python\norder_id = 32678866\ntracking = client.order.tracking(order_id)\nprint(tracking)\n# Will show this\n# {\n#     "status": 200,\n#     "data": {\n#         "lat": -12.0704984,\n#         "lon": -76.9816546\n#     }\n# }\n```\n\n### Get courier contact\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#getcouriercontact)\n\n```python\norder_id = 32678866\nglover_info = client.order.glover_info(order_id)\nprint(glover_info)\n# Will show this\n# {\n#     "status": 200,\n#     "data": {\n#         "courierName": "Courier names",\n#         "phone": "+99999999999"\n#     }\n# }\n```\n\n### Get orders\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#getorders)\n\n```python\nstart=12344566\nend=12544566\norder_list = client.order.list(data={\'from\': start, \'to\': end})\nprint(order_list)\n# Will show this\n# {\n#     "status": 200,\n#     "data": [\n#         {\n#             "scheduleTime": null,\n#             "description": "Useful description!",\n#             "addresses": [\n#                 <PICKUP ADDRESS>,\n#                 <DELIVERY ADDRESS>,\n#             ],\n#             "id": "40304538",\n#             "state": "DELIVERED",\n#             "reference": null\n#         },\n#         <OTHER ORDERS>\n#     ]\n# }\n```\n\n### Cancel order\n\nFrom [glovo docs](https://api-docs.glovoapp.com/b2b/index.html#cancelorder)\n\n```python\norder_id = 32678866\ncanceled_order = client.order.cancel(order_id)\nprint(glover_info)\n# Will show this\n# {\n#     status: 200,\n#     data: {\n#         "id": 32678866,\n#         "state": "CANCELED",\n#         "scheduleTime": 12344566,\n#         "description": "A 30cm by 30cm box",\n#         "addresses": [\n#             <PICKUP ADDRESS>,\n#             <DELIVERY ADDRESS>,\n#         ]\n#     }\n# }\n```\n',
    'author': 'SoftButterfly Development Team',
    'author_email': 'dev@softbutterfly.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/softbutterfly/glovo-api-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
