"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResilienceLambdaChecker = void 0;
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const ResilientLambdaTask_1 = require("../construct/ResilientLambdaTask");
/**
 * Define an aspect that validates all Lambda Invoke tasks and warns if AWS Lambda transient errors are not handled properly.
 */
class ResilienceLambdaChecker {
    constructor(props) {
        this._fail = props === null || props === void 0 ? void 0 : props.fail;
    }
    visit(construct) {
        if (construct instanceof tasks.LambdaInvoke) {
            const reporter = this._fail ? construct.node.addError : construct.node.addWarning;
            const retries = this.getRetryConfiguration(construct);
            if (retries.length > 0) {
                const unhandledErrors = this.getUnhandledTransientErrors(retries);
                if (unhandledErrors.length > 0) {
                    reporter.apply(construct.node, [`Missing retry for transient errors: ${unhandledErrors}.`]);
                }
            }
            else {
                reporter.apply(construct.node, ['No retry for AWS Lambda transient errors defined - consider using ResilientLambdaTask construct.']);
                //ResilientLambdaTask.addDefaultRetry(construct);
            }
        }
    }
    getUnhandledTransientErrors(retries) {
        return ResilientLambdaTask_1.ResilientLambdaTask.TransientErrors.filter(transientError => retries.every(config => { var _a; return !((_a = config.errors) === null || _a === void 0 ? void 0 : _a.includes(transientError)); }));
    }
    getRetryConfiguration(construct) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        return construct.retries || [];
    }
}
exports.ResilienceLambdaChecker = ResilienceLambdaChecker;
//# sourceMappingURL=data:application/json;base64,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