"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RetryWithJitterTask = void 0;
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const ResilientLambdaTask_1 = require("./ResilientLambdaTask");
/**
 * Define a construct that implements retry with backoff and jitter.
 */
class RetryWithJitterTask extends sfn.Parallel {
    constructor(scope, id, props) {
        const parallelProps = {
            comment: props.comment,
            inputPath: props.inputPath,
            outputPath: "$[0]",
            resultPath: props.resultPath,
            parameters: {
                "RetryCount.$": "$$.State.RetryCount",
                "Input.$": "$"
            }
        };
        super(scope, id, parallelProps);
        const calculateJitterLambda = new lambda.Function(this, 'CalculateJitterLambda', {
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(__dirname + '/../../lambda/jitter'),
            handler: "main.lambda_handler",
        });
        const calculateJitterTask = new ResilientLambdaTask_1.ResilientLambdaTask(this, this.createStateName("CalculateJitter"), {
            lambdaFunction: calculateJitterLambda,
            payload: sfn.TaskInput.fromObject({
                "RetryCount.$": "$.RetryCount",
                "Backoff": 2
            }),
            resultPath: "$.WaitSeconds",
        });
        const waitTask = new sfn.Wait(this, this.createStateName("WaitBetweenRetries"), {
            time: sfn.WaitTime.secondsPath("$.WaitSeconds"),
        });
        // Need this state to "unwrap" original input to the state.
        // Also, CDK does not support outputPath for Wait state, which I would use for retry path
        const unwrapArguments = new sfn.Pass(this, this.createStateName('Unwrap Input'), {
            outputPath: "$.Input"
        });
        const retryPath = calculateJitterTask.next(waitTask).next(unwrapArguments);
        const choiceState = new sfn.Choice(this, this.createStateName("CheckRetryCount"))
            .when(sfn.Condition.numberGreaterThan("$.RetryCount", 0), retryPath)
            .otherwise(unwrapArguments)
            .afterwards()
            .next(props.tryProcess);
        this.branch(choiceState);
        this.addRetry({
            interval: cdk.Duration.seconds(0),
            maxAttempts: props.retryProps.maxAttempts,
            errors: props.retryProps.errors
        });
    }
    createStateName(name) {
        return `${name}_${this.node.uniqueId}`;
    }
}
exports.RetryWithJitterTask = RetryWithJitterTask;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiUmV0cnlXaXRoSml0dGVyVGFzay5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIlJldHJ5V2l0aEppdHRlclRhc2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQXFDO0FBQ3JDLDhDQUE4QztBQUM5QyxrREFBa0Q7QUFDbEQsK0RBQTJEO0FBMEMzRDs7R0FFRztBQUNILE1BQWEsbUJBQW9CLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFFbkQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUEyQjtRQUN2RSxNQUFNLGFBQWEsR0FBRztZQUNwQixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLFVBQVUsRUFBRSxNQUFNO1lBQ2xCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixVQUFVLEVBQUU7Z0JBQ1YsY0FBYyxFQUFFLHFCQUFxQjtnQkFDckMsU0FBUyxFQUFFLEdBQUc7YUFDZjtTQUNGLENBQUE7UUFFRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxhQUFhLENBQUMsQ0FBQTtRQUUvQixNQUFNLHFCQUFxQixHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7WUFDL0UsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxHQUFHLHNCQUFzQixDQUFDO1lBQy9ELE9BQU8sRUFBRSxxQkFBcUI7U0FDL0IsQ0FBQyxDQUFDO1FBRUgsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLHlDQUFtQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLGlCQUFpQixDQUFDLEVBQUU7WUFFakcsY0FBYyxFQUFFLHFCQUFxQjtZQUNyQyxPQUFPLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQ2hDLGNBQWMsRUFBRSxjQUFjO2dCQUM5QixTQUFTLEVBQUUsQ0FBQzthQUNiLENBQUM7WUFDRixVQUFVLEVBQUUsZUFBZTtTQUM1QixDQUFDLENBQUE7UUFFRixNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsb0JBQW9CLENBQUMsRUFBRTtZQUM5RSxJQUFJLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsZUFBZSxDQUFDO1NBQ2hELENBQUMsQ0FBQTtRQUVGLDJEQUEyRDtRQUMzRCx5RkFBeUY7UUFDekYsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLGNBQWMsQ0FBQyxFQUFFO1lBQy9FLFVBQVUsRUFBRSxTQUFTO1NBQ3RCLENBQUMsQ0FBQTtRQUVGLE1BQU0sU0FBUyxHQUFHLG1CQUFtQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUE7UUFFMUUsTUFBTSxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDOUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsaUJBQWlCLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxFQUFFLFNBQVMsQ0FBQzthQUNuRSxTQUFTLENBQUMsZUFBZSxDQUFDO2FBQzFCLFVBQVUsRUFBRTthQUNaLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUE7UUFFekIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQTtRQUV4QixJQUFJLENBQUMsUUFBUSxDQUFDO1lBQ1osUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNqQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxXQUFXO1lBQ3pDLE1BQU0sRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLE1BQU07U0FDaEMsQ0FBQyxDQUFBO0lBQ0osQ0FBQztJQUVPLGVBQWUsQ0FBQyxJQUFZO1FBQ2xDLE9BQU8sR0FBRyxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztJQUN6QyxDQUFDO0NBQ0Y7QUE5REQsa0RBOERDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IFJlc2lsaWVudExhbWJkYVRhc2sgfSBmcm9tICcuL1Jlc2lsaWVudExhbWJkYVRhc2snXG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgZGVmaW5pbmcgYSByZXRyeSB3aXRoIGJhY2tvZmYgYW5kIGppdHRlciBjb25zdHJ1Y3QuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmV0cnlXaXRoSml0dGVyUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEFuIG9wdGlvbmFsIGRlc2NyaXB0aW9uIGZvciB0aGlzIHN0YXRlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBjb21tZW50XG4gICAgICovXG4gICAgcmVhZG9ubHkgY29tbWVudD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBKU09OUGF0aCBleHByZXNzaW9uIHRvIHNlbGVjdCBwYXJ0IG9mIHRoZSBzdGF0ZSB0byBiZSB0aGUgaW5wdXQgdG8gdGhpcyBzdGF0ZS5cbiAgICAgKlxuICAgICAqIE1heSBhbHNvIGJlIHRoZSBzcGVjaWFsIHZhbHVlIERJU0NBUkQsIHdoaWNoIHdpbGwgY2F1c2UgdGhlIGVmZmVjdGl2ZVxuICAgICAqIGlucHV0IHRvIGJlIHRoZSBlbXB0eSBvYmplY3Qge30uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAkXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5wdXRQYXRoPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEpTT05QYXRoIGV4cHJlc3Npb24gdG8gaW5kaWNhdGUgd2hlcmUgdG8gaW5qZWN0IHRoZSBzdGF0ZSdzIG91dHB1dFxuICAgICAqXG4gICAgICogTWF5IGFsc28gYmUgdGhlIHNwZWNpYWwgdmFsdWUgRElTQ0FSRCwgd2hpY2ggd2lsbCBjYXVzZSB0aGUgc3RhdGUnc1xuICAgICAqIGlucHV0IHRvIGJlY29tZSBpdHMgb3V0cHV0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgJFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlc3VsdFBhdGg/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBUcnkgY2hhaW4gdG8gZXhlY3V0ZS5cbiAgICAgKi9cbiAgICByZWFkb25seSB0cnlQcm9jZXNzOiBzZm4uSUNoYWluYWJsZTtcblxuICAgIC8qKlxuICAgICAqIFJldHJ5IGNvbmZpZ3VyYXRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmV0cnlQcm9wczogc2ZuLlJldHJ5UHJvcHM7XG59IFxuXG4vKipcbiAqIERlZmluZSBhIGNvbnN0cnVjdCB0aGF0IGltcGxlbWVudHMgcmV0cnkgd2l0aCBiYWNrb2ZmIGFuZCBqaXR0ZXIuXG4gKi9cbmV4cG9ydCBjbGFzcyBSZXRyeVdpdGhKaXR0ZXJUYXNrIGV4dGVuZHMgc2ZuLlBhcmFsbGVsIHtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFJldHJ5V2l0aEppdHRlclByb3BzKSB7XG4gICAgY29uc3QgcGFyYWxsZWxQcm9wcyA9IHtcbiAgICAgIGNvbW1lbnQ6IHByb3BzLmNvbW1lbnQsXG4gICAgICBpbnB1dFBhdGg6IHByb3BzLmlucHV0UGF0aCxcbiAgICAgIG91dHB1dFBhdGg6IFwiJFswXVwiLFxuICAgICAgcmVzdWx0UGF0aDogcHJvcHMucmVzdWx0UGF0aCxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgXCJSZXRyeUNvdW50LiRcIjogXCIkJC5TdGF0ZS5SZXRyeUNvdW50XCIsXG4gICAgICAgIFwiSW5wdXQuJFwiOiBcIiRcIlxuICAgICAgfVxuICAgIH0gXG5cbiAgICBzdXBlcihzY29wZSwgaWQsIHBhcmFsbGVsUHJvcHMpXG5cbiAgICBjb25zdCBjYWxjdWxhdGVKaXR0ZXJMYW1iZGEgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdDYWxjdWxhdGVKaXR0ZXJMYW1iZGEnLCB7XG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KF9fZGlybmFtZSArICcvLi4vLi4vbGFtYmRhL2ppdHRlcicpLFxuICAgICAgaGFuZGxlcjogXCJtYWluLmxhbWJkYV9oYW5kbGVyXCIsXG4gICAgfSk7XG5cbiAgICBjb25zdCBjYWxjdWxhdGVKaXR0ZXJUYXNrID0gbmV3IFJlc2lsaWVudExhbWJkYVRhc2sodGhpcywgdGhpcy5jcmVhdGVTdGF0ZU5hbWUoXCJDYWxjdWxhdGVKaXR0ZXJcIiksIHtcblxuICAgICAgbGFtYmRhRnVuY3Rpb246IGNhbGN1bGF0ZUppdHRlckxhbWJkYSxcbiAgICAgIHBheWxvYWQ6IHNmbi5UYXNrSW5wdXQuZnJvbU9iamVjdCh7XG4gICAgICAgIFwiUmV0cnlDb3VudC4kXCI6IFwiJC5SZXRyeUNvdW50XCIsXG4gICAgICAgIFwiQmFja29mZlwiOiAyXG4gICAgICB9KSxcbiAgICAgIHJlc3VsdFBhdGg6IFwiJC5XYWl0U2Vjb25kc1wiLCAgICBcbiAgICB9KVxuXG4gICAgY29uc3Qgd2FpdFRhc2sgPSBuZXcgc2ZuLldhaXQodGhpcywgdGhpcy5jcmVhdGVTdGF0ZU5hbWUoXCJXYWl0QmV0d2VlblJldHJpZXNcIiksIHtcbiAgICAgIHRpbWU6IHNmbi5XYWl0VGltZS5zZWNvbmRzUGF0aChcIiQuV2FpdFNlY29uZHNcIiksXG4gICAgfSkgXG5cbiAgICAvLyBOZWVkIHRoaXMgc3RhdGUgdG8gXCJ1bndyYXBcIiBvcmlnaW5hbCBpbnB1dCB0byB0aGUgc3RhdGUuXG4gICAgLy8gQWxzbywgQ0RLIGRvZXMgbm90IHN1cHBvcnQgb3V0cHV0UGF0aCBmb3IgV2FpdCBzdGF0ZSwgd2hpY2ggSSB3b3VsZCB1c2UgZm9yIHJldHJ5IHBhdGhcbiAgICBjb25zdCB1bndyYXBBcmd1bWVudHMgPSBuZXcgc2ZuLlBhc3ModGhpcywgdGhpcy5jcmVhdGVTdGF0ZU5hbWUoJ1Vud3JhcCBJbnB1dCcpLCB7XG4gICAgICBvdXRwdXRQYXRoOiBcIiQuSW5wdXRcIlxuICAgIH0pXG5cbiAgICBjb25zdCByZXRyeVBhdGggPSBjYWxjdWxhdGVKaXR0ZXJUYXNrLm5leHQod2FpdFRhc2spLm5leHQodW53cmFwQXJndW1lbnRzKVxuICAgIFxuICAgIGNvbnN0IGNob2ljZVN0YXRlID0gbmV3IHNmbi5DaG9pY2UodGhpcywgdGhpcy5jcmVhdGVTdGF0ZU5hbWUoXCJDaGVja1JldHJ5Q291bnRcIikpXG4gICAgICAud2hlbihzZm4uQ29uZGl0aW9uLm51bWJlckdyZWF0ZXJUaGFuKFwiJC5SZXRyeUNvdW50XCIsIDApLCByZXRyeVBhdGgpXG4gICAgICAub3RoZXJ3aXNlKHVud3JhcEFyZ3VtZW50cylcbiAgICAgIC5hZnRlcndhcmRzKClcbiAgICAgIC5uZXh0KHByb3BzLnRyeVByb2Nlc3MpXG5cbiAgICB0aGlzLmJyYW5jaChjaG9pY2VTdGF0ZSlcblxuICAgIHRoaXMuYWRkUmV0cnkoe1xuICAgICAgaW50ZXJ2YWw6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDApLFxuICAgICAgbWF4QXR0ZW1wdHM6IHByb3BzLnJldHJ5UHJvcHMubWF4QXR0ZW1wdHMsXG4gICAgICBlcnJvcnM6IHByb3BzLnJldHJ5UHJvcHMuZXJyb3JzIFxuICAgIH0pXG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVN0YXRlTmFtZShuYW1lOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiBgJHtuYW1lfV8ke3RoaXMubm9kZS51bmlxdWVJZH1gO1xuICB9XG59Il19