"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    console.log('Loading existing catalog (if present)...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    // If defined, the function will invoke itself again to resume the work from that key (this
    // happens only in "from scratch" or "rebuild" cases).
    let nextStartAfter;
    // If we don't have a catalog, or the event does not have a package, we're in for a full refresh!
    if (!(data === null || data === void 0 ? void 0 : data.Body) || !(event === null || event === void 0 ? void 0 : event.package)) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME, event.startAfter)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
            // If we're getting short on time (1 minute out of 15 left), we'll be continuing in a new
            // invocation after writing what we've done so far to S3...
            if (context.getRemainingTimeInMillis() <= 60000) {
                nextStartAfter = pkgKey;
                break;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Clean up existing entries if necessary. In particular, remove the license texts as they make
    // the catalog unnecessarily large, and may hinder some search queries' result quality.
    for (const entry of catalog.packages) {
        if (entry.metadata) {
            delete entry.metadata.licenseText;
        }
    }
    // Upload the result to S3 and exit.
    const result = await aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
    if (nextStartAfter != null) {
        console.log(`Will continue from ${nextStartAfter} in new invocation...`);
        const nextEvent = { ...event, startAfter: nextStartAfter };
        // We start it asynchronously, as this function has a provisionned
        // concurrency of 1 (so a synchronous attempt would always be throttled).
        await aws.lambda().invokeAsync({
            FunctionName: context.functionName,
            InvokeArgs: JSON.stringify(nextEvent, null, 2),
        }).promise();
    }
    return result;
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 *
 * @param bucket the bucket in which to list objects
 * @param startAfter the key to start reading from, if provided.
 */
async function* relevantObjects(bucket, startAfter) {
    var _a, _b;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
        StartAfter: startAfter,
    };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST docs in one language.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            const javaDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_JAVA}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) &&
                !(await aws.s3ObjectExists(bucket, pyDocs)) &&
                !(await aws.s3ObjectExists(bucket, javaDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    // If the version is === to the current latest, we'll be replacing that (so re-generated metadata are taken into account)
    if (found != null && version.compare(found.version) < 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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