"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT, language_1.DocumentationLanguage.JAVA];
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" throttled', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['Lambda.TooManyRequestsException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            // We have to prepare the input to be a JSON string, within an array, because the ECS task integration expects
            // an array of strings (the model if that of a CLI invocation).
            // Unfortunately, we have to split this in two Pass states, because I don't know how to make it work otherwise.
            return new aws_stepfunctions_1.Pass(this, `Prepare ${language}`, {
                parameters: { command: { 'bucket.$': '$.bucket', 'assembly.$': '$.assembly', '$TaskExecution.$': '$.$TaskExecution' } },
                resultPath: '$',
            })
                .next(new aws_stepfunctions_1.Pass(this, `Stringify ${language} input`, {
                parameters: { 'commands.$': 'States.Array(States.JsonToString($.command))' },
                resultPath: '$',
            })
                .next(this.transliterator.createEcsRunTask(this, `Generate ${language} docs`, {
                cluster: this.ecsCluster,
                inputPath: '$.commands',
                language,
                resultSelector: { result: { 'language': language, 'success.$': '$' } },
                vpcSubnets: props.vpcSubnets,
            })
                // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
                .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
                .addRetry({
                errors: [
                    'ECS.AmazonECSException',
                    'jsii-docgen.NpmError.E429',
                    'jsii-codgen.NpmError.EPROTO',
                ],
                ...THROTTLE_RETRY_POLICY,
            })
                .addRetry({ maxAttempts: 3 })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" timed out`, { parameters: { error: 'Timed out!', language } }), { errors: ['States.Timeout'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" service error`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['ECS.AmazonECSException'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', language } }), { errors: ['States.TaskFailed'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" fault`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['States.ALL'] })));
        }))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        this.reprocessAllFunction = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.bucket.grantRead(this.reprocessAllFunction);
        this.stateMachine.grantStartExecution(this.reprocessAllFunction);
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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