"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'awscdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const db = Nano(NPM_REPLICA_REGISTRY_URL).db.use('registry');
    const initialMarker = await loadLastTransactionMarker(db);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
                metrics.putMetric("LastSeq" /* LAST_SEQ */, updatedMarker, aws_embedded_metrics_1.Unit.None);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param registry a Nano database corresponding to the Npmjs.com CouchDB instance.
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(registry) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: constants_lambda_shared_1.MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            const dbUpdateSeq = (await registry.info()).update_seq;
            if (dbUpdateSeq < marker) {
                console.warn(`Current DB update_seq (${dbUpdateSeq}) is lower than marker (CouchDB instance was likely replaced), resetting to 0!`);
                return 0;
            }
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.warn(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from scratch`);
            return 0;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(constants_lambda_shared_1.MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const message = integrity_lambda_shared_1.integrity({
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            }, tarball);
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    ko(new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`));
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a, _b, _c, _d;
        if (infos.jsii == null) {
            return false;
        }
        // The "constructs" package is a sign of a constructs library
        return isConstructFrameworkPackage(infos.name)
            // Recursively apply on dependencies
            || Object.keys((_a = infos.dependencies) !== null && _a !== void 0 ? _a : {}).some(isConstructFrameworkPackage)
            || Object.keys((_b = infos.devDependencies) !== null && _b !== void 0 ? _b : {}).some(isConstructFrameworkPackage)
            || Object.keys((_c = infos.peerDependencies) !== null && _c !== void 0 ? _c : {}).some(isConstructFrameworkPackage)
            // Keyword-based fallback
            || ((_d = infos.keywords) === null || _d === void 0 ? void 0 : _d.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
    /**
     * Package is one of the known construct framework's first party packages:
     * - @aws-cdk/*
     * - @cdktf/*
     * - cdk8s or cdk8s-plus
     */
    function isConstructFrameworkPackage(name) {
        // IMPORTANT NOTE: Prefix matching should only be used for @scope/ names.
        // The low-level constructs package
        return name === 'constructs'
            // AWS CDK Packages
            || name === 'aws-cdk-lib'
            || name === 'monocdk'
            || name.startsWith('@aws-cdk/')
            // CDK8s packages
            || name === 'cdk8s'
            || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(name)
            // CDKTf packages
            || name === 'cdktf'
            || name.startsWith('@cdktf/');
    }
}
//# sourceMappingURL=data:application/json;base64,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