"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const clients = new Map();
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return withFakeHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        const created = new Array();
        for (const snsRecord of event.Records) {
            const s3Event = JSON.parse(snsRecord.Sns.Message);
            for (const record of s3Event.Records) {
                // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
                // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
                // that would not replace `@` in the position where it is in the keys... So we have to work on
                // the URI components instead.
                const inputKey = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
                const [, packageName, packageVersion] = (_a = inputKey.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
                if (packageName == null) {
                    throw new Error(`Invalid object key: "${inputKey}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
                }
                const packageFqn = `${packageName}@${packageVersion}`;
                const client = (clients.has(record.awsRegion)
                    ? clients
                    : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
                console.log(`Source Bucket:  ${record.s3.bucket.name}`);
                console.log(`Source Key:     ${inputKey}`);
                console.log(`Source Version: ${record.s3.object.versionId}`);
                console.log(`Fetching assembly: ${inputKey}`);
                const assemblyResponse = await client.getObject({ Bucket: record.s3.bucket.name, Key: inputKey }).promise();
                if (!assemblyResponse.Body) {
                    throw new Error(`Response body for assembly at key ${inputKey} is empty`);
                }
                const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
                const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
                async function generateDocs(lang) {
                    const uploads = new Map();
                    const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
                    function renderAndDispatch(submodule) {
                        var _a;
                        console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                        const page = docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render();
                        const key = inputKey.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                        console.log(`Uploading ${key}`);
                        const upload = client.putObject({
                            Bucket: record.s3.bucket.name,
                            Key: key,
                            Body: page,
                            ContentType: 'text/html',
                            Metadata: {
                                'Origin-Version-Id': (_a = record.s3.object.versionId) !== null && _a !== void 0 ? _a : 'N/A',
                                'Lambda-Log-Group': context.logGroupName,
                                'Lambda-Log-Stream': context.logStreamName,
                                'Lambda-Run-Id': context.awsRequestId,
                            },
                        }).promise();
                        uploads.set(key, upload);
                    }
                    renderAndDispatch();
                    for (const submodule of submodules) {
                        renderAndDispatch(submodule);
                    }
                    for (const [key, upload] of uploads.entries()) {
                        const response = await upload;
                        created.push({ bucket: record.s3.bucket.name, key: key, versionId: response.VersionId });
                        console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
                    }
                }
                await generateDocs(language);
            }
        }
        return created;
    });
}
exports.handler = handler;
async function withFakeHome(cb) {
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
    }
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
/**
 * Visible for testing. Clears the caches so that the next execution runs clean.
 */
function reset() {
    clients.clear();
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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