"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageDeployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const iam = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const lambda = require("aws-cdk-lib/aws-lambda-nodejs");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * `DockerImageDeployment` pushes an image from a local or external source to a specified external destination
 */
class DockerImageDeployment extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const handlerRole = new iam.Role(this, 'DockerImageDeployRole', {
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        const sourceConfig = props.source.bind(this, { handlerRole });
        const destinationConfig = props.destination.bind(handlerRole);
        const sourceUri = sourceConfig.imageUri;
        const destTag = destinationConfig.destinationTag ?? sourceConfig.imageTag;
        this.validateTag(destTag);
        const destUri = `${destinationConfig.destinationUri}:${destTag}`;
        const commands = [
            sourceConfig.loginConfig.loginCommand,
            `docker pull ${sourceUri}`,
            `docker tag ${sourceUri} ${destUri}`,
        ];
        if (sourceConfig.loginConfig.region !== destinationConfig.loginConfig.region || !sourceConfig.loginConfig.region) { // different regions or either undefined should logout and login
            commands.push('docker logout');
            commands.push(destinationConfig.loginConfig.loginCommand);
        }
        commands.push(`docker push ${destUri}`);
        commands.push('docker logout');
        this.cb = new codebuild.Project(this, 'DockerImageDeployProject', {
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: commands,
                    },
                },
            }),
            environment: {
                privileged: true,
                buildImage: codebuild.LinuxBuildImage.STANDARD_5_0,
            },
            role: handlerRole,
        });
        const onEventHandler = new lambda.NodejsFunction(this, 'onEventHandler', {
            entry: path.join(__dirname, 'codebuild-handler/index.js'),
            handler: 'onEventhandler',
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
        });
        const isCompleteHandler = new lambda.NodejsFunction(this, 'isCompleteHandler', {
            entry: path.join(__dirname, 'codebuild-handler/index.js'),
            handler: 'isCompleteHandler',
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
        });
        // https://github.com/aws/aws-cdk/issues/21721 issue to add grant methods to codebuild
        const grantOnEvent = iam.Grant.addToPrincipal({
            grantee: onEventHandler,
            actions: ['codebuild:StartBuild'],
            resourceArns: [this.cb.projectArn],
            scope: this,
        });
        const grantIsComplete = iam.Grant.addToPrincipal({
            grantee: isCompleteHandler,
            actions: [
                'codebuild:ListBuildsForProject',
                'codebuild:BatchGetBuilds',
            ],
            resourceArns: [this.cb.projectArn],
            scope: this,
        });
        const crProvider = new cr.Provider(this, 'CRProvider', {
            onEventHandler: onEventHandler,
            isCompleteHandler: isCompleteHandler,
            queryInterval: aws_cdk_lib_1.Duration.seconds(30),
            totalTimeout: aws_cdk_lib_1.Duration.minutes(30),
        });
        const customResource = new aws_cdk_lib_1.CustomResource(this, `CustomResource${Date.now().toString()}`, {
            serviceToken: crProvider.serviceToken,
            properties: {
                projectName: this.cb.projectName,
            },
        });
        customResource.node.addDependency(grantOnEvent, grantIsComplete);
        try {
            new aws_cdk_lib_1.CfnOutput(this, 'CustomResourceReport', {
                value: `${customResource.getAttString('Status')}, see the logs here: ${customResource.getAtt('LogsUrl')}`,
            });
        }
        catch (error) {
            throw new Error('Error getting the report from the custom resource');
        }
    }
    validateTag(tag) {
        if (aws_cdk_lib_1.Token.isUnresolved(tag)) {
            return; // if token tag is likely from source, so assume it is valid
        }
        if (tag.length > 128) {
            throw new Error(`Invalid tag: tags may contain a maximum of 128 characters; your tag ${tag} has ${tag.length} characters`);
        }
        if (!/^[^-.][a-zA-Z0-9-_.]+$/.test(tag)) {
            throw new Error(`Invalid tag: tags must contain alphanumeric characters and \'-\' \'_\' \'.\' only and must not begin with \'.\' or \'-\'; your tag was ${tag}`);
        }
    }
}
exports.DockerImageDeployment = DockerImageDeployment;
_a = JSII_RTTI_SYMBOL_1;
DockerImageDeployment[_a] = { fqn: "cdk-docker-image-deployment.DockerImageDeployment", version: "0.0.202" };
//# sourceMappingURL=data:application/json;base64,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