"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = () => [
    'export SYNTH_ARTIFACT_URI=`echo $SYNTH_ARTIFACT_LOCATION | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    'export SYNTH_SOURCE_URI=`echo $SYNTH_SOURCE_VERSION | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $SYNTH_ARTIFACT_URI dist.zip",
    "unzip source.zip",
    "unzip dist.zip -d cdk.out",
    "rm source.zip dist.zip",
];
const owaspScan = () => "npx owasp-dependency-check --format HTML --out reports --exclude '**/node_modules/**/*' --exclude '**/reports/**/*' --exclude '**/cdk.out/**/*' --exclude '**/.env/**/*' --exclude '**/dist/**/*' --exclude '**/.git/**/*' --scan . --enableExperimental --bin /tmp/dep-check --disableRetireJS";
const cfnNagScan = (cfnNagIgnorePath) => `cfn_nag ${cfnNagIgnorePath !== null && cfnNagIgnorePath !== void 0 ? cfnNagIgnorePath : ""} cdk.out/**/*.template.json --output-format=json > reports/cfn-nag-report.json`;
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            ...unpackSourceAndArtifacts(),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            owaspScan(),
                            cfnNagScan(props.cfnNagIgnorePath),
                            sonarqube_commands_1.sonarqubeScanner(),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                        {
                            name: "SYNTH_ARTIFACT_LOCATION",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.additional-information.artifact.location"),
                        },
                        {
                            name: "SYNTH_SOURCE_VERSION",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.additional-information.source-version"),
                        },
                    ],
                }),
            }),
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pdk_pipeline.SonarCodeScanner", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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