"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = exports.SampleLanguage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const console = require("console");
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported languages to generate sample code.
 */
var SampleLanguage;
(function (SampleLanguage) {
    SampleLanguage["TYPESCRIPT"] = "ts";
    SampleLanguage["PYTHON"] = "py";
    SampleLanguage["JAVA"] = "java";
})(SampleLanguage = exports.SampleLanguage || (exports.SampleLanguage = {}));
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            release: false,
            sampleCode: false,
            defaultReleaseBranch: "mainline",
        });
        this.implicitDependencies = {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(options.nxIgnorePatterns || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs
                .readFileSync("./node_modules/aws-prototyping-sdk/lib/pdk_projen/nx_monorepo/plugin/nx-monorepo-plugin.js")
                .toString("utf-8")
                .split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(options.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
        if (options.sampleLanguage) {
            if (fs.existsSync(path.join(this.outdir, ".projenrc.ts.bk"))) {
                console.log("Ignoring generation of sample code as this is a destructive action and should only be set on initial synthesis.");
            }
            else {
                fs.copyFileSync(path.join(this.outdir, ".projenrc.ts"), ".projenrc.ts.bk"); // Make a backup of the existing .projenrc.ts just in case
                let sampleProjenTs = fs
                    .readFileSync(path.join(this.outdir, `node_modules/aws-prototyping-sdk/samples/sample-nx-monorepo/src/nx-monorepo-sample-${options.sampleLanguage}.ts`))
                    .toString("utf-8");
                options.sampleContextJson &&
                    (sampleProjenTs = sampleProjenTs.replace("context: {}", `context: ${options.sampleContextJson}`));
                fs.writeFileSync(path.join(this.outdir, ".projenrc.ts"), sampleProjenTs);
            }
        }
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
        // Add workspaces for each subproject
        this.package.addField("workspaces", this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)));
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Check to see if a new subProject was added
        const newSubProject = this.subProjects.find((subProject) => !fs.existsSync(subProject.outdir));
        // Need to synth before generating the package.json otherwise the subdirectory won't exist
        newSubProject && super.synth();
        this.subProjects
            .filter((subProject) => !fs.existsSync(`${subProject.outdir}/package.json`) ||
            JSON.parse(fs.readFileSync(`${subProject.outdir}/package.json`).toString()).__pdk__)
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            manifest.name = subProject.name;
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
        super.synth();
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.pdk_projen.NxMonorepoProject", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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