"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonFileSeeds = exports.InlineSeeds = exports.S3Seeds = exports.Seeds = void 0;
const fs = require("fs");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
class Seeds {
    /**
     * @returns `S3Seeds` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Seeds(bucket, key, objectVersion);
    }
    /**
     * @returns `InlineSeeds` with inline seeds.
     * @param seeds The actual json code (limited to 4KiB)
     */
    static fromInline(seeds) {
        return new InlineSeeds(JSON.stringify(seeds));
    }
    /**
     * Loads the seeds from a local disk path and uploads it to s3.
     *
     * @returns `JsonFileSeeds` associated with the specified S3 object.
     * @param path Path to json seeds file.
     */
    static fromJsonFile(path, options) {
        return new JsonFileSeeds(path, options);
    }
}
exports.Seeds = Seeds;
/**
 * Seeds from an S3 archive.
 */
class S3Seeds extends Seeds {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('BucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Seeds = S3Seeds;
/**
 * Seeds from an inline json object (limited to 4KiB).
 */
class InlineSeeds extends Seeds {
    constructor(seeds) {
        super();
        this.seeds = seeds;
        if (seeds.length === 0) {
            throw new Error('Inline seeds cannot be empty');
        }
        if (seeds.length > 4096) {
            throw new Error('Inline seeds are too large, must be <= 4096 but is ' + seeds.length);
        }
    }
    bind(_scope) {
        return {
            inlineSeeds: this.seeds,
        };
    }
}
exports.InlineSeeds = InlineSeeds;
/**
 * Seeds from a local json file.
 */
class JsonFileSeeds extends Seeds {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same JsonFileSeeds is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new aws_s3_assets_1.Asset(scope, 'JsonFileSeeds', {
                path: this.path,
                ...this.options,
            });
        }
        try {
            JSON.parse(fs.readFileSync(this.path, 'utf-8'));
        }
        catch (e) {
            throw new Error('Could not convert file to JSON');
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.JsonFileSeeds = JsonFileSeeds;
//# sourceMappingURL=data:application/json;base64,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