"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = void 0;
const child_process_1 = require("child_process");
const console = require("console");
const os = require("os");
const path = require("path");
const process = require("process");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const jsii_rosetta_1 = require("jsii-rosetta");
const transliterate_1 = require("jsii-rosetta/lib/commands/transliterate");
const constants = require("../shared/constants.lambda-shared");
const clients = new Map();
const PACKAGE_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.PACKAGE_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
async function handler(event, context) {
    var _a, _b;
    console.log(JSON.stringify(event, null, 2));
    const created = new Array();
    for (const record of event.Records) {
        // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
        // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
        // that would not replace `@` in the position where it is in the keys... So we have to work on
        // the URI components instead.
        const inputKey = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
        const [, packageName, packageVersion] = (_a = inputKey.match(PACKAGE_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${inputKey}". It was expected to match ${PACKAGE_KEY_REGEX}!`);
        }
        const client = (clients.has(record.awsRegion)
            ? clients
            : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
        console.log(`Source Bucket:  ${record.s3.bucket.name}`);
        console.log(`Source Key:     ${inputKey}`);
        console.log(`Source Version: ${record.s3.object.versionId}`);
        const object = await client.getObject({
            Bucket: record.s3.bucket.name,
            Key: inputKey,
            VersionId: record.s3.object.versionId,
        }).promise();
        const workdir = await fs.mkdtemp(path.join(os.tmpdir(), 'workdir'));
        try {
            const tarball = path.join(workdir, `${packageName.replace('@', '').replace('/', '-')}-${packageVersion}.tgz`);
            await fs.writeFile(tarball, object.Body);
            await new Promise((ok, ko) => {
                // --ignore-scripts disables lifecycle hooks, in order to prevent execution of arbitrary code
                // --no-bin-links ensures npm does not insert anything in $PATH
                const npmInstall = child_process_1.spawn('npm', ['install', '--ignore-scripts', '--no-bin-links', '--no-save', tarball], {
                    cwd: workdir,
                    env: {
                        ...process.env,
                        HOME: os.tmpdir(),
                    },
                    stdio: ['ignore', 'inherit', 'inherit'],
                });
                npmInstall.once('error', ko);
                npmInstall.once('close', (code, signal) => {
                    if (code === 0) {
                        ok();
                    }
                    else {
                        ko(`"npm install" command ${code != null ? `exited with code ${code}` : `was terminated by signal ${signal}`}`);
                    }
                });
            });
            const packageDir = path.join(workdir, 'node_modules', ...packageName.split('/'));
            await transliterate_1.transliterateAssembly([packageDir], [jsii_rosetta_1.TargetLanguage.PYTHON]);
            // Payload object key => packages/[<@scope>/]<name>/v<version>/package.tgz
            // Output object key  => packages/[<@scope>/]<name>/v<version>/assembly-python.json
            const key = inputKey.replace(/\/[^/]+$/, constants.assemblyKeySuffix(jsii_rosetta_1.TargetLanguage.PYTHON));
            const response = await client.putObject({
                Bucket: record.s3.bucket.name,
                Key: key,
                Body: await fs.readFile(path.join(packageDir, '.jsii.python')),
                ContentType: 'text/json',
                Metadata: {
                    'Origin-Version-Id': (_b = record.s3.object.versionId) !== null && _b !== void 0 ? _b : 'N/A',
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise();
            created.push({
                bucket: record.s3.bucket.name,
                key,
                versionId: response.VersionId,
            });
        }
        finally {
            await fs.remove(workdir);
        }
    }
    return created;
}
exports.handler = handler;
/**
 * Visible for testing. Clears the caches so that the next execution runs clean.
 */
function reset() {
    clients.clear();
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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