"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const core_1 = require("@aws-cdk/core");
const validation = require("./private/validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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