"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhZ2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGFnZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw4Q0FBOEM7QUFDOUMscUNBQXFDO0FBTXJDLDJDQUEyQztBQUUzQzs7Ozs7OztHQU9HO0FBQ0gsTUFBYSxLQUFLO0lBU2hCOztPQUVHO0lBQ0gsWUFBWSxLQUFpQixFQUFFLFFBQWtCO1FBTGhDLGFBQVEsR0FBRyxJQUFJLEtBQUssRUFBd0IsQ0FBQztRQU01RCxVQUFVLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFbEQsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxTQUFTLEdBQUcsUUFBUSxDQUFDO1FBQzFCLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFekQsS0FBSyxNQUFNLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxJQUFJLEVBQUUsRUFBRTtZQUN4QyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3hCO0lBQ0gsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxpQkFBaUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQy9CLENBQUM7SUFFRCxJQUFXLE9BQU87UUFDaEIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVELElBQVcsUUFBUTtRQUNqQixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDeEIsQ0FBQztJQUVNLE1BQU07UUFDWCw2REFBNkQ7UUFDN0QsS0FBSyxNQUFNLE1BQU0sSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2xDLE1BQU0sZUFBZSxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUM7WUFFdkMsTUFBTSxjQUFjLEdBQUcsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDO1lBRXBFLEtBQUssTUFBTSxjQUFjLElBQUksZUFBZSxFQUFFO2dCQUM1QyxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksRUFBRTtvQkFDaEMsTUFBTSx1QkFBdUIsR0FBRyxZQUFZLElBQUksQ0FBQyxTQUFTLElBQUksTUFBTSxDQUFDLFVBQVUsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDO3dCQUM5RyxDQUFDLENBQUMsRUFBRTt3QkFDSixDQUFDLENBQUMsR0FBRyxHQUFHLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO29CQUN4RCxNQUFNLFlBQVksR0FBRyxvQkFBb0IsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO29CQUNsRSxjQUFzQixDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztpQkFDaEQ7YUFDRjtTQUNGO1FBRUQsT0FBTztZQUNMLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUztZQUNwQixPQUFPLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ2hFLENBQUM7SUFDSixDQUFDO0lBRU0sU0FBUyxDQUFDLE1BQWU7UUFDOUIsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQztRQUN0RCxvQkFBb0I7UUFDcEIsVUFBVSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFOUMsd0NBQXdDO1FBQ3hDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsVUFBVSxLQUFLLFVBQVUsQ0FBQyxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsU0FBUyxJQUFJLENBQUMsU0FBUywwQ0FBMEMsVUFBVSxHQUFHLENBQUMsQ0FBQztTQUNqRztRQUVELElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFTSxhQUFhLENBQUMsSUFBWSxFQUFFLE1BQTJCLEVBQUUsT0FBMEI7UUFDeEYsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDdkIsSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUNuQixVQUFVLEVBQUUsQ0FBQywyQ0FBMkMsQ0FBQztZQUN6RCxNQUFNLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQztZQUM1QixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztZQUN0QyxNQUFNLEVBQUU7Z0JBQ04sS0FBSyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQzthQUN4QjtTQUNGLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPO1lBQ0wsR0FBRyxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDNUIsR0FBRyxJQUFJLENBQUMsZUFBZSxFQUFFO1NBQzFCLENBQUM7SUFDSixDQUFDO0lBRU8sa0JBQWtCO1FBQ3hCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzlCLE9BQU8sQ0FBQyxVQUFVLElBQUksQ0FBQyxTQUFTLGlDQUFpQyxDQUFDLENBQUM7U0FDcEU7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFTyxlQUFlO1FBQ3JCLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFDaEMsS0FBSyxNQUFNLE1BQU0sSUFBSSxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDM0MsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztTQUMxQztRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztJQUVPLGNBQWMsQ0FBQyxNQUE0QjtRQUNqRCxPQUFPLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsY0FBYyxDQUFDLFNBQVMsRUFDOUYsTUFBTSxDQUFDLGNBQWMsQ0FBQyxTQUFTLEVBQUUsTUFBTSxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDO2FBQ2pFLE1BQU0sQ0FBQyxVQUFVLENBQUMsc0JBQXNCLENBQUMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLGNBQWMsQ0FBQyxVQUFVLEVBQ2xHLE1BQU0sQ0FBQyxjQUFjLENBQUMsVUFBVSxFQUFFLE1BQU0sQ0FBQyxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUNwRSxDQUFDO0lBQ04sQ0FBQztJQUVPLHNCQUFzQixDQUFDLE1BQWU7UUFDNUMsd0NBQXdDO1FBQ3hDLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN0RixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsdUJBQXVCLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRSxXQUFXLENBQUMsQ0FBQztJQUMzRSxDQUFDO0lBRU8sWUFBWSxDQUFDLE1BQTRCO1FBQy9DLE1BQU0sZUFBZSxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUM3SCxNQUFNLGNBQWMsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDM0gsT0FBTztZQUNMLElBQUksRUFBRSxNQUFNLENBQUMsVUFBVTtZQUN2QixjQUFjO1lBQ2QsZUFBZTtZQUNmLFlBQVksRUFBRTtnQkFDWixRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUU7Z0JBQ3BDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTztnQkFDdkIsS0FBSyxFQUFFLE1BQU0sQ0FBQyxLQUFLO2dCQUNuQixRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVE7YUFDMUI7WUFDRCxhQUFhLEVBQUUsTUFBTSxDQUFDLGFBQWE7WUFDbkMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxRQUFRO1lBQ3pCLE9BQU8sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN0RCxNQUFNLEVBQUUsTUFBTSxDQUFDLE1BQU07WUFDckIsU0FBUyxFQUFFLE1BQU0sQ0FBQyxTQUFTO1NBQzVCLENBQUM7SUFDSixDQUFDO0lBRU8sZUFBZSxDQUFDLFNBQXFCO1FBQzNDLE9BQU8sU0FBUzthQUNiLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUM7YUFDM0IsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsWUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzNDLENBQUM7Q0FDRjtBQXhKRCxzQkF3SkM7QUFFRCxTQUFTLG9CQUFvQixDQUFDLFlBQW9CO0lBQ2hELHNFQUFzRTtJQUN0RSw0QkFBNEI7SUFDNUIsT0FBTyxZQUFZLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztBQUMzQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgSUFjdGlvbiwgSVBpcGVsaW5lLCBJU3RhZ2UgfSBmcm9tICcuLi9hY3Rpb24nO1xuaW1wb3J0IHsgQXJ0aWZhY3QgfSBmcm9tICcuLi9hcnRpZmFjdCc7XG5pbXBvcnQgeyBDZm5QaXBlbGluZSB9IGZyb20gJy4uL2NvZGVwaXBlbGluZS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgUGlwZWxpbmUsIFN0YWdlUHJvcHMgfSBmcm9tICcuLi9waXBlbGluZSc7XG5pbXBvcnQgeyBGdWxsQWN0aW9uRGVzY3JpcHRvciB9IGZyb20gJy4vZnVsbC1hY3Rpb24tZGVzY3JpcHRvcic7XG5pbXBvcnQgKiBhcyB2YWxpZGF0aW9uIGZyb20gJy4vdmFsaWRhdGlvbic7XG5cbi8qKlxuICogQSBTdGFnZSBpbiBhIFBpcGVsaW5lLlxuICpcbiAqIFN0YWdlcyBhcmUgYWRkZWQgdG8gYSBQaXBlbGluZSBieSBjYWxsaW5nIHtAbGluayBQaXBlbGluZSNhZGRTdGFnZX0sXG4gKiB3aGljaCByZXR1cm5zIGFuIGluc3RhbmNlIG9mIHtAbGluayBjb2RlcGlwZWxpbmUuSVN0YWdlfS5cbiAqXG4gKiBUaGlzIGNsYXNzIGlzIHByaXZhdGUgdG8gdGhlIENvZGVQaXBlbGluZSBtb2R1bGUuXG4gKi9cbmV4cG9ydCBjbGFzcyBTdGFnZSBpbXBsZW1lbnRzIElTdGFnZSB7XG4gIC8qKlxuICAgKiBUaGUgUGlwZWxpbmUgdGhpcyBTdGFnZSBpcyBhIHBhcnQgb2YuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3RhZ2VOYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2NvcGU6IGNkay5Db25zdHJ1Y3Q7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3BpcGVsaW5lOiBQaXBlbGluZTtcbiAgcHJpdmF0ZSByZWFkb25seSBfYWN0aW9ucyA9IG5ldyBBcnJheTxGdWxsQWN0aW9uRGVzY3JpcHRvcj4oKTtcblxuICAvKipcbiAgICogQ3JlYXRlIGEgbmV3IFN0YWdlLlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM6IFN0YWdlUHJvcHMsIHBpcGVsaW5lOiBQaXBlbGluZSkge1xuICAgIHZhbGlkYXRpb24udmFsaWRhdGVOYW1lKCdTdGFnZScsIHByb3BzLnN0YWdlTmFtZSk7XG5cbiAgICB0aGlzLnN0YWdlTmFtZSA9IHByb3BzLnN0YWdlTmFtZTtcbiAgICB0aGlzLl9waXBlbGluZSA9IHBpcGVsaW5lO1xuICAgIHRoaXMuc2NvcGUgPSBuZXcgY2RrLkNvbnN0cnVjdChwaXBlbGluZSwgdGhpcy5zdGFnZU5hbWUpO1xuXG4gICAgZm9yIChjb25zdCBhY3Rpb24gb2YgcHJvcHMuYWN0aW9ucyB8fCBbXSkge1xuICAgICAgdGhpcy5hZGRBY3Rpb24oYWN0aW9uKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogR2V0IGEgZHVwbGljYXRlIG9mIHRoaXMgc3RhZ2UncyBsaXN0IG9mIGFjdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGFjdGlvbkRlc2NyaXB0b3JzKCk6IEZ1bGxBY3Rpb25EZXNjcmlwdG9yW10ge1xuICAgIHJldHVybiB0aGlzLl9hY3Rpb25zLnNsaWNlKCk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGFjdGlvbnMoKTogSUFjdGlvbltdIHtcbiAgICByZXR1cm4gdGhpcy5fYWN0aW9ucy5tYXAoYWN0aW9uRGVzY3JpcHRvciA9PiBhY3Rpb25EZXNjcmlwdG9yLmFjdGlvbik7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHBpcGVsaW5lKCk6IElQaXBlbGluZSB7XG4gICAgcmV0dXJuIHRoaXMuX3BpcGVsaW5lO1xuICB9XG5cbiAgcHVibGljIHJlbmRlcigpOiBDZm5QaXBlbGluZS5TdGFnZURlY2xhcmF0aW9uUHJvcGVydHkge1xuICAgIC8vIGZpcnN0LCBhc3NpZ24gbmFtZXMgdG8gb3V0cHV0IEFydGlmYWN0cyB3aG8gZG9uJ3QgaGF2ZSBvbmVcbiAgICBmb3IgKGNvbnN0IGFjdGlvbiBvZiB0aGlzLl9hY3Rpb25zKSB7XG4gICAgICBjb25zdCBvdXRwdXRBcnRpZmFjdHMgPSBhY3Rpb24ub3V0cHV0cztcblxuICAgICAgY29uc3QgdW5uYW1lZE91dHB1dHMgPSBvdXRwdXRBcnRpZmFjdHMuZmlsdGVyKG8gPT4gIW8uYXJ0aWZhY3ROYW1lKTtcblxuICAgICAgZm9yIChjb25zdCBvdXRwdXRBcnRpZmFjdCBvZiBvdXRwdXRBcnRpZmFjdHMpIHtcbiAgICAgICAgaWYgKCFvdXRwdXRBcnRpZmFjdC5hcnRpZmFjdE5hbWUpIHtcbiAgICAgICAgICBjb25zdCB1bnNhbml0aXplZEFydGlmYWN0TmFtZSA9IGBBcnRpZmFjdF8ke3RoaXMuc3RhZ2VOYW1lfV8ke2FjdGlvbi5hY3Rpb25OYW1lfWAgKyAodW5uYW1lZE91dHB1dHMubGVuZ3RoID09PSAxXG4gICAgICAgICAgICA/ICcnXG4gICAgICAgICAgICA6ICdfJyArICh1bm5hbWVkT3V0cHV0cy5pbmRleE9mKG91dHB1dEFydGlmYWN0KSArIDEpKTtcbiAgICAgICAgICBjb25zdCBhcnRpZmFjdE5hbWUgPSBzYW5pdGl6ZUFydGlmYWN0TmFtZSh1bnNhbml0aXplZEFydGlmYWN0TmFtZSk7XG4gICAgICAgICAgKG91dHB1dEFydGlmYWN0IGFzIGFueSkuX3NldE5hbWUoYXJ0aWZhY3ROYW1lKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBuYW1lOiB0aGlzLnN0YWdlTmFtZSxcbiAgICAgIGFjdGlvbnM6IHRoaXMuX2FjdGlvbnMubWFwKGFjdGlvbiA9PiB0aGlzLnJlbmRlckFjdGlvbihhY3Rpb24pKSxcbiAgICB9O1xuICB9XG5cbiAgcHVibGljIGFkZEFjdGlvbihhY3Rpb246IElBY3Rpb24pOiB2b2lkIHtcbiAgICBjb25zdCBhY3Rpb25OYW1lID0gYWN0aW9uLmFjdGlvblByb3BlcnRpZXMuYWN0aW9uTmFtZTtcbiAgICAvLyB2YWxpZGF0ZSB0aGUgbmFtZVxuICAgIHZhbGlkYXRpb24udmFsaWRhdGVOYW1lKCdBY3Rpb24nLCBhY3Rpb25OYW1lKTtcblxuICAgIC8vIGNoZWNrIGZvciBkdXBsaWNhdGUgQWN0aW9ucyBhbmQgbmFtZXNcbiAgICBpZiAodGhpcy5fYWN0aW9ucy5maW5kKGEgPT4gYS5hY3Rpb25OYW1lID09PSBhY3Rpb25OYW1lKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBTdGFnZSAke3RoaXMuc3RhZ2VOYW1lfSBhbHJlYWR5IGNvbnRhaW5zIGFuIGFjdGlvbiB3aXRoIG5hbWUgJyR7YWN0aW9uTmFtZX0nYCk7XG4gICAgfVxuXG4gICAgdGhpcy5fYWN0aW9ucy5wdXNoKHRoaXMuYXR0YWNoQWN0aW9uVG9QaXBlbGluZShhY3Rpb24pKTtcbiAgfVxuXG4gIHB1YmxpYyBvblN0YXRlQ2hhbmdlKG5hbWU6IHN0cmluZywgdGFyZ2V0PzogZXZlbnRzLklSdWxlVGFyZ2V0LCBvcHRpb25zPzogZXZlbnRzLlJ1bGVQcm9wcyk6IGV2ZW50cy5SdWxlIHtcbiAgICBjb25zdCBydWxlID0gbmV3IGV2ZW50cy5SdWxlKHRoaXMuc2NvcGUsIG5hbWUsIG9wdGlvbnMpO1xuICAgIHJ1bGUuYWRkVGFyZ2V0KHRhcmdldCk7XG4gICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgZGV0YWlsVHlwZTogWydDb2RlUGlwZWxpbmUgU3RhZ2UgRXhlY3V0aW9uIFN0YXRlIENoYW5nZSddLFxuICAgICAgc291cmNlOiBbJ2F3cy5jb2RlcGlwZWxpbmUnXSxcbiAgICAgIHJlc291cmNlczogW3RoaXMucGlwZWxpbmUucGlwZWxpbmVBcm5dLFxuICAgICAgZGV0YWlsOiB7XG4gICAgICAgIHN0YWdlOiBbdGhpcy5zdGFnZU5hbWVdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICByZXR1cm4gcnVsZTtcbiAgfVxuXG4gIHB1YmxpYyB2YWxpZGF0ZSgpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIFtcbiAgICAgIC4uLnRoaXMudmFsaWRhdGVIYXNBY3Rpb25zKCksXG4gICAgICAuLi50aGlzLnZhbGlkYXRlQWN0aW9ucygpLFxuICAgIF07XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlSGFzQWN0aW9ucygpOiBzdHJpbmdbXSB7XG4gICAgaWYgKHRoaXMuX2FjdGlvbnMubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gW2BTdGFnZSAnJHt0aGlzLnN0YWdlTmFtZX0nIG11c3QgaGF2ZSBhdCBsZWFzdCBvbmUgYWN0aW9uYF07XG4gICAgfVxuICAgIHJldHVybiBbXTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVBY3Rpb25zKCk6IHN0cmluZ1tdIHtcbiAgICBjb25zdCByZXQgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgIGZvciAoY29uc3QgYWN0aW9uIG9mIHRoaXMuYWN0aW9uRGVzY3JpcHRvcnMpIHtcbiAgICAgIHJldC5wdXNoKC4uLnRoaXMudmFsaWRhdGVBY3Rpb24oYWN0aW9uKSk7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlQWN0aW9uKGFjdGlvbjogRnVsbEFjdGlvbkRlc2NyaXB0b3IpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIHZhbGlkYXRpb24udmFsaWRhdGVBcnRpZmFjdEJvdW5kcygnaW5wdXQnLCBhY3Rpb24uaW5wdXRzLCBhY3Rpb24uYXJ0aWZhY3RCb3VuZHMubWluSW5wdXRzLFxuICAgICAgYWN0aW9uLmFydGlmYWN0Qm91bmRzLm1heElucHV0cywgYWN0aW9uLmNhdGVnb3J5LCBhY3Rpb24ucHJvdmlkZXIpXG4gICAgICAuY29uY2F0KHZhbGlkYXRpb24udmFsaWRhdGVBcnRpZmFjdEJvdW5kcygnb3V0cHV0JywgYWN0aW9uLm91dHB1dHMsIGFjdGlvbi5hcnRpZmFjdEJvdW5kcy5taW5PdXRwdXRzLFxuICAgICAgICBhY3Rpb24uYXJ0aWZhY3RCb3VuZHMubWF4T3V0cHV0cywgYWN0aW9uLmNhdGVnb3J5LCBhY3Rpb24ucHJvdmlkZXIpLFxuICAgICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgYXR0YWNoQWN0aW9uVG9QaXBlbGluZShhY3Rpb246IElBY3Rpb24pOiBGdWxsQWN0aW9uRGVzY3JpcHRvciB7XG4gICAgLy8gbm90aWZ5IHRoZSBQaXBlbGluZSBvZiB0aGUgbmV3IEFjdGlvblxuICAgIGNvbnN0IGFjdGlvblNjb3BlID0gbmV3IGNkay5Db25zdHJ1Y3QodGhpcy5zY29wZSwgYWN0aW9uLmFjdGlvblByb3BlcnRpZXMuYWN0aW9uTmFtZSk7XG4gICAgcmV0dXJuIHRoaXMuX3BpcGVsaW5lLl9hdHRhY2hBY3Rpb25Ub1BpcGVsaW5lKHRoaXMsIGFjdGlvbiwgYWN0aW9uU2NvcGUpO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJBY3Rpb24oYWN0aW9uOiBGdWxsQWN0aW9uRGVzY3JpcHRvcik6IENmblBpcGVsaW5lLkFjdGlvbkRlY2xhcmF0aW9uUHJvcGVydHkge1xuICAgIGNvbnN0IG91dHB1dEFydGlmYWN0cyA9IGNkay5MYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5yZW5kZXJBcnRpZmFjdHMoYWN0aW9uLm91dHB1dHMpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSk7XG4gICAgY29uc3QgaW5wdXRBcnRpZmFjdHMgPSBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQXJ0aWZhY3RzKGFjdGlvbi5pbnB1dHMpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSk7XG4gICAgcmV0dXJuIHtcbiAgICAgIG5hbWU6IGFjdGlvbi5hY3Rpb25OYW1lLFxuICAgICAgaW5wdXRBcnRpZmFjdHMsXG4gICAgICBvdXRwdXRBcnRpZmFjdHMsXG4gICAgICBhY3Rpb25UeXBlSWQ6IHtcbiAgICAgICAgY2F0ZWdvcnk6IGFjdGlvbi5jYXRlZ29yeS50b1N0cmluZygpLFxuICAgICAgICB2ZXJzaW9uOiBhY3Rpb24udmVyc2lvbixcbiAgICAgICAgb3duZXI6IGFjdGlvbi5vd25lcixcbiAgICAgICAgcHJvdmlkZXI6IGFjdGlvbi5wcm92aWRlcixcbiAgICAgIH0sXG4gICAgICBjb25maWd1cmF0aW9uOiBhY3Rpb24uY29uZmlndXJhdGlvbixcbiAgICAgIHJ1bk9yZGVyOiBhY3Rpb24ucnVuT3JkZXIsXG4gICAgICByb2xlQXJuOiBhY3Rpb24ucm9sZSA/IGFjdGlvbi5yb2xlLnJvbGVBcm4gOiB1bmRlZmluZWQsXG4gICAgICByZWdpb246IGFjdGlvbi5yZWdpb24sXG4gICAgICBuYW1lc3BhY2U6IGFjdGlvbi5uYW1lc3BhY2UsXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgcmVuZGVyQXJ0aWZhY3RzKGFydGlmYWN0czogQXJ0aWZhY3RbXSk6IENmblBpcGVsaW5lLklucHV0QXJ0aWZhY3RQcm9wZXJ0eVtdIHtcbiAgICByZXR1cm4gYXJ0aWZhY3RzXG4gICAgICAuZmlsdGVyKGEgPT4gYS5hcnRpZmFjdE5hbWUpXG4gICAgICAubWFwKGEgPT4gKHsgbmFtZTogYS5hcnRpZmFjdE5hbWUhIH0pKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBzYW5pdGl6ZUFydGlmYWN0TmFtZShhcnRpZmFjdE5hbWU6IHN0cmluZyk6IHN0cmluZyB7XG4gIC8vIHN0cmlwIG91dCBzb21lIGNoYXJhY3RlcnMgdGhhdCBhcmUgbGVnYWwgaW4gU3RhZ2UgYW5kIEFjdGlvbiBuYW1lcyxcbiAgLy8gYnV0IG5vdCBpbiBBcnRpZmFjdCBuYW1lc1xuICByZXR1cm4gYXJ0aWZhY3ROYW1lLnJlcGxhY2UoL1tALl0vZywgJycpO1xufVxuIl19