"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const codepipeline = require("../lib");
const validations = require("../lib/private/validation");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
/* eslint-disable quote-props */
nodeunit_shim_1.nodeunitShim({
    'artifact bounds validation': {
        'artifacts count exceed maximum'(test) {
            const result = boundsValidationResult(1, 0, 0);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/cannot have more than 0/), 'the validation should have failed');
            test.done();
        },
        'artifacts count below minimum'(test) {
            const result = boundsValidationResult(1, 2, 2);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/must have at least 2/), 'the validation should have failed');
            test.done();
        },
        'artifacts count within bounds'(test) {
            const result = boundsValidationResult(1, 0, 2);
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action type validation': {
        'must be source and is source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
        'must be source and is not source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only contain Source actions/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only occur in first stage/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is not source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action name validation': {
        'throws an exception when adding an Action with an empty name to the Pipeline'(test) {
            const stack = new cdk.Stack();
            const action = new fake_source_action_1.FakeSourceAction({
                actionName: '',
                output: new codepipeline.Artifact(),
            });
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Source' });
            test.throws(() => {
                stage.addAction(action);
            }, /Action name must match regular expression:/);
            test.done();
        },
    },
    'action Artifacts validation': {
        'validates that input Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                extraInputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Build\/Fake cannot have more than 3 input artifacts/);
            test.done();
        },
        'validates that output Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            const extraOutput4 = new codepipeline.Artifact('A4');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                    extraOutput4,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Source\/Fake cannot have more than 4 output artifacts/);
            test.done();
        },
    },
    'automatically assigns artifact names to the Actions'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'pipeline');
        const sourceOutput = new codepipeline.Artifact();
        const sourceAction = new fake_source_action_1.FakeSourceAction({
            actionName: 'CodeCommit',
            output: sourceOutput,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [sourceAction],
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: sourceOutput,
                    output: new codepipeline.Artifact(),
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Name': 'Source',
                    'Actions': [
                        {
                            'Name': 'CodeCommit',
                            'OutputArtifacts': [
                                {
                                    'Name': 'Artifact_Source_CodeCommit',
                                },
                            ],
                        },
                    ],
                },
                {
                    'Name': 'Build',
                    'Actions': [
                        {
                            'Name': 'CodeBuild',
                            'InputArtifacts': [
                                {
                                    'Name': 'Artifact_Source_CodeCommit',
                                },
                            ],
                            'OutputArtifacts': [
                                {
                                    'Name': 'Artifact_Build_CodeBuild',
                                },
                            ],
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'the same Action can be safely added to 2 different Stages'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'Source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const action = new fake_build_action_1.FakeBuildAction({ actionName: 'FakeAction', input: sourceOutput });
        const stage2 = {
            stageName: 'Stage2',
            actions: [action],
        };
        const stage3 = {
            stageName: 'Stage3',
            actions: [action],
        };
        pipeline.addStage(stage2);
        test.doesNotThrow(() => {
            pipeline.addStage(stage3);
        }, /FakeAction/);
        test.done();
    },
    'input Artifacts': {
        'can be added multiple times to an Action safely'(test) {
            const artifact = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact,
                    extraInputs: [artifact],
                });
            });
            test.done();
        },
        'can have duplicate names'(test) {
            const artifact1 = new codepipeline.Artifact('SomeArtifact');
            const artifact2 = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact1,
                    extraInputs: [artifact2],
                });
            });
            test.done();
        },
    },
    'output Artifacts': {
        'accept multiple Artifacts with the same name safely'(test) {
            test.doesNotThrow(() => {
                new fake_source_action_1.FakeSourceAction({
                    actionName: 'CodeBuild',
                    output: new codepipeline.Artifact('Artifact1'),
                    extraOutputs: [
                        new codepipeline.Artifact('Artifact1'),
                        new codepipeline.Artifact('Artifact1'),
                    ],
                });
            });
            test.done();
        },
    },
    'an Action with a non-AWS owner cannot have a Role passed for it'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const buildStage = pipeline.addStage({ stageName: 'Build' });
        // constructing it is fine
        const buildAction = new fake_build_action_1.FakeBuildAction({
            actionName: 'build',
            input: sourceOutput,
            owner: 'ThirdParty',
            role: new iam.Role(stack, 'Role', {
                assumedBy: new iam.AnyPrincipal(),
            }),
        });
        // an attempt to add it to the Pipeline is where things blow up
        test.throws(() => {
            buildStage.addAction(buildAction);
        }, /Role is not supported for actions with an owner different than 'AWS' - got 'ThirdParty' \(Action: 'build' in Stage: 'Build'\)/);
        test.done();
    },
    'actions can be retrieved from stages they have been added to'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const sourceStage = pipeline.stages[0];
        const buildStage = pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'build1',
                    input: sourceOutput,
                    runOrder: 11,
                }),
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'build2',
                    input: sourceOutput,
                    runOrder: 2,
                }),
            ],
        });
        test.equal(sourceStage.actions.length, 1);
        test.equal(sourceStage.actions[0].actionProperties.actionName, 'source');
        test.equal(buildStage.actions.length, 2);
        test.equal(buildStage.actions[0].actionProperties.actionName, 'build1');
        test.equal(buildStage.actions[1].actionProperties.actionName, 'build2');
        test.done();
    },
});
function boundsValidationResult(numberOfArtifacts, min, max) {
    const artifacts = [];
    for (let i = 0; i < numberOfArtifacts; i++) {
        artifacts.push(new codepipeline.Artifact(`TestArtifact${i}`));
    }
    return validations.validateArtifactBounds('output', artifacts, min, max, 'testCategory', 'testProvider');
}
//# sourceMappingURL=data:application/json;base64,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