"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
let app;
let stack;
let sourceArtifact;
let initialStages;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'PipelineStack', { env: { account: '2222', region: 'us-east-1' } });
    sourceArtifact = new codepipeline.Artifact();
    initialStages = [
        {
            stageName: 'Source',
            actions: [new fake_source_action_1.FakeSourceAction({
                    actionName: 'Source',
                    output: sourceArtifact,
                })],
        },
        {
            stageName: 'Build',
            actions: [new fake_build_action_1.FakeBuildAction({
                    actionName: 'Build',
                    input: sourceArtifact,
                })],
        },
    ];
});
describe('crossAccountKeys=false', () => {
    let pipeline;
    beforeEach(() => {
        pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            crossAccountKeys: false,
            stages: initialStages,
        });
    });
    test('creates a bucket but no keys', () => {
        // THEN
        expect(stack).not.toHaveResource('AWS::KMS::Key');
        expect(stack).toHaveResource('AWS::S3::Bucket');
    });
    describe('prevents adding a cross-account action', () => {
        const expectedError = 'crossAccountKeys: true';
        let stage;
        beforeEach(() => {
            stage = pipeline.addStage({ stageName: 'Deploy' });
        });
        test('by role', () => {
            // WHEN
            expect(() => {
                stage.addAction(new fake_build_action_1.FakeBuildAction({
                    actionName: 'Deploy',
                    input: sourceArtifact,
                    role: iam.Role.fromRoleArn(stack, 'Role', 'arn:aws:iam::1111:role/some-role'),
                }));
            }).toThrow(expectedError);
        });
        test('by resource', () => {
            // WHEN
            expect(() => {
                stage.addAction(new fake_build_action_1.FakeBuildAction({
                    actionName: 'Deploy',
                    input: sourceArtifact,
                    resource: s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
                        bucketName: 'foo',
                        account: '1111',
                    }),
                }));
            }).toThrow(expectedError);
        });
        test('by declared account', () => {
            // WHEN
            expect(() => {
                stage.addAction(new fake_build_action_1.FakeBuildAction({
                    actionName: 'Deploy',
                    input: sourceArtifact,
                    account: '1111',
                }));
            }).toThrow(expectedError);
        });
    });
    describe('also affects cross-region support stacks', () => {
        let stage;
        beforeEach(() => {
            stage = pipeline.addStage({ stageName: 'Deploy' });
        });
        test('when making a support stack', () => {
            // WHEN
            stage.addAction(new fake_build_action_1.FakeBuildAction({
                actionName: 'Deploy',
                input: sourceArtifact,
                // No resource to grab onto forces creating a fresh support stack
                region: 'eu-west-1',
            }));
            // THEN
            const asm = app.synth();
            const supportStack = asm.getStack(`${stack.stackName}-support-eu-west-1`);
            // THEN
            expect(supportStack).not.toHaveResource('AWS::KMS::Key');
            expect(supportStack).toHaveResource('AWS::S3::Bucket');
        });
        test('when twiddling another stack', () => {
            const stack2 = new core_1.Stack(app, 'Stack2', { env: { account: '2222', region: 'eu-west-1' } });
            // WHEN
            stage.addAction(new fake_build_action_1.FakeBuildAction({
                actionName: 'Deploy',
                input: sourceArtifact,
                resource: new iam.User(stack2, 'DoesntMatterWhatThisIs'),
            }));
            // THEN
            expect(stack2).not.toHaveResource('AWS::KMS::Key');
            expect(stack2).toHaveResource('AWS::S3::Bucket');
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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