"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const artifact_1 = require("../lib/artifact");
const pipeline_1 = require("../lib/pipeline");
const validation_1 = require("../lib/private/validation");
const fake_source_action_1 = require("./fake-source-action");
nodeunit_shim_1.nodeunitShim({
    'name validation'(test) {
        const cases = [
            { name: 'BlahBleep123.@-_', shouldPassValidation: true, explanation: 'should be valid' },
            { name: '', shouldPassValidation: false, explanation: 'the empty string should be invalid' },
            { name: ' BlahBleep', shouldPassValidation: false, explanation: 'spaces should be invalid' },
            { name: '!BlahBleep', shouldPassValidation: false, explanation: '\'!\' should be invalid' },
        ];
        cases.forEach(testCase => {
            const name = testCase.name;
            const validationBlock = () => { validation_1.validateName('test thing', name); };
            if (testCase.shouldPassValidation) {
                test.doesNotThrow(validationBlock, Error, `${name} failed validation but ${testCase.explanation}`);
            }
            else {
                test.throws(validationBlock, Error, `${name} passed validation but ${testCase.explanation}`);
            }
        });
        test.done();
    },
    'Stage validation': {
        'should fail if Stage has no Actions'(test) {
            const stage = stageForTesting();
            test.deepEqual(stage.validate().length, 1);
            test.done();
        },
    },
    'Pipeline validation': {
        'should fail if Pipeline has no Stages'(test) {
            const stack = new cdk.Stack();
            const pipeline = new pipeline_1.Pipeline(stack, 'Pipeline');
            test.deepEqual(cdk.ConstructNode.validate(pipeline.node).length, 1);
            test.done();
        },
        'should fail if Pipeline has a Source Action in a non-first Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new pipeline_1.Pipeline(stack, 'Pipeline');
            pipeline.addStage({
                stageName: 'FirstStage',
                actions: [
                    new fake_source_action_1.FakeSourceAction({
                        actionName: 'FakeSource',
                        output: new artifact_1.Artifact(),
                    }),
                ],
            });
            test.deepEqual(cdk.ConstructNode.validate(pipeline.node).length, 1);
            test.done();
        },
    },
});
function stageForTesting() {
    const stack = new cdk.Stack();
    const pipeline = new pipeline_1.Pipeline(stack, 'Pipeline');
    return pipeline.addStage({ stageName: 'stage' });
}
//# sourceMappingURL=data:application/json;base64,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