"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const nodeunit_shim_1 = require("nodeunit-shim");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
/* eslint-disable quote-props */
nodeunit_shim_1.nodeunitShim({
    'Pipeline': {
        'can be passed an IAM role during pipeline creation'(test) {
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                role,
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'RoleArn': {
                    'Fn::GetAtt': [
                        'Role1ABCC5F0',
                        'Arn',
                    ],
                },
            }));
            test.done();
        },
        'can be imported by ARN'(test) {
            const stack = new cdk.Stack();
            const pipeline = codepipeline.Pipeline.fromPipelineArn(stack, 'Pipeline', 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineArn, 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineName, 'MyPipeline');
            test.done();
        },
        'that is cross-region': {
            'validates that source actions are in the same region as the pipeline'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { region: 'us-west-1', account: '123456789012' } });
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
                const sourceStage = pipeline.addStage({
                    stageName: 'Source',
                });
                const sourceAction = new fake_source_action_1.FakeSourceAction({
                    actionName: 'FakeSource',
                    output: new codepipeline.Artifact(),
                    region: 'ap-southeast-1',
                });
                test.throws(() => {
                    sourceStage.addAction(sourceAction);
                }, /Source action 'FakeSource' must be in the same region as the pipeline/);
                test.done();
            },
            'allows passing an Alias in place of the KMS Key in the replication Bucket'(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const replicationStack = new cdk.Stack(app, 'ReplicationStack', {
                    env: { region: replicationRegion, account: '123456789012' },
                });
                const replicationKey = new kms.Key(replicationStack, 'ReplicationKey');
                const replicationAlias = replicationKey.addAlias('alias/my-replication-alias');
                const replicationBucket = new s3.Bucket(replicationStack, 'ReplicationBucket', {
                    encryptionKey: replicationAlias,
                    bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: replicationBucket,
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                expect(pipelineStack).toHaveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':kms:us-west-1:123456789012:alias/my-replication-alias',
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                });
                expect(replicationStack).toHaveResourceLike('AWS::KMS::Key', {
                    'KeyPolicy': {
                        'Statement': [
                            {
                            // owning account management permissions - we don't care about them in this test
                            },
                            {
                                // KMS verifies whether the principal given in its key policy exists when creating that key.
                                // Since the replication bucket must be deployed before the pipeline,
                                // we cannot put the pipeline role as the principal here -
                                // hence, we put the account itself
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': ['', [
                                                'arn:',
                                                { 'Ref': 'AWS::Partition' },
                                                ':iam::123456789012:root',
                                            ]],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                    },
                });
                test.done();
            },
            "generates ArtifactStores with the alias' ARN as the KeyID"(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'MyStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                expect(pipelineStack).toHaveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':kms:us-west-1:123456789012:alias/s-west-1tencryptionalias9b344b2b8e6825cb1f7d',
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                });
                expect(pipeline.crossRegionSupport[replicationRegion].stack).toHaveResourceLike('AWS::KMS::Alias', {
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                }, assert_1.ResourcePart.CompleteDefinition);
                test.done();
            },
            'allows passing an imported Bucket and Key for the replication Bucket'(test) {
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(undefined, undefined, {
                    env: { region: pipelineRegion },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: s3.Bucket.fromBucketAttributes(pipelineStack, 'ReplicationBucket', {
                            bucketArn: 'arn:aws:s3:::my-us-west-1-replication-bucket',
                            encryptionKey: kms.Key.fromKeyArn(pipelineStack, 'ReplicationKey', `arn:aws:kms:${replicationRegion}:123456789012:key/1234-5678-9012`),
                        }),
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                })],
                        },
                    ],
                });
                expect(pipelineStack).toHaveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'Location': 'my-us-west-1-replication-bucket',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': 'arn:aws:kms:us-west-1:123456789012:key/1234-5678-9012',
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                });
                test.done();
            },
            'generates the support stack containing the replication Bucket without the need to bootstrap in that environment'(test) {
                const app = new cdk.App({
                    treeMetadata: false,
                });
                app.node.setContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT, true);
                const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                    env: { region: 'us-west-2', account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: 'eu-south-1',
                                })],
                        },
                    ],
                });
                const assembly = app.synth();
                const supportStackArtifact = assembly.getStackByName('PipelineStack-support-eu-south-1');
                test.equal(supportStackArtifact.assumeRoleArn, 'arn:${AWS::Partition}:iam::123456789012:role/cdk-hnb659fds-deploy-role-123456789012-us-west-2');
                test.equal(supportStackArtifact.cloudFormationExecutionRoleArn, 'arn:${AWS::Partition}:iam::123456789012:role/cdk-hnb659fds-cfn-exec-role-123456789012-us-west-2');
                test.done();
            },
        },
        'that is cross-account': {
            'does not allow passing a dynamic value in the Action account property'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { account: '123456789012' } });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: cdk.Aws.ACCOUNT_ID,
                    }));
                }, /The 'account' property must be a concrete value \(action: 'FakeBuild'\)/);
                test.done();
            },
            'does not allow an env-agnostic Pipeline Stack if an Action account has been provided'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack');
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: '123456789012',
                    }));
                }, /Pipeline stack which uses cross-environment actions must have an explicitly set account/);
                test.done();
            },
        },
    },
});
describe('test with shared setup', () => {
    let stack;
    let sourceArtifact;
    beforeEach(() => {
        stack = new cdk.Stack();
        sourceArtifact = new codepipeline.Artifact();
    });
    test('can add actions to stages after creation', () => {
        // GIVEN
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Fetch', output: sourceArtifact })],
                },
                {
                    stageName: 'Build',
                    actions: [new fake_build_action_1.FakeBuildAction({ actionName: 'Gcc', input: sourceArtifact })],
                },
            ],
        });
        // WHEN
        pipeline.stage('Build').addAction(new fake_build_action_1.FakeBuildAction({ actionName: 'debug.com', input: sourceArtifact }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::CodePipeline::Pipeline', {
            Stages: assert_1.arrayWith({
                Name: 'Build',
                Actions: [
                    assert_1.objectLike({ Name: 'Gcc' }),
                    assert_1.objectLike({ Name: 'debug.com' }),
                ],
            }),
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInBpcGVsaW5lLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw0Q0FBNkc7QUFDN0csZ0NBQThCO0FBQzlCLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsc0NBQXNDO0FBQ3RDLHFDQUFxQztBQUNyQyx5Q0FBeUM7QUFDekMsaURBQW1EO0FBQ25ELHVDQUF1QztBQUN2QywyREFBc0Q7QUFDdEQsNkRBQXdEO0FBRXhELGdDQUFnQztBQUVoQyw0QkFBWSxDQUFDO0lBQ1gsVUFBVSxFQUFFO1FBQ1Ysb0RBQW9ELENBQUMsSUFBVTtZQUM3RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDRCQUE0QixDQUFDO2FBQ2xFLENBQUMsQ0FBQztZQUNILElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUMzQyxJQUFJO2FBQ0wsQ0FBQyxDQUFDO1lBRUgsZUFBUyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsNkJBQTZCLEVBQUU7Z0JBQ3hFLFNBQVMsRUFBRTtvQkFDVCxZQUFZLEVBQUU7d0JBQ1osY0FBYzt3QkFDZCxLQUFLO3FCQUNOO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsd0JBQXdCLENBQUMsSUFBVTtZQUNqQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixNQUFNLFFBQVEsR0FBRyxZQUFZLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUN0RSx3REFBd0QsQ0FBQyxDQUFDO1lBRTVELElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRSx3REFBd0QsQ0FBQyxDQUFDO1lBQzNGLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxZQUFZLENBQUMsQ0FBQztZQUVoRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsc0JBQXNCLEVBQUU7WUFDdEIsc0VBQXNFLENBQUMsSUFBVTtnQkFDL0UsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7Z0JBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUM3RyxNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO2dCQUM5RCxNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDO29CQUNwQyxTQUFTLEVBQUUsUUFBUTtpQkFDcEIsQ0FBQyxDQUFDO2dCQUNILE1BQU0sWUFBWSxHQUFHLElBQUkscUNBQWdCLENBQUM7b0JBQ3hDLFVBQVUsRUFBRSxZQUFZO29CQUN4QixNQUFNLEVBQUUsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFO29CQUNuQyxNQUFNLEVBQUUsZ0JBQWdCO2lCQUN6QixDQUFDLENBQUM7Z0JBRUgsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ2YsV0FBVyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQztnQkFDdEMsQ0FBQyxFQUFFLHVFQUF1RSxDQUFDLENBQUM7Z0JBRTVFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCwyRUFBMkUsQ0FBQyxJQUFVO2dCQUNwRixNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztnQkFFMUIsTUFBTSxpQkFBaUIsR0FBRyxXQUFXLENBQUM7Z0JBQ3RDLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxrQkFBa0IsRUFBRTtvQkFDOUQsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLGlCQUFpQixFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUU7aUJBQzVELENBQUMsQ0FBQztnQkFDSCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztnQkFDdkUsTUFBTSxnQkFBZ0IsR0FBRyxjQUFjLENBQUMsUUFBUSxDQUFDLDRCQUE0QixDQUFDLENBQUM7Z0JBQy9FLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLG1CQUFtQixFQUFFO29CQUM3RSxhQUFhLEVBQUUsZ0JBQWdCO29CQUMvQixVQUFVLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxrQkFBa0I7aUJBQ2hELENBQUMsQ0FBQztnQkFFSCxNQUFNLGNBQWMsR0FBRyxXQUFXLENBQUM7Z0JBQ25DLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxFQUFFO29CQUN4RCxHQUFHLEVBQUUsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUU7aUJBQ3pELENBQUMsQ0FBQztnQkFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDakQsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLGFBQWEsRUFBRSxVQUFVLEVBQUU7b0JBQ25ELDZCQUE2QixFQUFFO3dCQUM3QixDQUFDLGlCQUFpQixDQUFDLEVBQUUsaUJBQWlCO3FCQUN2QztvQkFDRCxNQUFNLEVBQUU7d0JBQ047NEJBQ0UsU0FBUyxFQUFFLFFBQVE7NEJBQ25CLE9BQU8sRUFBRSxDQUFDLElBQUkscUNBQWdCLENBQUM7b0NBQzdCLFVBQVUsRUFBRSxRQUFRO29DQUNwQixNQUFNLEVBQUUsWUFBWTtpQ0FDckIsQ0FBQyxDQUFDO3lCQUNKO3dCQUNEOzRCQUNFLFNBQVMsRUFBRSxPQUFPOzRCQUNsQixPQUFPLEVBQUUsQ0FBQyxJQUFJLG1DQUFlLENBQUM7b0NBQzVCLFVBQVUsRUFBRSxPQUFPO29DQUNuQixLQUFLLEVBQUUsWUFBWTtvQ0FDbkIsTUFBTSxFQUFFLGlCQUFpQjtpQ0FDMUIsQ0FBQyxDQUFDO3lCQUNKO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsa0JBQWtCLENBQUMsNkJBQTZCLEVBQUU7b0JBQ3RFLGdCQUFnQixFQUFFO3dCQUNoQjs0QkFDRSxRQUFRLEVBQUUsaUJBQWlCOzRCQUMzQixlQUFlLEVBQUU7Z0NBQ2YsTUFBTSxFQUFFLElBQUk7Z0NBQ1osZUFBZSxFQUFFO29DQUNmLE1BQU0sRUFBRSxLQUFLO29DQUNiLElBQUksRUFBRTt3Q0FDSixVQUFVLEVBQUU7NENBQ1YsRUFBRTs0Q0FDRjtnREFDRSxNQUFNO2dEQUNOO29EQUNFLEtBQUssRUFBRSxnQkFBZ0I7aURBQ3hCO2dEQUNELHdEQUF3RDs2Q0FDekQ7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsUUFBUSxFQUFFLGNBQWM7eUJBQ3pCO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxNQUFNLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlLEVBQUU7b0JBQzNELFdBQVcsRUFBRTt3QkFDWCxXQUFXLEVBQUU7NEJBQ1g7NEJBQ0UsZ0ZBQWdGOzZCQUNqRjs0QkFDRDtnQ0FDRSw0RkFBNEY7Z0NBQzVGLHFFQUFxRTtnQ0FDckUsMERBQTBEO2dDQUMxRCxtQ0FBbUM7Z0NBQ25DLFFBQVEsRUFBRTtvQ0FDUixhQUFhO29DQUNiLGlCQUFpQjtvQ0FDakIsYUFBYTtvQ0FDYixnQkFBZ0I7b0NBQ2hCLHNCQUFzQjtpQ0FDdkI7Z0NBQ0QsUUFBUSxFQUFFLE9BQU87Z0NBQ2pCLFdBQVcsRUFBRTtvQ0FDWCxLQUFLLEVBQUU7d0NBQ0wsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFO2dEQUNmLE1BQU07Z0RBQ04sRUFBRSxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7Z0RBQzNCLHlCQUF5Qjs2Q0FDMUIsQ0FBQztxQ0FDSDtpQ0FDRjtnQ0FDRCxVQUFVLEVBQUUsR0FBRzs2QkFDaEI7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDO2dCQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCwyREFBMkQsQ0FBQyxJQUFVO2dCQUNwRSxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztnQkFDMUIsTUFBTSxpQkFBaUIsR0FBRyxXQUFXLENBQUM7Z0JBRXRDLE1BQU0sY0FBYyxHQUFHLFdBQVcsQ0FBQztnQkFDbkMsTUFBTSxhQUFhLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUU7b0JBQ2xELEdBQUcsRUFBRSxFQUFFLE1BQU0sRUFBRSxjQUFjLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRTtpQkFDekQsQ0FBQyxDQUFDO2dCQUNILE1BQU0sWUFBWSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDO2dCQUNqRCxNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsYUFBYSxFQUFFLFVBQVUsRUFBRTtvQkFDcEUsTUFBTSxFQUFFO3dCQUNOOzRCQUNFLFNBQVMsRUFBRSxRQUFROzRCQUNuQixPQUFPLEVBQUUsQ0FBQyxJQUFJLHFDQUFnQixDQUFDO29DQUM3QixVQUFVLEVBQUUsUUFBUTtvQ0FDcEIsTUFBTSxFQUFFLFlBQVk7aUNBQ3JCLENBQUMsQ0FBQzt5QkFDSjt3QkFDRDs0QkFDRSxTQUFTLEVBQUUsT0FBTzs0QkFDbEIsT0FBTyxFQUFFLENBQUMsSUFBSSxtQ0FBZSxDQUFDO29DQUM1QixVQUFVLEVBQUUsT0FBTztvQ0FDbkIsS0FBSyxFQUFFLFlBQVk7b0NBQ25CLE1BQU0sRUFBRSxpQkFBaUI7aUNBQzFCLENBQUMsQ0FBQzt5QkFDSjtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsTUFBTSxDQUFDLGFBQWEsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDZCQUE2QixFQUFFO29CQUN0RSxnQkFBZ0IsRUFBRTt3QkFDaEI7NEJBQ0UsUUFBUSxFQUFFLGlCQUFpQjs0QkFDM0IsZUFBZSxFQUFFO2dDQUNmLE1BQU0sRUFBRSxJQUFJO2dDQUNaLGVBQWUsRUFBRTtvQ0FDZixNQUFNLEVBQUUsS0FBSztvQ0FDYixJQUFJLEVBQUU7d0NBQ0osVUFBVSxFQUFFOzRDQUNWLEVBQUU7NENBQ0Y7Z0RBQ0UsTUFBTTtnREFDTjtvREFDRSxLQUFLLEVBQUUsZ0JBQWdCO2lEQUN4QjtnREFDRCxnRkFBZ0Y7NkNBQ2pGO3lDQUNGO3FDQUNGO2lDQUNGOzZCQUNGO3lCQUNGO3dCQUNEOzRCQUNFLFFBQVEsRUFBRSxjQUFjO3lCQUN6QjtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsTUFBTSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGlCQUFpQixFQUFFO29CQUNqRyxnQkFBZ0IsRUFBRSxRQUFRO29CQUMxQixxQkFBcUIsRUFBRSxRQUFRO2lCQUNoQyxFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQztnQkFFcEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELHNFQUFzRSxDQUFDLElBQVU7Z0JBQy9FLE1BQU0saUJBQWlCLEdBQUcsV0FBVyxDQUFDO2dCQUV0QyxNQUFNLGNBQWMsR0FBRyxXQUFXLENBQUM7Z0JBQ25DLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO29CQUN4RCxHQUFHLEVBQUUsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFO2lCQUNoQyxDQUFDLENBQUM7Z0JBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7Z0JBQ2pELElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxhQUFhLEVBQUUsVUFBVSxFQUFFO29CQUNuRCw2QkFBNkIsRUFBRTt3QkFDN0IsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsb0JBQW9CLENBQUMsYUFBYSxFQUFFLG1CQUFtQixFQUFFOzRCQUN0RixTQUFTLEVBQUUsOENBQThDOzRCQUN6RCxhQUFhLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsYUFBYSxFQUFFLGdCQUFnQixFQUMvRCxlQUFlLGlCQUFpQixrQ0FBa0MsQ0FDbkU7eUJBQ0YsQ0FBQztxQkFDSDtvQkFDRCxNQUFNLEVBQUU7d0JBQ047NEJBQ0UsU0FBUyxFQUFFLFFBQVE7NEJBQ25CLE9BQU8sRUFBRSxDQUFDLElBQUkscUNBQWdCLENBQUM7b0NBQzdCLFVBQVUsRUFBRSxRQUFRO29DQUNwQixNQUFNLEVBQUUsWUFBWTtpQ0FDckIsQ0FBQyxDQUFDO3lCQUNKO3dCQUNEOzRCQUNFLFNBQVMsRUFBRSxPQUFPOzRCQUNsQixPQUFPLEVBQUUsQ0FBQyxJQUFJLG1DQUFlLENBQUM7b0NBQzVCLFVBQVUsRUFBRSxPQUFPO29DQUNuQixLQUFLLEVBQUUsWUFBWTtpQ0FDcEIsQ0FBQyxDQUFDO3lCQUNKO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsa0JBQWtCLENBQUMsNkJBQTZCLEVBQUU7b0JBQ3RFLGdCQUFnQixFQUFFO3dCQUNoQjs0QkFDRSxRQUFRLEVBQUUsaUJBQWlCOzRCQUMzQixlQUFlLEVBQUU7Z0NBQ2YsTUFBTSxFQUFFLElBQUk7Z0NBQ1osVUFBVSxFQUFFLGlDQUFpQztnQ0FDN0MsZUFBZSxFQUFFO29DQUNmLE1BQU0sRUFBRSxLQUFLO29DQUNiLElBQUksRUFBRSx1REFBdUQ7aUNBQzlEOzZCQUNGO3lCQUNGO3dCQUNEOzRCQUNFLFFBQVEsRUFBRSxjQUFjO3lCQUN6QjtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELGlIQUFpSCxDQUFDLElBQVU7Z0JBQzFILE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQztvQkFDdEIsWUFBWSxFQUFFLEtBQUs7aUJBQ3BCLENBQUMsQ0FBQztnQkFDSCxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsaUNBQWlDLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBRW5FLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxFQUFFO29CQUN4RCxHQUFHLEVBQUUsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUU7aUJBQ3RELENBQUMsQ0FBQztnQkFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDakQsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLGFBQWEsRUFBRSxVQUFVLEVBQUU7b0JBQ25ELE1BQU0sRUFBRTt3QkFDTjs0QkFDRSxTQUFTLEVBQUUsUUFBUTs0QkFDbkIsT0FBTyxFQUFFLENBQUMsSUFBSSxxQ0FBZ0IsQ0FBQztvQ0FDN0IsVUFBVSxFQUFFLFFBQVE7b0NBQ3BCLE1BQU0sRUFBRSxZQUFZO2lDQUNyQixDQUFDLENBQUM7eUJBQ0o7d0JBQ0Q7NEJBQ0UsU0FBUyxFQUFFLE9BQU87NEJBQ2xCLE9BQU8sRUFBRSxDQUFDLElBQUksbUNBQWUsQ0FBQztvQ0FDNUIsVUFBVSxFQUFFLE9BQU87b0NBQ25CLEtBQUssRUFBRSxZQUFZO29DQUNuQixNQUFNLEVBQUUsWUFBWTtpQ0FDckIsQ0FBQyxDQUFDO3lCQUNKO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQzdCLE1BQU0sb0JBQW9CLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO2dCQUN6RixJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDLGFBQWEsRUFDM0MsK0ZBQStGLENBQUMsQ0FBQztnQkFDbkcsSUFBSSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyw4QkFBOEIsRUFDNUQsaUdBQWlHLENBQUMsQ0FBQztnQkFFckcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO1FBRUQsdUJBQXVCLEVBQUU7WUFDdkIsdUVBQXVFLENBQUMsSUFBVTtnQkFDaEYsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7Z0JBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsZUFBZSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFDeEYsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7Z0JBQ2pELE1BQU0sUUFBUSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO29CQUM1RCxNQUFNLEVBQUU7d0JBQ047NEJBQ0UsU0FBUyxFQUFFLFFBQVE7NEJBQ25CLE9BQU8sRUFBRSxDQUFDLElBQUkscUNBQWdCLENBQUMsRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDO3lCQUNoRjtxQkFDRjtpQkFDRixDQUFDLENBQUM7Z0JBQ0gsTUFBTSxVQUFVLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO2dCQUU3RCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtvQkFDZixVQUFVLENBQUMsU0FBUyxDQUFDLElBQUksbUNBQWUsQ0FBQzt3QkFDdkMsVUFBVSxFQUFFLFdBQVc7d0JBQ3ZCLEtBQUssRUFBRSxZQUFZO3dCQUNuQixPQUFPLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVO3FCQUM1QixDQUFDLENBQUMsQ0FBQztnQkFDTixDQUFDLEVBQUUseUVBQXlFLENBQUMsQ0FBQztnQkFFOUUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELHNGQUFzRixDQUFDLElBQVU7Z0JBQy9GLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO2dCQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLGVBQWUsQ0FBQyxDQUFDO2dCQUNsRCxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDakQsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7b0JBQzVELE1BQU0sRUFBRTt3QkFDTjs0QkFDRSxTQUFTLEVBQUUsUUFBUTs0QkFDbkIsT0FBTyxFQUFFLENBQUMsSUFBSSxxQ0FBZ0IsQ0FBQyxFQUFFLFVBQVUsRUFBRSxRQUFRLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxDQUFDLENBQUM7eUJBQ2hGO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztnQkFDSCxNQUFNLFVBQVUsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7Z0JBRTdELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO29CQUNmLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxtQ0FBZSxDQUFDO3dCQUN2QyxVQUFVLEVBQUUsV0FBVzt3QkFDdkIsS0FBSyxFQUFFLFlBQVk7d0JBQ25CLE9BQU8sRUFBRSxjQUFjO3FCQUN4QixDQUFDLENBQUMsQ0FBQztnQkFDTixDQUFDLEVBQUUseUZBQXlGLENBQUMsQ0FBQztnQkFFOUYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO0tBQ0Y7Q0FDRixDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO0lBQ3RDLElBQUksS0FBZ0IsQ0FBQztJQUNyQixJQUFJLGNBQXFDLENBQUM7SUFDMUMsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN4QixjQUFjLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDL0MsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMENBQTBDLEVBQUUsR0FBRyxFQUFFO1FBQ3BELFFBQVE7UUFDUixNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUM1RCxNQUFNLEVBQUU7Z0JBQ047b0JBQ0UsU0FBUyxFQUFFLFFBQVE7b0JBQ25CLE9BQU8sRUFBRSxDQUFDLElBQUkscUNBQWdCLENBQUMsRUFBRSxVQUFVLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO2lCQUNqRjtnQkFDRDtvQkFDRSxTQUFTLEVBQUUsT0FBTztvQkFDbEIsT0FBTyxFQUFFLENBQUMsSUFBSSxtQ0FBZSxDQUFDLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztpQkFDN0U7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxRQUFRLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFNBQVMsQ0FBQyxJQUFJLG1DQUFlLENBQUMsRUFBRSxVQUFVLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFM0csT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUM5RCxNQUFNLEVBQUUsa0JBQVMsQ0FBQztnQkFDaEIsSUFBSSxFQUFFLE9BQU87Z0JBQ2IsT0FBTyxFQUFFO29CQUNQLG1CQUFVLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLENBQUM7b0JBQzNCLG1CQUFVLENBQUMsRUFBRSxJQUFJLEVBQUUsV0FBVyxFQUFFLENBQUM7aUJBQ2xDO2FBQ0YsQ0FBQztTQUNILENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QgYXMgb3VyRXhwZWN0LCBSZXNvdXJjZVBhcnQsIGFycmF5V2l0aCwgb2JqZWN0TGlrZSwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGttcyBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tICdAYXdzLWNkay9jeC1hcGknO1xuaW1wb3J0IHsgbm9kZXVuaXRTaGltLCBUZXN0IH0gZnJvbSAnbm9kZXVuaXQtc2hpbSc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IEZha2VCdWlsZEFjdGlvbiB9IGZyb20gJy4vZmFrZS1idWlsZC1hY3Rpb24nO1xuaW1wb3J0IHsgRmFrZVNvdXJjZUFjdGlvbiB9IGZyb20gJy4vZmFrZS1zb3VyY2UtYWN0aW9uJztcblxuLyogZXNsaW50LWRpc2FibGUgcXVvdGUtcHJvcHMgKi9cblxubm9kZXVuaXRTaGltKHtcbiAgJ1BpcGVsaW5lJzoge1xuICAgICdjYW4gYmUgcGFzc2VkIGFuIElBTSByb2xlIGR1cmluZyBwaXBlbGluZSBjcmVhdGlvbicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2NvZGVwaXBlbGluZS5hbWF6b25hd3MuY29tJyksXG4gICAgICB9KTtcbiAgICAgIG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUoc3RhY2ssICdQaXBlbGluZScsIHtcbiAgICAgICAgcm9sZSxcbiAgICAgIH0pO1xuXG4gICAgICBvdXJFeHBlY3Qoc3RhY2ssIHRydWUpLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgJ1JvbGVBcm4nOiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAnUm9sZTFBQkNDNUYwJyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIGJlIGltcG9ydGVkIGJ5IEFSTicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IHBpcGVsaW5lID0gY29kZXBpcGVsaW5lLlBpcGVsaW5lLmZyb21QaXBlbGluZUFybihzdGFjaywgJ1BpcGVsaW5lJyxcbiAgICAgICAgJ2Fybjphd3M6Y29kZXBpcGVsaW5lOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6TXlQaXBlbGluZScpO1xuXG4gICAgICB0ZXN0LmVxdWFsKHBpcGVsaW5lLnBpcGVsaW5lQXJuLCAnYXJuOmF3czpjb2RlcGlwZWxpbmU6dXMtZWFzdC0xOjEyMzQ1Njc4OTAxMjpNeVBpcGVsaW5lJyk7XG4gICAgICB0ZXN0LmVxdWFsKHBpcGVsaW5lLnBpcGVsaW5lTmFtZSwgJ015UGlwZWxpbmUnKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd0aGF0IGlzIGNyb3NzLXJlZ2lvbic6IHtcbiAgICAgICd2YWxpZGF0ZXMgdGhhdCBzb3VyY2UgYWN0aW9ucyBhcmUgaW4gdGhlIHNhbWUgcmVnaW9uIGFzIHRoZSBwaXBlbGluZScodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnUGlwZWxpbmVTdGFjaycsIHsgZW52OiB7IHJlZ2lvbjogJ3VzLXdlc3QtMScsIGFjY291bnQ6ICcxMjM0NTY3ODkwMTInIH0gfSk7XG4gICAgICAgIGNvbnN0IHBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShzdGFjaywgJ1BpcGVsaW5lJyk7XG4gICAgICAgIGNvbnN0IHNvdXJjZVN0YWdlID0gcGlwZWxpbmUuYWRkU3RhZ2Uoe1xuICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBzb3VyY2VBY3Rpb24gPSBuZXcgRmFrZVNvdXJjZUFjdGlvbih7XG4gICAgICAgICAgYWN0aW9uTmFtZTogJ0Zha2VTb3VyY2UnLFxuICAgICAgICAgIG91dHB1dDogbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdCgpLFxuICAgICAgICAgIHJlZ2lvbjogJ2FwLXNvdXRoZWFzdC0xJyxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICAgIHNvdXJjZVN0YWdlLmFkZEFjdGlvbihzb3VyY2VBY3Rpb24pO1xuICAgICAgICB9LCAvU291cmNlIGFjdGlvbiAnRmFrZVNvdXJjZScgbXVzdCBiZSBpbiB0aGUgc2FtZSByZWdpb24gYXMgdGhlIHBpcGVsaW5lLyk7XG5cbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuXG4gICAgICAnYWxsb3dzIHBhc3NpbmcgYW4gQWxpYXMgaW4gcGxhY2Ugb2YgdGhlIEtNUyBLZXkgaW4gdGhlIHJlcGxpY2F0aW9uIEJ1Y2tldCcodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuXG4gICAgICAgIGNvbnN0IHJlcGxpY2F0aW9uUmVnaW9uID0gJ3VzLXdlc3QtMSc7XG4gICAgICAgIGNvbnN0IHJlcGxpY2F0aW9uU3RhY2sgPSBuZXcgY2RrLlN0YWNrKGFwcCwgJ1JlcGxpY2F0aW9uU3RhY2snLCB7XG4gICAgICAgICAgZW52OiB7IHJlZ2lvbjogcmVwbGljYXRpb25SZWdpb24sIGFjY291bnQ6ICcxMjM0NTY3ODkwMTInIH0sXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCByZXBsaWNhdGlvbktleSA9IG5ldyBrbXMuS2V5KHJlcGxpY2F0aW9uU3RhY2ssICdSZXBsaWNhdGlvbktleScpO1xuICAgICAgICBjb25zdCByZXBsaWNhdGlvbkFsaWFzID0gcmVwbGljYXRpb25LZXkuYWRkQWxpYXMoJ2FsaWFzL215LXJlcGxpY2F0aW9uLWFsaWFzJyk7XG4gICAgICAgIGNvbnN0IHJlcGxpY2F0aW9uQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChyZXBsaWNhdGlvblN0YWNrLCAnUmVwbGljYXRpb25CdWNrZXQnLCB7XG4gICAgICAgICAgZW5jcnlwdGlvbktleTogcmVwbGljYXRpb25BbGlhcyxcbiAgICAgICAgICBidWNrZXROYW1lOiBjZGsuUGh5c2ljYWxOYW1lLkdFTkVSQVRFX0lGX05FRURFRCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgcGlwZWxpbmVSZWdpb24gPSAndXMtd2VzdC0yJztcbiAgICAgICAgY29uc3QgcGlwZWxpbmVTdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnUGlwZWxpbmVTdGFjaycsIHtcbiAgICAgICAgICBlbnY6IHsgcmVnaW9uOiBwaXBlbGluZVJlZ2lvbiwgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgICAgICAgbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShwaXBlbGluZVN0YWNrLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgICAgY3Jvc3NSZWdpb25SZXBsaWNhdGlvbkJ1Y2tldHM6IHtcbiAgICAgICAgICAgIFtyZXBsaWNhdGlvblJlZ2lvbl06IHJlcGxpY2F0aW9uQnVja2V0LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3RhZ2VzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtuZXcgRmFrZVNvdXJjZUFjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgIH0pXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHN0YWdlTmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlQnVpbGRBY3Rpb24oe1xuICAgICAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdCdWlsZCcsXG4gICAgICAgICAgICAgICAgaW5wdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgICAgICAgICAgICByZWdpb246IHJlcGxpY2F0aW9uUmVnaW9uLFxuICAgICAgICAgICAgICB9KV0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGV4cGVjdChwaXBlbGluZVN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgICAnQXJ0aWZhY3RTdG9yZXMnOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdSZWdpb24nOiByZXBsaWNhdGlvblJlZ2lvbixcbiAgICAgICAgICAgICAgJ0FydGlmYWN0U3RvcmUnOiB7XG4gICAgICAgICAgICAgICAgJ1R5cGUnOiAnUzMnLFxuICAgICAgICAgICAgICAgICdFbmNyeXB0aW9uS2V5Jzoge1xuICAgICAgICAgICAgICAgICAgJ1R5cGUnOiAnS01TJyxcbiAgICAgICAgICAgICAgICAgICdJZCc6IHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgJ1JlZic6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgICAgJzprbXM6dXMtd2VzdC0xOjEyMzQ1Njc4OTAxMjphbGlhcy9teS1yZXBsaWNhdGlvbi1hbGlhcycsXG4gICAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgICdSZWdpb24nOiBwaXBlbGluZVJlZ2lvbixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgZXhwZWN0KHJlcGxpY2F0aW9uU3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpLTVM6OktleScsIHtcbiAgICAgICAgICAnS2V5UG9saWN5Jzoge1xuICAgICAgICAgICAgJ1N0YXRlbWVudCc6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIC8vIG93bmluZyBhY2NvdW50IG1hbmFnZW1lbnQgcGVybWlzc2lvbnMgLSB3ZSBkb24ndCBjYXJlIGFib3V0IHRoZW0gaW4gdGhpcyB0ZXN0XG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAvLyBLTVMgdmVyaWZpZXMgd2hldGhlciB0aGUgcHJpbmNpcGFsIGdpdmVuIGluIGl0cyBrZXkgcG9saWN5IGV4aXN0cyB3aGVuIGNyZWF0aW5nIHRoYXQga2V5LlxuICAgICAgICAgICAgICAgIC8vIFNpbmNlIHRoZSByZXBsaWNhdGlvbiBidWNrZXQgbXVzdCBiZSBkZXBsb3llZCBiZWZvcmUgdGhlIHBpcGVsaW5lLFxuICAgICAgICAgICAgICAgIC8vIHdlIGNhbm5vdCBwdXQgdGhlIHBpcGVsaW5lIHJvbGUgYXMgdGhlIHByaW5jaXBhbCBoZXJlIC1cbiAgICAgICAgICAgICAgICAvLyBoZW5jZSwgd2UgcHV0IHRoZSBhY2NvdW50IGl0c2VsZlxuICAgICAgICAgICAgICAgICdBY3Rpb24nOiBbXG4gICAgICAgICAgICAgICAgICAna21zOkRlY3J5cHQnLFxuICAgICAgICAgICAgICAgICAgJ2ttczpEZXNjcmliZUtleScsXG4gICAgICAgICAgICAgICAgICAna21zOkVuY3J5cHQnLFxuICAgICAgICAgICAgICAgICAgJ2ttczpSZUVuY3J5cHQqJyxcbiAgICAgICAgICAgICAgICAgICdrbXM6R2VuZXJhdGVEYXRhS2V5KicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAnRWZmZWN0JzogJ0FsbG93JyxcbiAgICAgICAgICAgICAgICAnUHJpbmNpcGFsJzoge1xuICAgICAgICAgICAgICAgICAgJ0FXUyc6IHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogWycnLCBbXG4gICAgICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAgICAgIHsgJ1JlZic6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgICAgICAgICAgICAgICAnOmlhbTo6MTIzNDU2Nzg5MDEyOnJvb3QnLFxuICAgICAgICAgICAgICAgICAgICBdXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAnUmVzb3VyY2UnOiAnKicsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgXCJnZW5lcmF0ZXMgQXJ0aWZhY3RTdG9yZXMgd2l0aCB0aGUgYWxpYXMnIEFSTiBhcyB0aGUgS2V5SURcIih0ZXN0OiBUZXN0KSB7XG4gICAgICAgIGNvbnN0IGFwcCA9IG5ldyBjZGsuQXBwKCk7XG4gICAgICAgIGNvbnN0IHJlcGxpY2F0aW9uUmVnaW9uID0gJ3VzLXdlc3QtMSc7XG5cbiAgICAgICAgY29uc3QgcGlwZWxpbmVSZWdpb24gPSAndXMtd2VzdC0yJztcbiAgICAgICAgY29uc3QgcGlwZWxpbmVTdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnTXlTdGFjaycsIHtcbiAgICAgICAgICBlbnY6IHsgcmVnaW9uOiBwaXBlbGluZVJlZ2lvbiwgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgICAgICAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHBpcGVsaW5lU3RhY2ssICdQaXBlbGluZScsIHtcbiAgICAgICAgICBzdGFnZXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgc3RhZ2VOYW1lOiAnU291cmNlJyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlU291cmNlQWN0aW9uKHtcbiAgICAgICAgICAgICAgICBhY3Rpb25OYW1lOiAnU291cmNlJyxcbiAgICAgICAgICAgICAgICBvdXRwdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgICAgICAgICAgfSldLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgc3RhZ2VOYW1lOiAnQnVpbGQnLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbbmV3IEZha2VCdWlsZEFjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgICAgICBpbnB1dDogc291cmNlT3V0cHV0LFxuICAgICAgICAgICAgICAgIHJlZ2lvbjogcmVwbGljYXRpb25SZWdpb24sXG4gICAgICAgICAgICAgIH0pXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgZXhwZWN0KHBpcGVsaW5lU3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgICAgICdBcnRpZmFjdFN0b3Jlcyc6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ1JlZ2lvbic6IHJlcGxpY2F0aW9uUmVnaW9uLFxuICAgICAgICAgICAgICAnQXJ0aWZhY3RTdG9yZSc6IHtcbiAgICAgICAgICAgICAgICAnVHlwZSc6ICdTMycsXG4gICAgICAgICAgICAgICAgJ0VuY3J5cHRpb25LZXknOiB7XG4gICAgICAgICAgICAgICAgICAnVHlwZSc6ICdLTVMnLFxuICAgICAgICAgICAgICAgICAgJ0lkJzoge1xuICAgICAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAnUmVmJzogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAgICAnOmttczp1cy13ZXN0LTE6MTIzNDU2Nzg5MDEyOmFsaWFzL3Mtd2VzdC0xdGVuY3J5cHRpb25hbGlhczliMzQ0YjJiOGU2ODI1Y2IxZjdkJyxcbiAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ1JlZ2lvbic6IHBpcGVsaW5lUmVnaW9uLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICBleHBlY3QocGlwZWxpbmUuY3Jvc3NSZWdpb25TdXBwb3J0W3JlcGxpY2F0aW9uUmVnaW9uXS5zdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OktNUzo6QWxpYXMnLCB7XG4gICAgICAgICAgJ0RlbGV0aW9uUG9saWN5JzogJ0RlbGV0ZScsXG4gICAgICAgICAgJ1VwZGF0ZVJlcGxhY2VQb2xpY3knOiAnRGVsZXRlJyxcbiAgICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbik7XG5cbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuXG4gICAgICAnYWxsb3dzIHBhc3NpbmcgYW4gaW1wb3J0ZWQgQnVja2V0IGFuZCBLZXkgZm9yIHRoZSByZXBsaWNhdGlvbiBCdWNrZXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgY29uc3QgcmVwbGljYXRpb25SZWdpb24gPSAndXMtd2VzdC0xJztcblxuICAgICAgICBjb25zdCBwaXBlbGluZVJlZ2lvbiA9ICd1cy13ZXN0LTInO1xuICAgICAgICBjb25zdCBwaXBlbGluZVN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgICAgICAgIGVudjogeyByZWdpb246IHBpcGVsaW5lUmVnaW9uIH0sXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBzb3VyY2VPdXRwdXQgPSBuZXcgY29kZXBpcGVsaW5lLkFydGlmYWN0KCk7XG4gICAgICAgIG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUocGlwZWxpbmVTdGFjaywgJ1BpcGVsaW5lJywge1xuICAgICAgICAgIGNyb3NzUmVnaW9uUmVwbGljYXRpb25CdWNrZXRzOiB7XG4gICAgICAgICAgICBbcmVwbGljYXRpb25SZWdpb25dOiBzMy5CdWNrZXQuZnJvbUJ1Y2tldEF0dHJpYnV0ZXMocGlwZWxpbmVTdGFjaywgJ1JlcGxpY2F0aW9uQnVja2V0Jywge1xuICAgICAgICAgICAgICBidWNrZXRBcm46ICdhcm46YXdzOnMzOjo6bXktdXMtd2VzdC0xLXJlcGxpY2F0aW9uLWJ1Y2tldCcsXG4gICAgICAgICAgICAgIGVuY3J5cHRpb25LZXk6IGttcy5LZXkuZnJvbUtleUFybihwaXBlbGluZVN0YWNrLCAnUmVwbGljYXRpb25LZXknLFxuICAgICAgICAgICAgICAgIGBhcm46YXdzOmttczoke3JlcGxpY2F0aW9uUmVnaW9ufToxMjM0NTY3ODkwMTI6a2V5LzEyMzQtNTY3OC05MDEyYCxcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgc3RhZ2VzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtuZXcgRmFrZVNvdXJjZUFjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgIH0pXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHN0YWdlTmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlQnVpbGRBY3Rpb24oe1xuICAgICAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdCdWlsZCcsXG4gICAgICAgICAgICAgICAgaW5wdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgICAgICAgICAgfSldLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICBleHBlY3QocGlwZWxpbmVTdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgICAgJ0FydGlmYWN0U3RvcmVzJzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnUmVnaW9uJzogcmVwbGljYXRpb25SZWdpb24sXG4gICAgICAgICAgICAgICdBcnRpZmFjdFN0b3JlJzoge1xuICAgICAgICAgICAgICAgICdUeXBlJzogJ1MzJyxcbiAgICAgICAgICAgICAgICAnTG9jYXRpb24nOiAnbXktdXMtd2VzdC0xLXJlcGxpY2F0aW9uLWJ1Y2tldCcsXG4gICAgICAgICAgICAgICAgJ0VuY3J5cHRpb25LZXknOiB7XG4gICAgICAgICAgICAgICAgICAnVHlwZSc6ICdLTVMnLFxuICAgICAgICAgICAgICAgICAgJ0lkJzogJ2Fybjphd3M6a21zOnVzLXdlc3QtMToxMjM0NTY3ODkwMTI6a2V5LzEyMzQtNTY3OC05MDEyJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ1JlZ2lvbic6IHBpcGVsaW5lUmVnaW9uLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICdnZW5lcmF0ZXMgdGhlIHN1cHBvcnQgc3RhY2sgY29udGFpbmluZyB0aGUgcmVwbGljYXRpb24gQnVja2V0IHdpdGhvdXQgdGhlIG5lZWQgdG8gYm9vdHN0cmFwIGluIHRoYXQgZW52aXJvbm1lbnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgY29uc3QgYXBwID0gbmV3IGNkay5BcHAoe1xuICAgICAgICAgIHRyZWVNZXRhZGF0YTogZmFsc2UsIC8vIHdlIGNhbid0IHNldCB0aGUgY29udGV4dCBvdGhlcndpc2UsIGJlY2F1c2UgQXBwIHdpbGwgaGF2ZSBhIGNoaWxkXG4gICAgICAgIH0pO1xuICAgICAgICBhcHAubm9kZS5zZXRDb250ZXh0KGN4YXBpLk5FV19TVFlMRV9TVEFDS19TWU5USEVTSVNfQ09OVEVYVCwgdHJ1ZSk7XG5cbiAgICAgICAgY29uc3QgcGlwZWxpbmVTdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnUGlwZWxpbmVTdGFjaycsIHtcbiAgICAgICAgICBlbnY6IHsgcmVnaW9uOiAndXMtd2VzdC0yJywgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHNvdXJjZU91dHB1dCA9IG5ldyBjb2RlcGlwZWxpbmUuQXJ0aWZhY3QoKTtcbiAgICAgICAgbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShwaXBlbGluZVN0YWNrLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgICAgc3RhZ2VzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtuZXcgRmFrZVNvdXJjZUFjdGlvbih7XG4gICAgICAgICAgICAgICAgYWN0aW9uTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICAgIH0pXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIHN0YWdlTmFtZTogJ0J1aWxkJyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlQnVpbGRBY3Rpb24oe1xuICAgICAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdCdWlsZCcsXG4gICAgICAgICAgICAgICAgaW5wdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgICAgICAgICAgICByZWdpb246ICdldS1zb3V0aC0xJyxcbiAgICAgICAgICAgICAgfSldLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuICAgICAgICBjb25zdCBzdXBwb3J0U3RhY2tBcnRpZmFjdCA9IGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKCdQaXBlbGluZVN0YWNrLXN1cHBvcnQtZXUtc291dGgtMScpO1xuICAgICAgICB0ZXN0LmVxdWFsKHN1cHBvcnRTdGFja0FydGlmYWN0LmFzc3VtZVJvbGVBcm4sXG4gICAgICAgICAgJ2Fybjoke0FXUzo6UGFydGl0aW9ufTppYW06OjEyMzQ1Njc4OTAxMjpyb2xlL2Nkay1obmI2NTlmZHMtZGVwbG95LXJvbGUtMTIzNDU2Nzg5MDEyLXVzLXdlc3QtMicpO1xuICAgICAgICB0ZXN0LmVxdWFsKHN1cHBvcnRTdGFja0FydGlmYWN0LmNsb3VkRm9ybWF0aW9uRXhlY3V0aW9uUm9sZUFybixcbiAgICAgICAgICAnYXJuOiR7QVdTOjpQYXJ0aXRpb259OmlhbTo6MTIzNDU2Nzg5MDEyOnJvbGUvY2RrLWhuYjY1OWZkcy1jZm4tZXhlYy1yb2xlLTEyMzQ1Njc4OTAxMi11cy13ZXN0LTInKTtcblxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG4gICAgfSxcblxuICAgICd0aGF0IGlzIGNyb3NzLWFjY291bnQnOiB7XG4gICAgICAnZG9lcyBub3QgYWxsb3cgcGFzc2luZyBhIGR5bmFtaWMgdmFsdWUgaW4gdGhlIEFjdGlvbiBhY2NvdW50IHByb3BlcnR5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIGNvbnN0IGFwcCA9IG5ldyBjZGsuQXBwKCk7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayhhcHAsICdQaXBlbGluZVN0YWNrJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicgfSB9KTtcbiAgICAgICAgY29uc3Qgc291cmNlT3V0cHV0ID0gbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdCgpO1xuICAgICAgICBjb25zdCBwaXBlbGluZSA9IG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUoc3RhY2ssICdQaXBlbGluZScsIHtcbiAgICAgICAgICBzdGFnZXM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgc3RhZ2VOYW1lOiAnU291cmNlJyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlU291cmNlQWN0aW9uKHsgYWN0aW9uTmFtZTogJ1NvdXJjZScsIG91dHB1dDogc291cmNlT3V0cHV0IH0pXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGJ1aWxkU3RhZ2UgPSBwaXBlbGluZS5hZGRTdGFnZSh7IHN0YWdlTmFtZTogJ0J1aWxkJyB9KTtcblxuICAgICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgICAgYnVpbGRTdGFnZS5hZGRBY3Rpb24obmV3IEZha2VCdWlsZEFjdGlvbih7XG4gICAgICAgICAgICBhY3Rpb25OYW1lOiAnRmFrZUJ1aWxkJyxcbiAgICAgICAgICAgIGlucHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgICAgICBhY2NvdW50OiBjZGsuQXdzLkFDQ09VTlRfSUQsXG4gICAgICAgICAgfSkpO1xuICAgICAgICB9LCAvVGhlICdhY2NvdW50JyBwcm9wZXJ0eSBtdXN0IGJlIGEgY29uY3JldGUgdmFsdWUgXFwoYWN0aW9uOiAnRmFrZUJ1aWxkJ1xcKS8pO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgJ2RvZXMgbm90IGFsbG93IGFuIGVudi1hZ25vc3RpYyBQaXBlbGluZSBTdGFjayBpZiBhbiBBY3Rpb24gYWNjb3VudCBoYXMgYmVlbiBwcm92aWRlZCcodGVzdDogVGVzdCkge1xuICAgICAgICBjb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soYXBwLCAnUGlwZWxpbmVTdGFjaycpO1xuICAgICAgICBjb25zdCBzb3VyY2VPdXRwdXQgPSBuZXcgY29kZXBpcGVsaW5lLkFydGlmYWN0KCk7XG4gICAgICAgIGNvbnN0IHBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShzdGFjaywgJ1BpcGVsaW5lJywge1xuICAgICAgICAgIHN0YWdlczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBzdGFnZU5hbWU6ICdTb3VyY2UnLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbbmV3IEZha2VTb3VyY2VBY3Rpb24oeyBhY3Rpb25OYW1lOiAnU291cmNlJywgb3V0cHV0OiBzb3VyY2VPdXRwdXQgfSldLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3QgYnVpbGRTdGFnZSA9IHBpcGVsaW5lLmFkZFN0YWdlKHsgc3RhZ2VOYW1lOiAnQnVpbGQnIH0pO1xuXG4gICAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgICBidWlsZFN0YWdlLmFkZEFjdGlvbihuZXcgRmFrZUJ1aWxkQWN0aW9uKHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWU6ICdGYWtlQnVpbGQnLFxuICAgICAgICAgICAgaW5wdXQ6IHNvdXJjZU91dHB1dCxcbiAgICAgICAgICAgIGFjY291bnQ6ICcxMjM0NTY3ODkwMTInLFxuICAgICAgICAgIH0pKTtcbiAgICAgICAgfSwgL1BpcGVsaW5lIHN0YWNrIHdoaWNoIHVzZXMgY3Jvc3MtZW52aXJvbm1lbnQgYWN0aW9ucyBtdXN0IGhhdmUgYW4gZXhwbGljaXRseSBzZXQgYWNjb3VudC8pO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICB9LFxuICB9LFxufSk7XG5cbmRlc2NyaWJlKCd0ZXN0IHdpdGggc2hhcmVkIHNldHVwJywgKCkgPT4ge1xuICBsZXQgc3RhY2s6IGNkay5TdGFjaztcbiAgbGV0IHNvdXJjZUFydGlmYWN0OiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3Q7XG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIHNvdXJjZUFydGlmYWN0ID0gbmV3IGNvZGVwaXBlbGluZS5BcnRpZmFjdCgpO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gYWRkIGFjdGlvbnMgdG8gc3RhZ2VzIGFmdGVyIGNyZWF0aW9uJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnUGlwZWxpbmUnLCB7XG4gICAgICBzdGFnZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlU291cmNlQWN0aW9uKHsgYWN0aW9uTmFtZTogJ0ZldGNoJywgb3V0cHV0OiBzb3VyY2VBcnRpZmFjdCB9KV0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBzdGFnZU5hbWU6ICdCdWlsZCcsXG4gICAgICAgICAgYWN0aW9uczogW25ldyBGYWtlQnVpbGRBY3Rpb24oeyBhY3Rpb25OYW1lOiAnR2NjJywgaW5wdXQ6IHNvdXJjZUFydGlmYWN0IH0pXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgcGlwZWxpbmUuc3RhZ2UoJ0J1aWxkJykuYWRkQWN0aW9uKG5ldyBGYWtlQnVpbGRBY3Rpb24oeyBhY3Rpb25OYW1lOiAnZGVidWcuY29tJywgaW5wdXQ6IHNvdXJjZUFydGlmYWN0IH0pKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2RlUGlwZWxpbmU6OlBpcGVsaW5lJywge1xuICAgICAgU3RhZ2VzOiBhcnJheVdpdGgoe1xuICAgICAgICBOYW1lOiAnQnVpbGQnLFxuICAgICAgICBBY3Rpb25zOiBbXG4gICAgICAgICAgb2JqZWN0TGlrZSh7IE5hbWU6ICdHY2MnIH0pLFxuICAgICAgICAgIG9iamVjdExpa2UoeyBOYW1lOiAnZGVidWcuY29tJyB9KSxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgIH0pO1xuICB9KTtcbn0pO1xuIl19