"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const codepipeline = require("../lib");
const stage_1 = require("../lib/private/stage");
/* eslint-disable quote-props */
nodeunit_shim_1.nodeunitShim({
    'Pipeline Stages': {
        'can be inserted before another Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const secondStage = pipeline.addStage({ stageName: 'SecondStage' });
            pipeline.addStage({
                stageName: 'FirstStage',
                placement: {
                    rightBefore: secondStage,
                },
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    { 'Name': 'FirstStage' },
                    { 'Name': 'SecondStage' },
                ],
            }));
            test.done();
        },
        'can be inserted after another Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const firstStage = pipeline.addStage({ stageName: 'FirstStage' });
            pipeline.addStage({ stageName: 'ThirdStage' });
            pipeline.addStage({
                stageName: 'SecondStage',
                placement: {
                    justAfter: firstStage,
                },
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    { 'Name': 'FirstStage' },
                    { 'Name': 'SecondStage' },
                    { 'Name': 'ThirdStage' },
                ],
            }));
            test.done();
        },
        "attempting to insert a Stage before a Stage that doesn't exist results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            const anotherPipeline = new codepipeline.Pipeline(stack, 'AnotherPipeline');
            test.throws(() => {
                anotherPipeline.addStage({
                    stageName: 'AnotherStage',
                    placement: {
                        rightBefore: stage,
                    },
                });
            }, /before/i);
            test.done();
        },
        "attempting to insert a Stage after a Stage that doesn't exist results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            const anotherPipeline = new codepipeline.Pipeline(stack, 'AnotherPipeline');
            test.throws(() => {
                anotherPipeline.addStage({
                    stageName: 'AnotherStage',
                    placement: {
                        justAfter: stage,
                    },
                });
            }, /after/i);
            test.done();
        },
        'providing more than one placement value results in an error'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            test.throws(() => {
                pipeline.addStage({
                    stageName: 'SecondStage',
                    placement: {
                        rightBefore: stage,
                        justAfter: stage,
                    },
                });
                // incredibly, an arrow function below causes nodeunit to crap out with:
                // "TypeError: Function has non-object prototype 'undefined' in instanceof check"
            }, /(rightBefore.*justAfter)|(justAfter.*rightBefore)/);
            test.done();
        },
        'can be retrieved from a pipeline after it has been created'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'FirstStage',
                    },
                ],
            });
            pipeline.addStage({ stageName: 'SecondStage' });
            test.equal(pipeline.stages.length, 2);
            test.equal(pipeline.stages[0].stageName, 'FirstStage');
            test.equal(pipeline.stages[1].stageName, 'SecondStage');
            // adding stages to the returned array should have no effect
            pipeline.stages.push(new stage_1.Stage({
                stageName: 'ThirdStage',
            }, pipeline));
            test.equal(pipeline.stageCount, 2);
            test.done();
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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