import requests as requests_http
from . import utils
from formance.models import operations, shared
from typing import Any, Optional

class Orchestration:
    _client: requests_http.Session
    _security_client: requests_http.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests_http.Session, security_client: requests_http.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version
        
    def create_workflow(self, request: shared.CreateWorkflowRequest) -> operations.CreateWorkflowResponse:
        r"""Create workflow
        Create a workflow
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix('/') + '/api/orchestration/flows'
        
        headers = {}
        req_content_type, data, form = utils.serialize_request_body(request, "request", 'json')
        if req_content_type not in ('multipart/form-data', 'multipart/mixed'):
            headers['content-type'] = req_content_type
        
        client = self._security_client
        
        http_res = client.request('POST', url, data=data, files=form, headers=headers)
        content_type = http_res.headers.get('Content-Type')

        res = operations.CreateWorkflowResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 201:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.CreateWorkflowResponse])
                res.create_workflow_response = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    def get_flow(self, request: operations.GetFlowRequest) -> operations.GetFlowResponse:
        r"""Get a flow by id
        Get a flow by id
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(operations.GetFlowRequest, base_url, '/api/orchestration/flows/{flowId}', request)
        
        
        client = self._security_client
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.GetFlowResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.GetWorkflowResponse])
                res.get_workflow_response = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    def get_workflow_occurrence(self, request: operations.GetWorkflowOccurrenceRequest) -> operations.GetWorkflowOccurrenceResponse:
        r"""Get a workflow occurrence by id
        Get a workflow occurrence by id
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(operations.GetWorkflowOccurrenceRequest, base_url, '/api/orchestration/flows/{flowId}/runs/{runId}', request)
        
        
        client = self._security_client
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.GetWorkflowOccurrenceResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.GetWorkflowOccurrenceResponse])
                res.get_workflow_occurrence_response = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    def list_flows(self) -> operations.ListFlowsResponse:
        r"""List registered flows
        List registered flows
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix('/') + '/api/orchestration/flows'
        
        
        client = self._security_client
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.ListFlowsResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.ListWorkflowsResponse])
                res.list_workflows_response = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    def list_runs(self, request: operations.ListRunsRequest) -> operations.ListRunsResponse:
        r"""List occurrences of a workflow
        List occurrences of a workflow
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(operations.ListRunsRequest, base_url, '/api/orchestration/flows/{flowId}/runs', request)
        
        
        client = self._security_client
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.ListRunsResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[Any])
                res.list_runs_response = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    def orchestrationget_server_info(self) -> operations.OrchestrationgetServerInfoResponse:
        r"""Get server info
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix('/') + '/api/orchestration/_info'
        
        
        client = self._security_client
        
        http_res = client.request('GET', url)
        content_type = http_res.headers.get('Content-Type')

        res = operations.OrchestrationgetServerInfoResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 200:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.ServerInfo])
                res.server_info = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    def run_workflow(self, request: operations.RunWorkflowRequest) -> operations.RunWorkflowResponse:
        r"""Run workflow
        Run workflow
        """
        
        base_url = self._server_url
        
        url = utils.generate_url(operations.RunWorkflowRequest, base_url, '/api/orchestration/flows/{flowId}/runs', request)
        
        headers = {}
        req_content_type, data, form = utils.serialize_request_body(request, "request_body", 'json')
        if req_content_type not in ('multipart/form-data', 'multipart/mixed'):
            headers['content-type'] = req_content_type
        query_params = utils.get_query_params(operations.RunWorkflowRequest, request)
        
        client = self._security_client
        
        http_res = client.request('POST', url, params=query_params, data=data, files=form, headers=headers)
        content_type = http_res.headers.get('Content-Type')

        res = operations.RunWorkflowResponse(status_code=http_res.status_code, content_type=content_type, raw_response=http_res)
        
        if http_res.status_code == 201:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.RunWorkflowResponse])
                res.run_workflow_response = out
        else:
            if utils.match_content_type(content_type, 'application/json'):
                out = utils.unmarshal_json(http_res.text, Optional[shared.Error])
                res.error = out

        return res

    