"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT, language_1.DocumentationLanguage.JAVA];
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" throttled', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['Lambda.TooManyRequestsException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            // We have to prepare the input to be a JSON string, within an array, because the ECS task integration expects
            // an array of strings (the model if that of a CLI invocation).
            // Unfortunately, we have to split this in two Pass states, because I don't know how to make it work otherwise.
            return new aws_stepfunctions_1.Pass(this, `Prepare ${language}`, {
                parameters: { command: { 'bucket.$': '$.bucket', 'assembly.$': '$.assembly', '$TaskExecution.$': '$.$TaskExecution' } },
                resultPath: '$',
            })
                .next(new aws_stepfunctions_1.Pass(this, `Stringify ${language} input`, {
                parameters: { 'commands.$': 'States.Array(States.JsonToString($.command))' },
                resultPath: '$',
            })
                .next(this.transliterator.createEcsRunTask(this, `Generate ${language} docs`, {
                cluster: this.ecsCluster,
                inputPath: '$.commands',
                language,
                resultSelector: { result: { 'language': language, 'success.$': '$' } },
                vpcSubnets: props.vpcSubnets,
            })
                // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
                .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
                .addRetry({
                errors: [
                    'ECS.AmazonECSException',
                    'ECS.InvalidParameterException',
                    'jsii-docgen.NpmError.E429',
                    'jsii-codgen.NpmError.EPROTO',
                ],
                ...THROTTLE_RETRY_POLICY,
            })
                .addRetry({ maxAttempts: 3 })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" timed out`, { parameters: { error: 'Timed out!', language } }), { errors: ['States.Timeout'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" service error`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', language } }), { errors: ['States.TaskFailed'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" fault`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['States.ALL'] })));
        }))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it.
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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