"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const monitored_certificate_1 = require("../monitored-certificate");
const cache_invalidator_1 = require("./cache-invalidator");
const config_1 = require("./config");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_OPTIMIZED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        new cache_invalidator_1.CacheInvalidator(this, 'CacheInvalidator', { bucket: props.packageData, distribution: this.distribution });
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            sources: [s3deploy.Source.asset(webappDir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
        });
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            sources: [s3deploy.Source.asset(config.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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